/****************************************************************************/
/*                                                                          */
/*  The FreeType project - a Free and Portable Quality TrueType Renderer.   */
/*                                                                          */
/*  Copyright 1996, 1997 by                                                 */
/*  D. Turner, R.Wilhelm, and W. Lemberg                                    */
/*                                                                          */
/*  timer : A simple performance benchmark. Now with graylevel rendering    */
/*          with the '-g' option.                                           */
/*                                                                          */
/*          Be aware that the timer program benchmarks different things     */
/*          in each release of the FreeType library. Thus, performance      */
/*          should only be compared between similar release numbers.        */
/*                                                                          */
/*                                                                          */
/*  NOTE : This is just a test program that is used to show off and         */
/*         debug the current engine; which is still in alpha. In no         */
/*         way does it shows the final high-level interface that            */
/*         client applications will use. Wait for at least a beta for       */
/*         this.                                                            */
/*                                                                          */
/****************************************************************************/

#include "tttypes.h"
#include "ttcalc.h"
#include "tttables.h"
#include "ttmemory.h"
#include "ttraster.h"
#include "ttfile.h"
#include "tterror.h"
#include "ttexec.h"

#include "gmain.h"
#include "gdriver.h"
#include "gevents.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>    /* libc ANSI   */
#include <time.h>    /* for 'clock' */

/* SunOS 4.1.* does not define CLOCKS_PER_SEC, so include <sys/param.h> */
/* to get the HZ macro which is the equivalent.                         */
#if defined(__sun__) && !defined(SVR4) && !defined(__SVR4)
#include <sys/param.h>
#define CLOCKS_PER_SEC HZ
#endif


#define MAX_GLYPHS  512   /* Maximum number of glyphs rendered at one time */

#define Profile_Buff_Size  64000       /* Size of the render pool   */
                                       /* Minimum is around 4 Kb    */
                                       /* experiment to see how its */
                                       /* size impacts on raster    */
                                       /* performance..             */

#define Font_Buff_Size     400000      /* this buffer holds all     */
                                       /* font specific data.       */

  char              Header[128];

  TT_Stream         stream;
  PResident_Record  resident;
  PInstance_Record  instance;

  PStorage          Font_Buffer;

  extern PGlyph     max_glyph;

  TT_PCoordinates   cur_x;
  TT_PCoordinates   cur_y;

  PShort            cur_endContour;
  PByte             cur_touch;

  Int  max_points;
  Int  max_contours;

  Int  start_point[MAX_GLYPHS];
  Int  num_points [MAX_GLYPHS];

  Int  start_contour[MAX_GLYPHS];
  Int  num_contours [MAX_GLYPHS];

  Int  num_glyphs;
  Int  tab_glyphs;
  Int  cur_glyph;
  Int  cur_point;
  Int  cur_contour;

  TGlyphRecord  gRec;
  TRasterBlock  Bit;

  Int  Fail;
  Int  Num;

  char  GrayLines[1024];

  Bool  visual;      /* display glyphs while rendering */
  Bool  gray_render; /* smooth fonts with gray levels  */

  static void Clear_Buffer();


/*******************************************************************/
/*                                                                 */
/*  Get_Time:                                                      */
/*                                                                 */
/*    returns the current time in milliseconds.                    */
/*                                                                 */
/*******************************************************************/

  long  Get_Time()
  {
    return clock() * 1000 / CLOCKS_PER_SEC;
  }


/*******************************************************************/
/*                                                                 */
/*  Fatal1 & Fatal2                                                */
/*                                                                 */
/*    Fatal error message display then exit(1)                     */
/*                                                                 */
/*******************************************************************/

  void  Fatal1( char*  message )
  {
    printf( message );
    exit( 1 );
  }


  void  Fatal2( char*  message, char*  param )
  {
    printf( message, param );
    exit( 1 );
  }


/*******************************************************************/
/*                                                                 */
/*  Init_Engine :                                                  */
/*                                                                 */
/*    Allocate bitmap, render pool and other structs...            */
/*                                                                 */
/*******************************************************************/

  void  Init_Engine()
  {
    PByte  p;
    Bool   result;

    Bit.rows  = vio_Height;   /* The whole window */
    Bit.width = vio_Width;

    if ( gray_render )
    {
      Bit.cols = Bit.width;
      Bit.flow = TT_Flow_Up;
      Bit.size = Bit.rows * Bit.width;
    }
    else
    {
      Bit.cols = (Bit.width + 7) / 8;       /* convert to # of bytes */
      Bit.flow = TT_Flow_Up;
      Bit.size = Bit.rows * Bit.cols;       /* number of bytes in buffer */
    }

    Bit.bitmap = (void*)malloc( Bit.size );
    if ( !Bit.bitmap )
      Fatal1( "ERROR: Not enough memory to allocate bitmap!\n" );

    p = (PByte)malloc( Profile_Buff_Size );
    if ( !p )
      Fatal1( "ERROR: Not enough memory to allocate render pool!\n" );

    if ( gray_render )
      result = InitRasterizer( p, Profile_Buff_Size, GrayLines, 1024 );
    else
      result = InitRasterizer( p, Profile_Buff_Size, NULL, 0 );

    if ( !result )
      Fatal1( "ERROR: Could not initialize rasterizer\n" );

    Set_High_Precision( FALSE );

    Clear_Buffer();
  }


/*******************************************************************/
/*                                                                 */
/*  Clear_Buffer                                                   */
/*                                                                 */
/*    Clear current bitmap                                         */
/*                                                                 */
/*******************************************************************/

  static void  Clear_Buffer()
  {
    if ( gray_render )
      memset( Bit.bitmap, gray_palette[0], Bit.size );
    else
      memset( Bit.bitmap, 0, Bit.size );
  }


/*******************************************************************/
/*                                                                 */
/*  LoadTrueTypeChar                                               */
/*                                                                 */
/*    Load a glyph into memory.                                    */
/*                                                                 */
/*******************************************************************/

  Bool  LoadTrueTypeChar( Int  idx )
  {
    Short            numP, numC;

    TT_PCoordinates  cx, cy;   /* current x and y arrays */

    PShort           ccontour;  /* current contour array */
    PByte            ctouch;  /* current touch array */

    Int  j;


    if ( idx < 0 || idx > num_glyphs ) return FAILURE;

    if ( !Load_TrueType_Glyph( instance, idx, TRUE ) )
      return FAILURE;

    TT_Transform_Glyph( instance );     /* this call scales the glyph */

    numP = instance->pts.n;
    numC = instance->glyph->numberOfContours;

    numP -= 2;  /* get rid of phantom points */

    start_point[cur_glyph] = cur_point;
    num_points [cur_glyph] = numP;

    start_contour[cur_glyph] = cur_contour;
    num_contours [cur_glyph] = numC;

    cx = cur_x + cur_point;
    cy = cur_y + cur_point;

    ctouch = cur_touch + cur_point;

    ccontour = cur_endContour + cur_contour;

    for ( j = 0; j < numP; j++ )
    {
      cx[j] = instance->pts.cur_x[j] + vio_Width * 16;
      cy[j] = instance->pts.cur_y[j] + vio_Height * 16;

      ctouch[j] = instance->pts.touch[j] & TT_Flag_On_Curve;
    }

    for ( j = 0; j < numC; j++ )
    {
      ccontour[j] = instance->glyph->contours[j].finish;
    }

    cur_point   += numP;
    cur_contour += numC;
    cur_glyph++;

    return SUCCESS;
  }


/*******************************************************************/
/*                                                                 */
/*  ConvertRaster                                                  */
/*                                                                 */
/*    Perform scan conversion                                      */
/*                                                                 */
/*******************************************************************/

  Bool  ConvertRaster( Int  index )
  {
    gRec.xCoord = cur_x + start_point[index];
    gRec.yCoord = cur_y + start_point[index];

    gRec.flag      = cur_touch + start_point[index];
    gRec.points    = num_points[index];
    gRec.outlines  = num_contours[index];
    gRec.outStarts = cur_endContour + start_contour[index];

    /* Note : for gray-levels, we use the default palette. */
    /*        Make up your own if necessary.               */

    if ( gray_render )
      return Render_Gray_Glyph( &gRec, &Bit, 0, gray_palette );
    else
      return Render_Glyph( &gRec, &Bit, 0 );
  }


  int  main( int  argc, char**  argv )
  {
    int    i, total, mode, base, rendered_glyphs;
    char   filename[128 + 4];
    char*  execname;

    Long   t, t0;


    Font_Buffer = (PStorage)malloc( Font_Buff_Size );
    if ( !Font_Buffer )
      Fatal1( "ERROR: Not enough memory to allocate font pool!\n" );

    Init_FontPool( Font_Buffer, Font_Buff_Size );

    execname    = argv[0];

    gray_render = 0;
    visual      = 0;

    while ( argc > 1 && argv[1][0] == '-' )
    {
      switch (argv[1][1])
      {
        case 'g':
          gray_render = 1;
          break;

        case 'v':
          visual = 1;
          break;

        default:
          Fatal2( "ERROR: Unknown argument '%s'!\n", argv[1] );
      }
      argc--;
      argv++;
    }

    if ( argc != 2 )
    {
      printf( "Time: simple performance timer - part of the FreeType project\n" );
      printf( "-------------------------------------------------------------\n\n" );
      printf( "Usage: %s [-g] [-v] fontname[.ttf]\n\n", execname );
      printf( "  where '-g' asks for gray-levels rendering\n" );
      printf( "        '-v' displays while rendering (slower)\n" );
      exit( 1 );
    }

    i = strlen( argv[1] );
    while ( i > 0 && argv[1][i] != '\\' )
    {
      if ( argv[1][i] == '.' )
        i = 0;
      i--;
    }

    filename[128] = 0;
    strncpy( filename, argv[1], 128 );
    if ( i >= 0 )
      strncpy( filename + strlen( filename ), ".ttf", 4 );

    if ( !TT_Open_File( filename , &stream ) )
      Fatal2( "Error: could not find/open %s\n", filename );

    if ( !TT_Load_Resident_Table( stream, &resident ) ) 
    {
      if ( Error == TT_Err_Out_Of_Memory )
      {
        printf( "Error: This font is too large to be loaded\n" );
        Fatal1( "       increase the font pool then recompile\n" );
      }
      Fatal2( "Error, could not load data from %s\n", filename );
    }

    num_glyphs = resident->numGlyphs;
        
    tab_glyphs = MAX_GLYPHS;
    if ( tab_glyphs > num_glyphs ) tab_glyphs = num_glyphs;

    max_points = tab_glyphs * resident->maxProfile.maxPoints;

    cur_x = malloc( max_points * sizeof ( TT_F26Dot6 ) );
    cur_y = malloc( max_points * sizeof ( TT_F26Dot6 ) );

    cur_touch = malloc( max_points );

    max_contours = tab_glyphs * resident->maxProfile.maxContours;

    cur_endContour = malloc( max_contours * sizeof(Short) );

    if ( !cur_x || !cur_y || !cur_touch || !cur_endContour )
      Panic1( "Error: not enough memory for preload array\n" );

    if ( !TT_Load_Instance_Data( resident, &instance ) ) 
    {
      if ( Error == TT_Err_Out_Of_Memory )
      {
        printf( "Error: This font is too large to be loaded\n" );
        Fatal1( "       increase the font pool then recompile\n" );
      }
      Fatal2( "Error, could not open instance for %s\n", filename );
    }

    if ( !Create_Context( instance ) )
      Fatal1( "Error : could not create context\n" );

    if ( !Reset_Context( instance, 400, 96 ) )
      Fatal1( "Error : could not reset context\n" );

    if ( gray_render )
      mode = Graphics_Mode_Gray;
    else
      mode = Graphics_Mode_Mono;

    if ( visual )
    {
      if ( !SetGraphScreen( mode ) )
        Fatal1( "Error : could not set graphics mode\n" );
    }
    else
    {
      /* This is the default bitmap size used */
      vio_Width  = 640;
      vio_Height = 450;
    }

    Init_Engine();

    Num  = 0;
    Fail = 0;

    total = num_glyphs;
    base  = 0;

    rendered_glyphs = 0;

    t0 = 0;  /* Initial time */

    while ( total > 0 )
    {
      /* First, preload 'tab_glyphs' in memory */

      cur_glyph   = 0;
      cur_point   = 0;
      cur_contour = 0;
  
      printf( "loading %d glyphs", tab_glyphs );

      for ( Num = 0; Num < tab_glyphs; Num++ )
      {
        if ( !LoadTrueTypeChar( base + Num ) )
          Fail++;

        total--;
      }

      base += tab_glyphs;

      if ( tab_glyphs > total ) 
        tab_glyphs = total;

      printf( ", rendering... " );

      /* Now, render the loaded glyphs */

      t = Get_Time();

      for ( Num = 0; Num < cur_glyph; Num++ )
      {
        if ( !ConvertRaster( Num ) )
          Fail++;
        else
        {
          rendered_glyphs ++;

          if ( visual )
          {
            sprintf( Header, "Glyph : %5d", Num );
            Display_Bitmap_On_Screen( Bit.bitmap, Bit.rows, Bit.cols );
            Clear_Buffer();
          }
        }
      }

      t = Get_Time() - t;
      if ( t < 0 ) t += 100 * 60 * 60;

      printf( " = %f s\n", (double)t / 1000 );
      t0 += t;
    }

    if ( visual ) RestoreScreen();

    TT_Close_File( stream );
 
    printf( "\n" );
    printf( "rendered glyphs  = %d\n", rendered_glyphs );
    printf( "render time      = %f s\n", (double)t0 / 1000 );
    printf( "fails            = %d\n", Fail );
    printf( "average glyphs/s = %f\n", 
            (double)rendered_glyphs / t0 * 1000 );

    return 0; 
  }


/* End */
