/* (C) 1999-2000 Samuel Audet <guardia@cam.org>

Profitable use of this source code based on its execution or sale
excluding the cost of the media, shipping, manwork or supporting
hardware is not allowed unless granted by the author himself.  Any
modifications or inclusion of this code in other non-profitable programs
must contain this message and the original author's name. Programs based
on any of this source code must therefore contain the original or
modified source code files.  Use of this source code in a commercial
program will require permission from the author.  No more than 50% of
the original size of the source code from Disk Indexer can be used in a
non-commercial program, unless granted by the author.

*/

import java.io.*;
import java.util.*;

public class HDataPage
{
   // stuff saved in database
   int primaryPointer = -1, parentPointer = -1,
       prevBrother = -1,    nextBrother = -1;
   short used = -1, startAt = -1, amount = -1;

   // space of all the preceding
   public static final short headerSize = 22;

   // deducted by area used in database
   int myPointer = -1;

   // for the buffer manager
   long timestamp = System.currentTimeMillis();
   boolean needUpdateOnDisk = false;
   Vector nodes = new Vector(10,0);
   HDataNode rootNode = null;
   byte[] blobData = null;

public HDataPage(HDataNode rootNode)
{
   super();
   this.rootNode = rootNode;
}

public boolean checkFreePage(RandomAccessFile dataFile, int pointer) throws IOException
{
   synchronized(rootNode)
   {

   myPointer = pointer;
   dataFile.seek(pointer * HDatabase.PAGE_SIZE);

   readHeader(dataFile);

   if(primaryPointer == -2) // found a free page
      return true;
   else
      return false;
   }
}

public boolean readPage(RandomAccessFile dataFile, int pointer) throws IOException
{
   synchronized(rootNode)
   {

   timestamp = System.currentTimeMillis();

   myPointer = pointer;
   dataFile.seek(pointer * HDatabase.PAGE_SIZE);

   readHeader(dataFile);

   if(primaryPointer == -2) // free page
      return true;

   if(amount == -1)
   {
      blobData = new byte[used-headerSize];
      dataFile.read(blobData);
   }
   else
      readNodes(dataFile);

   }

   return true;
}


protected boolean readHeader(RandomAccessFile dataFile) throws IOException
{
   synchronized(rootNode)
   {

   primaryPointer = dataFile.readInt();
   parentPointer = dataFile.readInt();
   prevBrother = dataFile.readInt();
   nextBrother = dataFile.readInt();
   used = dataFile.readShort();
   startAt = dataFile.readShort();
   amount = dataFile.readShort();

   }

   return true;
}

protected boolean readNodes(RandomAccessFile dataFile) throws IOException
{
   synchronized(rootNode)
   {

   nodes.ensureCapacity(amount);

   for(int i = 0; i < amount; i++)
   {
      HDataNode aNode = new HDataNode();
      aNode.readFrom(dataFile);
      aNode.primaryPointer = primaryPointer;
      aNode.myPointer = myPointer;
      aNode.index = i;
      aNode.parentPointer = parentPointer;
      nodes.addElement(aNode);
   }

   }

   return true;
}


public boolean writePage(RandomAccessFile dataFile) throws IOException
{
   synchronized(rootNode)
   {

   try
   {
      dataFile.seek(myPointer * HDatabase.PAGE_SIZE);
   }
   // man this sucks, why can't seek() return any more precise information?
   catch(IOException e)
   {
      dataFile.seek(dataFile.length());
      HDatabase.writeBlanks(dataFile, (int) ((long) myPointer * HDatabase.PAGE_SIZE - dataFile.length()));
   }

   writeHeader(dataFile);

   if(amount == -1)
      dataFile.write(blobData);
   else
      writeNodes(dataFile);

   needUpdateOnDisk = false;
   }

   return true;
}


protected boolean writeHeader(RandomAccessFile dataFile) throws IOException
{
   synchronized(rootNode)
   {

   dataFile.writeInt(primaryPointer);
   dataFile.writeInt(parentPointer);
   dataFile.writeInt(prevBrother);
   dataFile.writeInt(nextBrother);
   dataFile.writeShort(used);
   dataFile.writeShort(startAt);
   dataFile.writeShort(amount);

   }

   return true;
}

protected boolean writeNodes(RandomAccessFile dataFile) throws IOException
{
   synchronized(rootNode)
   {

   for(int i = 0; i < nodes.size(); i++)
   {
      HDataNode aNode = (HDataNode) nodes.elementAt(i);
      aNode.writeTo(dataFile);
   }

   if(HDatabase.PAGE_SIZE - used > 0)
      HDatabase.writeBlanks(dataFile, HDatabase.PAGE_SIZE - used);

   }

   return true;
}




public HDataNode getNode(int index)
{
   timestamp = System.currentTimeMillis();

   return (HDataNode) nodes.elementAt(index);
}

public int getNodeSize()
{
   timestamp = System.currentTimeMillis();

   return nodes.size();
}

public int getIndexOf(HDataNode aNode)
{
   timestamp = System.currentTimeMillis();
   return nodes.indexOf(aNode);
}

public boolean addNode(HDataNode newNode)
{
   synchronized(rootNode)
   {

   short newSize = (short) (newNode.size() + used);

   if(newSize <= HDatabase.PAGE_SIZE)
   {
      used = newSize;
      amount++;
      newNode.index = nodes.size();
      newNode.primaryPointer = primaryPointer;
      newNode.myPointer = myPointer;
      newNode.parentPointer = parentPointer;
      nodes.addElement(newNode);
      timestamp = System.currentTimeMillis();
      needUpdateOnDisk = true;
      return true;
   }
   else
      return false;

   }
}

public boolean setNode(HDataNode newNode, int index)
{
   synchronized(rootNode)
   {

   HDataNode oldNode = (HDataNode) nodes.elementAt(index);
   short newSize = (short) (used-oldNode.size() + newNode.size());

   if(newSize <= HDatabase.PAGE_SIZE)
   {
      used = newSize;
      newNode.primaryPointer = primaryPointer;
      newNode.myPointer = myPointer;
      newNode.index = index;
      newNode.parentPointer = parentPointer;
      nodes.setElementAt(newNode,index);
      timestamp = System.currentTimeMillis();
      needUpdateOnDisk = true;

      return true;
   }
   else
      return false;

   }
}

public boolean removeNode(int index)
{
   synchronized(rootNode)
   {

   timestamp = System.currentTimeMillis();
   needUpdateOnDisk = true;

   HDataNode oldNode = (HDataNode) nodes.elementAt(index);
   if(oldNode != null)
   {
      nodes.removeElementAt(index);
      used -= oldNode.size();
      amount--;
   }

   }

   return true;
}


public void setStartAt(short startAt)
{
   needUpdateOnDisk = true;
   this.startAt = startAt;
}

public void setPrevBrother(int prevBrother)
{
   needUpdateOnDisk = true;
   this.prevBrother = prevBrother;
}

public void setNextBrother(int nextBrother)
{
   needUpdateOnDisk = true;
   this.nextBrother = nextBrother;
}

public boolean setBlobData(byte[] blobData, int off, int len)
{
   if(len > HDatabase.PAGE_SIZE-headerSize)
      return false;
   else
   {
      needUpdateOnDisk = true;
      this.blobData = new byte[len];
      used = (short) (headerSize + len);
      startAt = 0;
      amount = -1;
      System.arraycopy(blobData,off,this.blobData,0,len);
      return true;
   }
}



public void setParentPointer(int parentPointer)
{
   needUpdateOnDisk = true;
   this.parentPointer = parentPointer;
}

public String toString()
{
   return String.valueOf(primaryPointer) + " " +
          String.valueOf(parentPointer) + " " +
          String.valueOf(prevBrother) + " " +
          String.valueOf(nextBrother) + " " +
          String.valueOf(used) + " " +
          String.valueOf(startAt) + " " +
          String.valueOf(amount) + " " +
          String.valueOf(myPointer) + " " +
          String.valueOf(timestamp) + " " +
          String.valueOf(needUpdateOnDisk) + " NODES: " +
          nodes;
}

protected void finalize() throws Throwable
{
   nodes = null;
   rootNode = null;
   blobData = null;

   super.finalize();
}


}
