//---------------------------------------------------------------------------
//
// %FILE     wd.c
// %VSS-REV  $Revision: 8 $
// %CREATED  1996.02.28
// %REVISED  $Date: 4/18/97 4:27p $
// %AUTHOR   Michael C. Draeger 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF, NS486SXL
// %SUMMARY  Watchdog Timer code
//
// %VSS      $Author: Miked $ $Date: 4/18/97 4:27p $ $Revision: 8 $
//
// DESCRIPTION
//
//   Functions to control the WatchDog timer.
//
// HISTORY
//
/*
 *
 * $History: wd.c $ 
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 4/18/97    Time: 4:27p
 * Updated in $/nsdemo
 *  New header (comment) changes.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 7/23/96    Time: 2:26p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 4/18/96    Time: 11:06a
 * Updated in $/nsdemo
 * Clean up for release.
 * 
 * *****************  Version 2  *****************
 * User: Miked        Date: 4/04/96    Time: 3:19p
 * Updated in $/board test
 * changed headers for VSS
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "wd.h"

//---------------------------------------------------------------------------

// INPUT       ULONG time - time in ms (1-65K) 
//             int mode - result of watchdog timeout
//               WD_IRQ - generate IRQ on timeout (reset on second)
//               WD_NMI - generate NMI on timeout (reset on second)
//               WD_RST - reset on timeout
//
// OUTPUT      none 
// RETURN      USHORT INVALID_PARAMETER for invalid parameters
//                    FAIL if PIT setup fails
//                    SUCCESS if WD set up correctly
//
// DESCRIPTION this function starts the watchdog timer.  This function 
//             can be called while the WD is running to restart the WD
//             with a different time or mode.  For the WD_IRQ mode, the
//             user must map the WD to an IRQ and setup the PIT.  Note also
//             that in WD_NMI mode a normal IRQ (as well as an NMI) will
//             be generated if the WD is mapped to an interrupt that is
//             enabled.

USHORT WD_Set( ULONG time, int mode )
{
  
  PIT_SET Set;
  USHORT  retval;

  // Validate Parameters

  if ( mode > 2 )
    return INVALID_PARAMETER;

  // Make sure WD is disabled

  WD_Cancel();

  // Start Timer

  Set.Timer = 2;            // channel 2
  Set.Mode  = 5;            // mode 5
  Set.Time  = time;         // time specified
  Set.Gate  = PIT_GateNA;   // gate is NA for WD

  retval = PIT_Set( & Set );
  
  if ( retval == INVALID_PARAMETER )
    return INVALID_PARAMETER;
  else if ( retval != SUCCESS )
    return FAIL;

  // Enable Watchdog

  if ( mode == WD_RST )
    IOW_BYTE( WD_CONTROL, WTE | WT_RST );
  else if ( mode == WD_NMI )
    IOW_BYTE( WD_CONTROL, WTE | WT_NMI );
  else
    IOW_BYTE( WD_CONTROL, WTE );

  // Watchdog is running

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       none 
// OUTPUT      none 
// RETURN      USHORT - SUCCESS indicates watchdog was canceled
//
// DESCRIPTION stop the watchdog timer 

USHORT WD_Cancel( void )
{
  
  // Disable Watchdog if it is enabled

  if ( ( IOR_BYTE( WD_CONTROL ) & WTE ) == WTE ) 
  {
    IOW_BYTE( WD_CONTROL,   WD_DISABLE1 );   // first byte
    IOW_BYTE( WD_RETRIGGER, WD_DISABLE2 );   // second byte
  }

  // All done

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       none 
// OUTPUT      none 
// RETURN      USHORT - SUCCESS indicates the watchdog was retriggered
//
// DESCRIPTION this function retriggers the watchdog (restarts it from 
//             time zero).  This is normally used to prevent a watchdog
//             timeout.

USHORT WD_Retrigger( void )
{
  
  // Retrigger the watchdog

  IOW_WORD( WD_RETRIGGER, RETRIGGER );

  // All done

  return SUCCESS;

}

//---------------------------------------------------------------------------
// END       wd.c
//---------------------------------------------------------------------------
