//---------------------------------------------------------------------------
//
// %FILE     rtc.c
// %VSS-REV  $Revision: 15 $
// %CREATED  1995.12.20
// %REVISED  $Date: 4/21/97 11:52a $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF (B0+), NS486SXL
// %SUMMARY  Real Time Clock module
//     
// %VSS      $Author: Miked $ $Date: 4/21/97 11:52a $ $Revision: 15 $
//
// DESCRIPTION
//
//   This file contains code for the NS486 Real Time Clock (RTC).
//   The NS486 RTC is compatible with the Motorolla MC146818.  However,
//   there are several differences.  See NOTES below.
//
// INPUTS
//
//   Register locations ( #define <WORD> )
//
//     RTC_INDEX - Index Register
//     RTC_DATA  - Data Register
//
// HISTORY
//
/*
 *
 * $History: rtc.c $
 * 
 * *****************  Version 15  *****************
 * User: Miked        Date: 4/21/97    Time: 11:52a
 * Updated in $/nsdemo
 * Added missing IO write in Init.  Removed unused variable assignment.
 * 
 * *****************  Version 14  *****************
 * User: Miked        Date: 4/18/97    Time: 4:21p
 * Updated in $/nsdemo
 * Init function now will leave RTC mode (binary/bcd, 24/12, daylight
 * savings mode) alone if VRT is valid.  Also indicates as return value if
 * VRT was valid to assist user code.  Added SET_MODE and
 * GET_PERIODIC_ALARM commands to RTC_Command.  Also now return mode info
 * with the GET_DATE_TIME command and return interrupt pending info with
 * the CLEAR_INTERRUPT command to allow polling or identifying of
 * interrupts.   New header (comment) changes.
 * 
 * *****************  Version 13  *****************
 * User: Miked        Date: 12/04/96   Time: 4:07p
 * Updated in $/nsdemo
 * Moved function prototypes to header file.
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 12/04/96   Time: 1:37p
 * Updated in $/nsdemo
 * Updated version to 1.6.
 * 
 * *****************  Version 11  *****************
 * User: Miked        Date: 10/03/96   Time: 4:53p
 * Updated in $/nsdemo
 * Added functions to disable Alarm Interrupt and Periodic Interrupt after
 * enabling.  Also added function to clear pending interrupts by reading
 * register C.  Changed function to set periodic alarm to use parameter to
 * determine periodic rate (was hardcoded).  Finally changed CMOS
 * functions to use #define value from rtc.h instead of a number (0xE) for
 * the CMOS offset, and got rid of uneless parameter check.
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 8/16/96    Time: 11:26a
 * Updated in $/nsdemo
 * Updated the comments in the NOTES section to fix some incorrect
 * statements and draw attention to the proper RTC initialization
 * sequence.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 7/01/96    Time: 11:36a
 * Updated in $/nsdemo
 * Removed stuff about Dates Of Month Alarm - which is not useful.  An
 * update to the datasheet will explain how this register works.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 4  *****************
 * User: Noreen       Date: 4/18/96    Time: 12:35p
 * Updated in $/nsdemo
 * Updated comments for version release
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 4/17/96    Time: 4:05p
 * Updated in $/nsdemo
 * Added some type casts where needed.
 * 
 * *****************  Version 2  *****************
 * User: Noreen       Date: 4/12/96    Time: 3:01p
 * Updated in $/nstest
 * Updated headers for VSS
 *
 */
//
// COPYRIGHT
//
//      (c) 1995, 1996, 1997 National Semiconductor Corporation
//
// NOTES
//
//   Differences between MC 146818 and National Semiconductor NS486 RTC
//
//     * Our daylight saving's feature works as correct in the US.
//       i.e., it switches to daylight savings on the FIRST Sunday in
//       April.  (The MC 146818 uses the last sunday in april).
//
//     * At index 0x7F is a "Dates of Month Alarm.  This register is not
//       useful as it only works in certain power down modes which will
//       not wake up the NS486SXF.  It is advisable to set this register to
//       0xFF (Don't care), although it should not matter if this register
//       is set to a valid value.
//
//     * There are 50 bytes or RAM from 0x0E to 0x3F.  This is the same as
//       the MC 146818, but apparently some similar devices used in PC's
//       provide 114 bytes.
//
//     * The DV[2:0] bits in Control Register A work a little differently
//       (but in general compatible), since the NS486 RTC clock is
//       fixed at 32.768KHz.
//
//     * Square Wave Enable is not supported. (CRB bit 3 is zero)
//
//   Previous versions of this code indicated that the NS486 had a "user"
//   copy of the time registers, and that unlike the Motorolla part, setting
//   the SET bit did not stop the time from updating.  This was incorrect.
//   The SET bit works in the same manner as on the Motorolla part.
//   This was a result of incorrect documentation in the data sheet.
//   The August 1996 and newer versions of the data sheet should be
//   corrected.
//
//   Another problem with the older data sheet was that it was not
//   completely accurate in the effect of reset on the control register
//   bits.  Because some bits in the control registers are not affected
//   by reset, the control registers should be initialized to absolute
//   values before using the RTC.  (You should not use a read-modify-write
//   type IO access until the registers are properly initialized.)
//   This code follows (and has followed) the proper initialization
//   sequence, which is:
//
//      o Read control register D to check the VRT status.  If VRT is "0"
//        all registers should be considered invalid.
//      o Initialize control registers A and B to known absolute
//        values.  For the RTC to count properly, be sure the divider
//        chain is set to normal operation.  The SET bit in CRB needs
//        to be "0" for the time updates to occur, although this can be
//        set now or later in code.
//      o Read control register C to clear the interrupt status bits.
//
//   Note that the above steps should be taken regardless of the VRT status.
//   After this initialization, the time, date, and alarm registers should
//   be initialized to known values if the VRT was read as "0".
//
//   One final note, this code uses the SET bit to read and write to time
//   and date registers.  This can cause some small ammount of inaccuracy
//   in time keeping, as updates are not occuring when the SET bit is a
//   "1".  For more accurate time keeping, one of the other methods of
//   accessing the time registers can be used.  The easiest way is to
//   poll the UIP bit in CRA.  If a "0" is read, you have at least
//   244us before an update, in which time you can read or write time
//   registers without disturbing the time keeping.  This is described in
//   the data sheet.
//
//---------------------------------------------------------------------------

#include "rtc.h"

//---------------------------------------------------------------------------
// functions used internally in this module

// Simple function to write index and read data
PRIVATE BYTE ReadClockPort(BYTE Port)
{
	IOW_BYTE(RTC_INDEX, Port);
	return IOR_BYTE(RTC_DATA);
}

// Simple function to write index and write data
PRIVATE void WriteClockPort( BYTE Port, BYTE Value )
{
	IOW_BYTE(RTC_INDEX, Port);
	IOW_BYTE(RTC_DATA, Value);
}

// Disable Clock.  This will set CRB bit 7 which will freeze a copy of the
// time registers in the user time registers.  Used to allow access to
// the time registers without worrying about updates.
PRIVATE void DisableClock()
{
  WriteClockPort(RTC_CRB, (BYTE) (CRB_DisableClock | ReadClockPort(RTC_CRB)) );
}

// Enable Clock.  If user time registers have been written since
// CRB bit 7 was set high, the time registers will be updated.  Use
// after DisableClock, or during initialization.
PRIVATE void EnableClock()
{
  WriteClockPort(RTC_CRB, (BYTE) (~CRB_DisableClock & ReadClockPort(RTC_CRB)) );
}

//---------------------------------------------------------------------------
//
// FUNCTION    RTC_Initialize
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if VRT bit read at 0, indicating that the registers
//                      and CMOS are invalid.  In this case, the CRA and
//                      CRB values are set to initial values.  The user code
//                      should set the time, date, and other registers before
//                      assuming they are accurate or useable.
//               NO_RESOURCE - if BIU has not been configured to enable accesses
//                             to RTC
//
// DESCRIPTION
//       This function is called to initialize the Real Time Clock
//
//---------------------------------------------------------------------------

USHORT RTC_Initialize(void)
{

  // Temporary byte
  BYTE tByte;

  // Used to keep track of whether VRT bit set on initial read
  BOOL VRT;

  // If RTC is not enabled in the BIU, return FAIL
  if ((IOR_BYTE(BIU_CONTROL1) & 0x04) != 0x04)
     return NO_RESOURCE;

  // See if the RTC and RAM is valid
  if(ReadClockPort(RTC_CRD) & CRD_VALID_TIME)
    VRT = TRUE;
  else
    VRT = FALSE;

  // Set divider chain to normal and disable periodic interrupts regardless
  // or VRT state.  If RAM was not valid, this will also force these bits
  // to a known state.
  WriteClockPort(RTC_CRA, CRA_InitVal);

  // Always clear SET bit and all interrupt enabled bits.  If RAM was valid,
  // leave DM and 24/12 in current states so that we do not set the RTC in
  // a time mode that is not consistant with the actual time/date register
  // values.  For example, if the RTC was used in BCD mode or 12 hour mode,
  // we could make the time invalid by setting it to 24 hour binary mode.
  // This lets the user code decide what mode to use.  If the RAM is not
  // valid we use a default (24 hour binary mode).

  if ( VRT == TRUE )
  {
    tByte = ReadClockPort(RTC_CRB);
    tByte &= 
      (~( CRB_SET_TIME | CRB_PERIODIC_ENABLE | CRB_ALARM_ENABLE | CRB_UIE ));
    tByte |= CRB_DAYLIGHT_SAVING_ENABLE;
    WriteClockPort(RTC_CRB, tByte);
  }
  else
    WriteClockPort(RTC_CRB, CRB_InitVal);    // default if RAM is invalid

  // Always read CRC to clear any interrupt flags set
  ReadClockPort(RTC_CRC);

  // Return FAIL if RAM was invalid initially.  Else return SUCCESS
  if ( VRT == TRUE )
    return SUCCESS;
  else
    return FAIL;
    		
}

//---------------------------------------------------------------------------
//
// FUNCTION    RTC_Command
//
// INPUT/OUTPUT  RTC_SET * pTimeInfo - a pointer to a structure which contains
//                                     settings for the Real Time Clock/Calendar
// RETURN      USHORT
//               SUCCESS - if call successful
//               INVALID_PARAMETER - if NULL pointer given as pTimeInfo
//
// DESCRIPTION
//
//---------------------------------------------------------------------------

USHORT RTC_Command(RTC_SET * pTimeInfo)
{
    // Temporary Bytes
    BYTE tByte1, tByte2;

	// Check for NULL pointer
	if(pTimeInfo == NULL)
	   return INVALID_PARAMETER;

    // Need to add parameter checking to this function still!

	switch (pTimeInfo->Command)
	{
    case SET_MODE:
      // Update only DM, 24/12, and DSE in the CRA register
      tByte1 = ReadClockPort(RTC_CRB);
      tByte1 &= 
        (~( CRB_DATA_BINARY | CRB_TIME_24 | CRB_DAYLIGHT_SAVING_ENABLE ) );
      tByte2 = ( (BYTE)pTimeInfo->ModeData &
        ( CRB_DATA_BINARY | CRB_TIME_24 | CRB_DAYLIGHT_SAVING_ENABLE ) );
      tByte1 = ( tByte1 | tByte2 );
      WriteClockPort(RTC_CRB, tByte1);

  	case SET_TIME:
	      DisableClock();
	      WriteClockPort(RTC_SECONDS, (BYTE)pTimeInfo->Seconds);
	      WriteClockPort(RTC_MINUTES, (BYTE) pTimeInfo->Minutes);
	      WriteClockPort(RTC_HOURS, (BYTE) pTimeInfo->Hours);
	      EnableClock();
	      break;

	case SET_DATE:
	      DisableClock();
	      WriteClockPort(RTC_DAYOFWEEK, (BYTE) pTimeInfo->DayOfWeek);
	      WriteClockPort(RTC_DAYOFMONTH, (BYTE) pTimeInfo->DateOfMonth);
	      WriteClockPort(RTC_MONTH, (BYTE)pTimeInfo->Month);
	      WriteClockPort(RTC_YEAR, (BYTE)pTimeInfo->Year);
	      EnableClock();
	      break;

	case GET_ALARM:
	      pTimeInfo->SecondsAlarm = ReadClockPort(RTC_ALARM_SECONDS);
	      pTimeInfo->MinutesAlarm = ReadClockPort(RTC_ALARM_MINUTES);
	      pTimeInfo->HoursAlarm = ReadClockPort(RTC_ALARM_HOURS);
          pTimeInfo->ModeData = ( ReadClockPort(RTC_CRB) &
              ( CRB_DATA_BINARY | CRB_TIME_24 | CRB_DAYLIGHT_SAVING_ENABLE ) );
	      break;

	case SET_ALARM:
	      WriteClockPort(RTC_ALARM_SECONDS, (BYTE)pTimeInfo->SecondsAlarm);
	      WriteClockPort(RTC_ALARM_MINUTES, (BYTE)pTimeInfo->MinutesAlarm);
	      WriteClockPort(RTC_ALARM_HOURS, (BYTE)pTimeInfo->HoursAlarm);
	      WriteClockPort(RTC_ALARM_DAYOFMONTH, ALARM_DONT_CARE);
	      WriteClockPort(RTC_CRB, (BYTE) (CRB_ALARM_ENABLE | ReadClockPort(RTC_CRB)) );
	      break;
	
	case DISABLE_ALARM:
	     //Disable Alarm Interrupt
	     WriteClockPort(RTC_CRB, (BYTE) ( (~CRB_ALARM_ENABLE) & ReadClockPort(RTC_CRB)) );
	     // Read CRC to clear interrupt
	     ReadClockPort(RTC_CRC);
	     break;
	
	case CLEAR_INTERRUPT:
	     // Read CRC to clear interrupt.  Also returns CRC value in IntStatus
         // to allow polling and detection of which interrupt occured.
	     pTimeInfo->IntStatus = ReadClockPort(RTC_CRC);
	     break;

	case SET_PERIODIC_ALARM:
	     //Program ControlRegisterA
	     WriteClockPort(RTC_CRA, (BYTE) 
	       ( ( CRA_PERIOD_INTRATE & (BYTE)pTimeInfo->PeriodicRate ) 
	         | ReadClockPort(RTC_CRA)) );
	     //Enable Alarm Interrupt
	     WriteClockPort(RTC_CRB, (BYTE) (CRB_PERIODIC_ENABLE | ReadClockPort(RTC_CRB)) );
	     break;

	case GET_PERIODIC_ALARM:
	     pTimeInfo->PeriodicRate = ( ReadClockPort(RTC_CRA) & CRA_PERIOD_INTRATE );
	     break;

	case DISABLE_PERIODIC_ALARM:
	     //Disable Alarm Interrupt
	     WriteClockPort(RTC_CRB, (BYTE) ( (~CRB_PERIODIC_ENABLE) & ReadClockPort(RTC_CRB)) );
	     // Read CRC to clear interrupt
	     ReadClockPort(RTC_CRC);
	     break;

	case GET_TIME_DATE:
	default:
	      DisableClock();
	      pTimeInfo->Seconds = ReadClockPort(RTC_SECONDS);
	      pTimeInfo->Minutes = ReadClockPort(RTC_MINUTES);
	      pTimeInfo->Hours = ReadClockPort(RTC_HOURS);
	      pTimeInfo->DayOfWeek = ReadClockPort(RTC_DAYOFWEEK);
	      pTimeInfo->DateOfMonth = ReadClockPort(RTC_DAYOFMONTH);
	      pTimeInfo->Month = ReadClockPort(RTC_MONTH);
	      pTimeInfo->Year = ReadClockPort(RTC_YEAR);
          pTimeInfo->ModeData = ( ReadClockPort(RTC_CRB) &
              ( CRB_DATA_BINARY | CRB_TIME_24 | CRB_DAYLIGHT_SAVING_ENABLE ) );
	      EnableClock();
	      break;
	}

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    RTC_RAM_Read
//
// INPUT       USHORT offset - The location of RAM
// OUTPUT      BYTE * value - a pointer to the value read from RAM
//
// RETURN      USHORT
//               SUCCESS - if call successful
//               INVALID_PARAMETER - if location of RAM specified to read from
//                                   is invalid
//
// DESCRIPTION
//   This function simply reads a byte from RAM
//
//---------------------------------------------------------------------------

USHORT RTC_RAM_Read(USHORT offset, USHORT * pvalue)
{
	//Check for NULL pointer
	if(pvalue == NULL)
	   return INVALID_PARAMETER;

	//Check for Valid RAM Location
	if(offset > (RTC_CMOS_SIZE-1) )
	   return INVALID_PARAMETER;

	//Read RAM Location
	*pvalue = ReadClockPort((BYTE)(offset+RTC_CMOS_BASE));

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    RTC_RAM_Write
//
// INPUT       USHORT offset - The location of RAM
//             USHORT value - The value to write to RAM
//
// RETURN      USHORT
//               SUCCESS - if call successful
//               INVALID_PARAMETER - if location of RAM specified to read from
//                                   is invalid
//
// DESCRIPTION
//   This function simply writes a byte to RAM
//
//---------------------------------------------------------------------------

USHORT RTC_RAM_Write(USHORT offset, USHORT value)
{
	//Check for Valid RAM Location
	if(offset > (RTC_CMOS_SIZE-1) )
	   return INVALID_PARAMETER;

	//Write to RAM Location
	WriteClockPort((BYTE)(offset+RTC_CMOS_BASE), (BYTE)value);

	return SUCCESS;
}

//---------------------------------------------------------------------------
// END       rtc.c
//---------------------------------------------------------------------------
