//---------------------------------------------------------------------------
//
// %FILE     lm75.c
// %VSS-REV  $Revision: 3 $
// %CREATED  1997.03.04
// %REVISED  $Date: 4/16/97 3:04p $
// %AUTHOR   Rhinda Bal-Firpo 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF, NS486SXL 
// %SUMMARY  Access.bus test module using NSC LM75 demo board.
//     
// %VSS      $Author: Rhinda $ $Date: 4/16/97 3:04p $ $Revision: 3 $
//
// DESCRIPTION
//
//   This file contains code to test the NS486 Access.bus (I2C) interface.
//   The National Semiconductor LM75 Digital Temperature Sensor is used
//   for this purpose.  A modified LM75 demo board must be connected to the
//   NS486 evaluation board feature connector for this code to run.  Please
//   see the application note "Using the Access.bus interface on the
//   NS486SXF Evaluation Board" for additional information on the hardware
//   and software.
//
// HISTORY
//
/*
 *
 * $History: Lm75.c $ 
 * 
 * *****************  Version 3  *****************
 * User: Rhinda       Date: 4/16/97    Time: 3:04p
 * Updated in $/nsdemo
 * needed it to do some testing
 * 
 * *****************  Version 2  *****************
 * User: Miked        Date: 4/11/97    Time: 6:29p
 * Updated in $/nsdemo
 * Re-did conversion routines (simpler).  Formatting changes.
 * 
 * *****************  Version 1  *****************
 * User: Rhinda       Date: 3/21/97    Time: 4:04p
 * Created in $/nsdemo
 * Initial version of LM75 test code, which tests the I2C interface with a
 * National LM75 evaluation board.
 *
 */
//
// COPYRIGHT
//
//      (c) 1997 National Semiconductor Corporation
//
// NOTES
//
//   As documented in the LM75 datasheet, the LM75 supports temperatures
//   from -55 degrees to 125 degrees Celsius.
// 
//---------------------------------------------------------------------------

#include "lm75.h"

//---------------------------------------------------------------------------

// INPUT        none
// OUTPUT       none
// RETURN       none
//
// DESCRIPTION  Performs a simple test of the Access.bus interface using
//              the LM75 demo hardware.

USHORT LM75()
{

  int degreeCelsius;   //temperature in degree Celsius
  BYTE value_read[2];  //array that stores the temperature read from the LM75
  BYTE value_write[2]; //array that stores the temperature to write to the  LM75
  BOOL halfDegree;     //store 0.5 degree result

  // print message indicating start of test
  dprintf("Access.bus Test:\r\n");

  // initialize Access.bus interface
  if  (I2C_Initialize() != SUCCESS)
  {
	dprintf("  Initialization - FAILED\r\n\n");  
    return FAIL;
  }

  // try to read temperature

  if  (I2C_Data_Read(LM75_ADDRESS, LM75_TEMP, 2, value_read) == FAIL)
  {
    dprintf("  Unable to read data!  Could indicate failure, or\r\n");
	dprintf("  LM75 test board not pluged in to feature connector!\r\n\n");
	return FAIL;
  }

  // convert result to temperature and display

  LM75_Convert_to_Celsius(value_read, &degreeCelsius, &halfDegree);
  if (halfDegree == TRUE)
    dprintf("  Temperature register at %d.5 degrees Celsius.\r\n",
	    degreeCelsius);
  else 
    dprintf("  Temperature register at %d degrees Celsius.\r\n", degreeCelsius);

  // write and read Alarm Threshold Register to verify writes 

  dprintf("  Setting Alarm Threshold Register to -35 degrees Celsius.\r\n");

  LM75_Convert_Write(-35, value_write);

  if (I2C_Data_Write(LM75_ADDRESS, LM75_TOS, 2, value_write ) == FAIL)
  {
    dprintf("  Write to Alarm Threshold register - FAILED\r\n\n");
    return FAIL;
  }
  if  (I2C_Data_Read(LM75_ADDRESS, LM75_TOS, 2, value_read) == FAIL)
  {
    dprintf("  Read from Alarm Threshold register - FAILED\r\n\n");
    return FAIL;
  }

  LM75_Convert_to_Celsius(value_read, &degreeCelsius, &halfDegree);
  if (halfDegree == TRUE)
    dprintf("  Alarm Threshold register is at %d.5 degree Celsius. \r\n", 
	    degreeCelsius);
  else 
    dprintf("  Alarm Threshold register is at %d degree Celsius. \r\n",
	    degreeCelsius);
   
  if ( degreeCelsius == -35 )
    dprintf("  Alarm Threshold register test - PASSED\r\n");
  else
  {
    dprintf("  Alarm Threshold register test - FAILED\r\n\n");
    return FAIL;
  }
  
  // thats all

  dprintf("\r\n");
  return SUCCESS;

} 
   
     
//---------------------------------------------------------------------------

// INPUT        BYTE value_read[2]: value read from LM75 temperature register
// OUTPUT       int *pDegree:       temperature in celsius (signed)
//              BOOL *pHalfDegree:  TRUE if LSB (0.5 degrees) is set
//                                    -24 degrees would become -24.5 degrees
//                                     52 degrees would become 52.5 degrees
// RETURN       none
//
// DESCRIPTION  This function is called to convert the data read from the LM75 
//              to degree Celsius
//
// NOTES        Temperature data is represented by a 9-bit, two's coplement 
//              word with an LSB equal to 0.5 degree Celsius.

void LM75_Convert_to_Celsius(BYTE value_read[2], int *pDegree, 
			     BOOL *pHalfDegree)
{

  int CelsiusTemp;
  BYTE temperature;
  BOOL isNegative;

  temperature = value_read[0];      // this is temperature (except for 0.5
				    // degree portion) in 2's complement form
  
  // adjust if negative

  if ( temperature > 127 )
    isNegative = TRUE;
  else
    isNegative = FALSE;

  if ( isNegative == TRUE )
    temperature = ~temperature + 0x01;   

  // assign to integer

  CelsiusTemp = (int) temperature;  // at this point, this is the temperature
				    // in an integer without sign or 0.5
				    // degree portion

  // make negative if applicable

  if ( isNegative == TRUE )
    CelsiusTemp = ( 0 - CelsiusTemp );  // now this is temperature with correct
					// sign.  still no 0.5 degree portion

  // assign to return variable

  *pDegree = CelsiusTemp;

  // set half degree flag accordingly

  if ( (value_read[1] & 0x80)==0x80 )
    *pHalfDegree = TRUE;
  else
    *pHalfDegree = FALSE;
   
}  

//---------------------------------------------------------------------------

// INPUT        int convert_value_write: temperature to convert
//                                       must be from -55 to 125 degrees C
//                                       half degrees are not supported
// OUTPUT       BYTE *pwrite_bytes:  data to write to LM75 register
// RETURN       none
//
// DESCRIPTION  This function takes an input of the desired temperature and
//              returns the proper LM75 register value for the given
//              temperature
//
//              This function will return 0 in pwrite_bytes if the input
//              temperature is not valid.

void LM75_Convert_Write(int convert_value_write, BYTE write_bytes[2])
{

  // if out of range, return 0

  if ( ( convert_value_write < -55 ) || ( convert_value_write > 125 ) )
  {
    write_bytes[0] = 0;
    write_bytes[1] = 0;
    return;
  }

  // write_bytes[1] will always be 0 since 0.5 degree increments are not
  // supported in this function

  write_bytes[1] = 0;

  // if we cast the integer as a BYTE, C will automatically get the sign
  // and everything correct, since convert_value_write is signed.

  write_bytes[0] = (BYTE) convert_value_write;

}


//---------------------------------------------------------------------------
// END       lm75.c
//---------------------------------------------------------------------------
