;---------------------------------------------------------------------------
; %FILE     dramsize.asm
; %VSS-REV  $Revision: 2 $
; %VERSION  1.1
; %CREATED  1996.04.02
; %REVISED  $Date: 12/02/96 4:23p $
; %AUTHOR   Michael C. Draeger 
; %PROJECT  NS486SXF evaluation board software
; %PART     NS486SXF 
; %SUMMARY  sample assembly program for NS486SXF eval board
;
; %VSS      $Author: Miked $ $Date: 12/02/96 4:23p $ $Revision: 2 $
;
; DESCRIPTION
;
;   This assembly file computes the DRAM size installed on the NS486SXF
;   evaluation board.  After programming the DRAM controller for the
;   proper size, it jumps to a definable user address. 
;
;   This code determines the DRAM size by writing to a few key addresses.
;   The evaluation board only supports 1MB, 4MB, and 16MB SIMMs.  From
;   what I could find:
;
;      1MB SIMM has a 9 bit page size
;      4MB SIMM has a 10 bit page size
;     16MB SIMM has a 11 bit page size
;
;   We start with 1MB (.5 MB per bank).  We set up BANK 1 as 2MB and
;   write to 0x200000 and 0x200400.  If the DRAM only has a 9 bit page
;   size, these addresses will be the same, so we will read back the
;   second value written from both addresses.  In this case, we
;   know the page size is 8. 
;
;   If we pass the 4MB check we configure bank 0 to also be 2MB, then
;   do a similar test for a 16MB SIMM (8MB per bank).
;
;   The reason I configure bank 1 first, do the check, and the configure
;   bank 0 is that this shows that a program could still use DRAM
;   storage while sizing the DRAM.  This program does not use any
;   stack or data, but this shows that it could be done.  You would
;   have to keep all data in one bank while you size the other.  Then
;   you could copy the data to the newly sized bank and then setup the
;   other bank.
;
;   The impact of this test on DRAM is minimal.  The test will write to the
;   following areas in DRAM (depending on installed SIMM size).  However,
;   this version will save and restore the original DRAM values at these
;   areas, so no changed to the DRAM contents will be made by this code.
;
;   1MB SIMM - will write data to 0x80000  - 0x80003   <- failed 4MB  addr #1
;   4MB SIMM - will write data to 0x200000 - 0x200003  <- passed 4MB  addr #1
;                                 0x200400 - 0x200403  <- passed 4MB  addr #2
;                                 0x200000 - 0x200003  <- failed 16MB addr #1
;  16MB SIMM - will write data to 0x800000 - 0x800003  <- passed 4MB  addr #1
;                                 0x800400 - 0x800403  <- passed 4MB  addr #2
;                                 0x800000 - 0x800003  <- passed 16MB addr #1
;                                 0x800800 - 0x800803  <- passed 16MB addr #2
;
; RELEASE HISTORY
;
;   1.0   1996.04.02  mcd  genesis - from boiler.asm 1.4
;   1.1   1996.12.02  mcd  added save/restore code to undo DRAM changes
;                          changed header style to be consistant with NS486 SW
;
; COPYRIGHT
;
;      (c) 1996 National Semiconductor Corporation
;
;---------------------------------------------------------------------------

; allow all instructions in assembler

.486p

;---------------------------------------------------------------------------

; include

include ns486reg.inc       ; NS486SXF registers
include io.inc             ; macros for port IO

;---------------------------------------------------------------------------
        
        ; 100 us count - value to count down in DEC, jcc loop to guarentee
        ; at least 100us delay before DRAM is used.

DRAM_COUNT      equ     04E2h   ; 4E2 should give about 5000 clocks
                                ; delay, or 200us at 25MHz.
                                ; (and more for slower clock speeds)

;---------------------------------------------------------------------------

MAGIC1          equ     0FEDCBA98h    ; DRAM test value 1
MAGIC2          equ     001234567h    ; DRAM test value 2

;---------------------------------------------------------------------------
        
        ; where to jump after setup

jump_location equ 0FFFCFFF0h

;---------------------------------------------------------------------------

CODESG    SEGMENT PARA USE32 'code'
DRAMSIZE  PROC    FAR
          ASSUME  CS:CODESG

;---------------------------------------------------------------------------
; ************ Code ********************************************************
;---------------------------------------------------------------------------

        cli              
        cld

;---------------------------------------------------------------------------
; Initialize DRAM
;---------------------------------------------------------------------------
; set up DRAM to minimum evaluation board setup prior to sizing.  If this
; code is run with the DRAM already set up to 1MB, this can be skipped.

        outpb DRAM_BANK,       000h   ;Both banks half meg

        outpb DRAM_MASK0,      0FFh   ;Mask for 1/2 megabytes
        outpb DRAM_MASK1,      0FFh   ;Mask for 1/2 megabytes
        outpb DRAM_ADDR1,      001h   ;Bank 1 at .5MB

        outpw DRAM_REFRESH,    625    ;15.625us between each refresh
                                      ;  cycle, or 16ms to refresh
                                      ;  the total DRAM (1024 refreshes)
                                      ;  This is set for 20MHz operation.
                                      ;  Could be bumped to about 780
                                      ;  for 25MHz operation, but
                                      ;  performance difference will be
                                      ;  trivial.

        outpw DRAM_RAS,        4000   ;100us maximum RAS low time.
                                      ;  Again this is at 20MHz.
                                      ;  Since this number is greater
                                      ;  than the refresh cycle time,
                                      ;  it is irrelevant, as refresh
                                      ;  will force RAS low.  This would
                                      ;  be 5000 at 25MHz, but as long as
                                      ;  this is greater than the
                                      ;  refresh time, it is irrelevant.

        outpw DRAM_CONTROL,    0903h
                                                ;Parity Error NMI Disabled
                                                ;Bank 1, 9 row addresses
                                                ;Bank 0, 9 row addresses
                                                ;Three Cycle Page Miss
                                                ;Normal CAS Low
                                                ;Normal Refresh
                                                ;Parity Disabled
                                                ;Bank 1 present
                                                ;Bank 0 present

        ; Wait minimum 100us specified in Spec

        mov     cx, DRAM_COUNT  ; defined at top of file
centuryus:
        dec     cx
        jnz     centuryus

        ; Now make sure the DRAM ready bit is set

        mov     dx, 0ef86h
eskimo:
        in      ax, dx
        test    ax, 01h
        jz      eskimo

;---------------------------------------------------------------------------
; see if we have 4MB or greater
;---------------------------------------------------------------------------
; configure upper bank of DRAM for 2MB at 2MB.  Then see if address        
; 0x200000 == 0x200400.  If they do not, the DRAM page size is at least
; 10, which is a 4MB or bigger SIMM.

size_4mb:

        outpb DRAM_ADDR1,      004h   ;Bank 1 at 2MB
        outpb DRAM_BANK,       020h   ;Bank 1 2MB, Bank 0 .5 MB
        outpb DRAM_MASK1,      0FCh   ;Mask for 2 megabytes
        outpw DRAM_CONTROL,    01103h ;bank 1 now 10 bit page size

        mov     esi, 0200000h           ; first address to check
        mov     edi, 0200400h           ; second address to check

        ; first check

        mov     eax, dword ptr [esi]    ; save first value
        mov     ebx, dword ptr [edi]    ; save second value
        mov     dword ptr [esi], MAGIC1 ; write first value to DRAM
        mov     dword ptr [edi], MAGIC2 ; write second value to DRAM
        mov     ecx, dword ptr [esi]    ; read first value
        mov     edx, dword ptr [edi]    ; read second value
        mov     dword ptr [esi], eax    ; restore first value
        mov     dword ptr [edi], ebx    ; restore second value
        cmp     ecx, MAGIC1             ; check first
        jne     fail_4mb                ; if wrong, bail out
        cmp     edx, MAGIC2             ; check second
        jne     fail_4mb                ; if wrong, bail out

        ; second check

        mov     ebx, dword ptr [edi]    ; save second value
        mov     eax, dword ptr [esi]    ; save first value
        mov     dword ptr [edi], MAGIC1 ; write second value to DRAM
        mov     dword ptr [esi], MAGIC2 ; write first value to DRAM
        mov     edx, dword ptr [edi]    ; read second value
        mov     ecx, dword ptr [esi]    ; read first value
        mov     dword ptr [edi], ebx    ; restore second value
        mov     dword ptr [esi], eax    ; restore first value
        cmp     edx, MAGIC1             ; check second
        jne     fail_4mb                ; if wrong, bail out
        cmp     ecx, MAGIC2             ; check first
        jne     fail_4mb                ; if wrong, bail out

pass_4mb:

        ; setup bank 0 for 2MB also

        outpb DRAM_BANK,       022h     ;Bank 1 2MB, Bank 0 2 MB
        outpb DRAM_MASK0,      0FCh     ;Mask for 2 megabytes
        outpw DRAM_CONTROL,    01203h   ;both banks 10 bit page size
        
        ; see if really 16MB SIMM

        jmp     size_16mb               ; lets check for 16MB

fail_4mb:
        
        ; restore 1MB setting

        outpw DRAM_CONTROL,    00903h ;bank 1 back to 9 bit page size
        outpb DRAM_MASK1,      0FFh   ;Mask for .5 megabytes
        outpb DRAM_BANK,       000h   ;Bank 1 .5 MB, Bank 0 .5 MB
        outpb DRAM_ADDR1,      001h   ;Bank 1 at .5 MB

        ; abort test

        jmp     end_dram_sizing         ; all done

;---------------------------------------------------------------------------
; see if we have 16MB or greater
;---------------------------------------------------------------------------
; configure upper bank of DRAM for 8MB at 8MB.  Then see if address        
; 0x800000 == 0x800800.  If they do not, the DRAM page size is at least
; 11, which is a 16MB or bigger SIMM.

size_16mb:

        outpb DRAM_ADDR1,      010h   ;Bank 1 at 8MB
        outpb DRAM_BANK,       042h   ;Bank 1 8MB, Bank 0 2 MB
        outpb DRAM_MASK1,      0F0h   ;Mask for 8 megabytes
        outpw DRAM_CONTROL,    01A03h ;bank 1 now 11 bit page size

        mov     esi, 0800000h           ; first address to check
        mov     edi, 0800800h           ; second address to check

        ; first check

        mov     eax, dword ptr [esi]    ; save first value
        mov     ebx, dword ptr [edi]    ; save second value
        mov     dword ptr [esi], MAGIC1 ; write first value to DRAM
        mov     dword ptr [edi], MAGIC2 ; write second value to DRAM
        mov     ecx, dword ptr [esi]    ; read first value
        mov     edx, dword ptr [edi]    ; read second value
        mov     dword ptr [esi], eax    ; restore first value
        mov     dword ptr [edi], ebx    ; restore second value
        cmp     ecx, MAGIC1             ; check first
        jne     fail_16mb                ; if wrong, bail out
        cmp     edx, MAGIC2             ; check second
        jne     fail_16mb                ; if wrong, bail out

        ; second check

        mov     ebx, dword ptr [edi]    ; save second value
        mov     eax, dword ptr [esi]    ; save first value
        mov     dword ptr [edi], MAGIC1 ; write second value to DRAM
        mov     dword ptr [esi], MAGIC2 ; write first value to DRAM
        mov     edx, dword ptr [edi]    ; read second value
        mov     ecx, dword ptr [esi]    ; read first value
        mov     dword ptr [edi], ebx    ; restore second value
        mov     dword ptr [esi], eax    ; restore first value
        cmp     edx, MAGIC1             ; check second
        jne     fail_16mb                ; if wrong, bail out
        cmp     ecx, MAGIC2             ; check first
        jne     fail_16mb                ; if wrong, bail out

pass_16mb:

        ; set up bank 0 for 8MB also

        outpb DRAM_BANK,       044h     ;Bank 1 8MB, Bank 0 8 MB
        outpb DRAM_MASK0,      0F0h     ;Mask for 8 megabytes
        outpw DRAM_CONTROL,    01B03h   ;both banks 11 bit page size
        
        ; all done with sizing

        jmp     end_dram_sizing         ; all done
        
fail_16mb:
        
        ; restore 4MB setting
        
        outpw DRAM_CONTROL,    01203h ;bank 1 now 10 bit page size
        outpb DRAM_MASK1,      0FCh   ;Mask for 2 megabytes
        outpb DRAM_BANK,       022h   ;Bank 1 2MB, Bank 0 2 MB
        outpb DRAM_ADDR1,      004h   ;Bank 1 at 2MB
        
        ; all done with sizing

        jmp     end_dram_sizing         ; all done

;---------------------------------------------------------------------------
; end of DRAM sizing
;---------------------------------------------------------------------------

end_dram_sizing:

        ; you could initialize DRAM to a known value here if desired

;---------------------------------------------------------------------------
; Jump to user code
;---------------------------------------------------------------------------

        mov eax, jump_location
        jmp eax
        
;---------------------------------------------------------------------------
; Thats it.
;---------------------------------------------------------------------------
        
        HLT

;---------------------------------------------------------------------------

DRAMSIZE  ENDP
CODESG    ENDS
        
; end of program - list entry proc (FAR)

          END     DRAMSIZE

;---------------------------------------------------------------------------
; END       dramsize.asm 
;---------------------------------------------------------------------------

