/* @(#)vnode.h	7.14 88/10/26 17:21:41
 *
 * vnode, gnode and vnode operations structures and defines
 *
 * COPYRIGHT 1987, 1988 IBM CORP.
 *
 * defines:
 *	struct vnode
 *	struct gnode
 *	struct vnodeops
 *	vnodeops calling macros
 *	struct exec_data
 *	struct open_data
 *	struct create_data
 *
 */

#ifndef _h_VNODE
#define _h_VNODE

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/stat.h>

/*
 * vnode ("virtual inode") structure
 *	vnodes are used to keep the name and place of an object in the
 * filesystem straight, so there may be more than one vnode per object.
 * A "gnode" is used to abstract the object itself.
 * NOTE: there is always at least one vnode per gnode.
 */
struct vnode {
        ushort	v_flag;		/* see definitions below */
        ushort	v_count;	/* the use count of this vnode */
	int	v_lock[2];	/* for locking */
        struct vfs *v_vfsp;	/* pointer to the vfs of this vnode */
	int	v_vfsgen;	/* generation number for the vfs */
        struct vfs *v_mvfsp;	/* pointer to vfs which was mounted over this */
				/* vnode; NULL if no mount has occurred */
	struct gnode *v_gnode;	/* ptr to implementation gnode */
	struct vnode *v_next;	/* ptr to other vnodes that share same gnode */
	struct vnode *v_vfsnext; /* ptr to next vnode on list off of vfs */
	struct vnode *v_vfsprev; /* ptr to prev vnode on list off of vfs */
};

/*
 * Definitions for v_flag field
 * of vnode structure.
 */
#define V_ROOT		0x01		/* vnode is the root of a vfs	*/
#ifdef SUN_VNODE
#define VROOT V_ROOT
#endif /*SUN_VNODE*/
#define VFS_UNMOUNTED	0x02		/* vnode's vfs has been	unmounted */

#define V_INTRANSIT	0x04		/* vnode is midway through	*/
					/*   vfs_vget processing	*/
#define V_DMA		0x08		/* buffer bypass		*/
#define V_TEXT		0x10		/* currently being executed	*/
#define V_RMDC		0x20		/* Usable by remote directory	*/
					/*   cache			*/
#define V_RENAME	0x40		/* Rename is in process         */
#define V_LOCK		0x80		/* Serialize exec's		*/

/* BEGIN DUBIOUS - we will probably change this */
/*
 * Vnode hashing grunt
 */
#define NHVNODE 256			/* Should be a power of two	*/

struct hvnode
{
	struct vnode *v_next;		/* Vnode pointer		*/
};
extern struct hvnode hvnode[];

#define VN_LOCK(vp)	while (((struct vnode *)(vp))->v_flag & V_LOCK) \
				sleep(&(((struct vnode *)(vp))->v_flag),PZERO);\
			((struct vnode *)(vp))->v_flag |= V_LOCK;

#define VN_UNLOCK(vp)	wakeup(&(((struct vnode *)(vp))->v_flag));\
			((struct vnode *)(vp))->v_flag &= ~V_LOCK;


/* END DUBIOUS - we will probably change this */


/* this is only used as a template for per vfs data attached to gnodes */
struct gn_vfsdata {
	struct gn_vfsdata *gnv_next;	/* next in chain, NULL ends chain */
	struct gnode	*gnv_gnode;	/* pointer back to gnode */
	int	gn_gfstype;		/* gfs type of this vfs */
	/* vfs specific stuff here */
};

/*
 * gnode type is compressed to fit into a char, so it is not an enum, but
 * it uses the defines as given in the enum herein.
 * DO NOT rearrange/redefine these first 10!
 *
 * VNON means no type.
 */
enum vtype { VNON, VREG, VDIR, VBLK, VCHR, VLNK, VSOCK, VBAD, VFIFO, VMPC };
#define VUNDEF	VNON		/* undefined is same as nothing */

/*
 * gnode structure - represents a real object in the file system;
 * i.e. there is a 1:1 correspondance between an gnode and a file on a disk.
 */
struct gnode {
	enum vtype gn_type;	/* type of object: VDIR,VREG,... */
	short	gn_flags;	/* attributes of object, only MPX now */
	ulong	gn_seg;			/* segment into which file is mapped */
	cnt_t	gn_segcnt;		/* segment use count */
	cnt_t	gn_rdcnt;		/* total opens for read */
	cnt_t	gn_wrcnt;		/* total opens for write */
	cnt_t	gn_excnt;		/* total opens for exec */
	struct vnodeops *gn_ops;
	struct vnode *gn_vnode;	/* ptr to list of vnodes per this gnode */
	int	gn_lock[2];	/* lock for: gnode, its vnodes list, ... */
	dev_t	gn_rdev;	/* for devices, their "dev_t" */

	long	gn_tok[4];	/* 4 longs reserved for a token mgmt scheme */
				/* either gnode tokens or full vfs tokens */

	struct gn_vfsdata *gn_vfsdp;	/* ptr to linked list of per vfs data */
				/* i.e. DS's per file per client lists */

	caddr_t	gn_data;	/* ptr to private data (usually contiguous) */
};

/*
 * definitions of gn_flags
 */
#define	GNF_TCB		0x0001	/* gnode corresponds to a file in the TCB */

/*
 * some defines so old things keep working
 */
#define v_type		v_gnode->gn_type
#define v_vntype	v_gnode->gn_type
#define v_rdev		v_gnode->gn_rdev
#define v_data		v_gnode->gn_data
/* NOTE v_ops is NOT defined, so the VNOP macros/glue will be only path! */

struct vnodeops {
	/* creation/naming/deletion */
	int	(*vn_link)();		/* (vp,dp,tnm) */
	int	(*vn_mkdir)();		/* (vp,dirp,mode) */
	int	(*vn_mknod)();		/* (vp,nm,mode,dev) */
	int	(*vn_remove)();		/* (dp,nm,vp) */
	int	(*vn_rename)();		/* (vp,dp,nm,tp,tdp,tnm) */
	int	(*vn_rmdir)();		/* (vp,dp,nm) */
	/* lookup, file handle stuff */
	int	(*vn_lookup)();		/* (dp,vpp,dirp,ubuf,flags) */
	int	(*vn_fid)();		/* (vp,fhp) */
	/* access to files */
	int	(*vn_open)();		/* (vp,flags,ext,vinfop,exec_data) */
	int	(*vn_create)();		/* (dp,vpp,flags,dirp,mode,vinfop) */
	int	(*vn_hold)();		/* (vp) */
	int	(*vn_rele)();		/* (vp) */
	int	(*vn_close)();		/* (vp, flags,channel,ext,vinfo) */
	int	(*vn_map)();		/* (vp,vinfo) */
	int	(*vn_unmap)();		/* (vp,vinfo) */
	/* manipulate attributes of files */
	int	(*vn_access)();		/* (vp, mode) */
	int	(*vn_getattr)();	/* (vp,ubuf) */
	int	(*vn_setattr)();	/* (vp,op,arg1,arg2,arg3) */
	/* data update operations */
	int	(*vn_fclear)();		/* (vp,flags,offp,len,vinfo) */
	int	(*vn_fsync)();		/* (vp,flags,vinfo) */
	int	(*vn_ftrunc)();		/* (vp,flags,length,vinfo) */
	int	(*vn_rdwr)();		/* (vp,op,flags,offp,uiop,ext,vinfo) */
	int	(*vn_lockctl)();	/* (vp,offset,lckdat,cmd,vinfo) */
	/* extensions */
	int	(*vn_ioctl)();		/* (vp,flags,cmd,arg,vinfo,ext) */
	int	(*vn_readlink)();	/* (vp,buf,len) */
	int	(*vn_select)();		/* (vp, whence, vinfo, mpx ) */
	int	(*vn_symlink)();	/* (vp,dirp,link) */
	int	(*vn_readdir)();	/* (vp,uiop) */
	/* buffer ops */
	int	(*vn_strategy)();	/* (vp,bp) */
	/* security things */
	int	(*vn_revoke)();		/* (vp) */
};

#ifdef KERNEL
/*
 * define macros for calling vnode operations functions
 */
#define VNOP_ACCESS(vp,mode,edp) \
	(*(*(vp)->v_gnode->gn_ops->vn_access))(vp, mode,edp)
#define VNOP_CLOSE(vp,flags,channel,ext,vinfo) \
	(*(*(vp)->v_gnode->gn_ops->vn_close))(vp, flags,channel,ext,vinfo)
#define VNOP_CREATE(dp,vpp,flags,name,mode,vinfo) \
	(*(*(dp)->v_gnode->gn_ops->vn_create))(dp,vpp,flags,name,mode,vinfo)
#define VNOP_FCLEAR(vp,flags,offp,len,vinfo) \
	(*(*(vp)->v_gnode->gn_ops->vn_fclear))(vp,flags,offp,len,vinfo)
#define VNOP_FID(vp,fid) \
	(*(*(vp)->v_gnode->gn_ops->vn_fid))(vp,fid)
#define VNOP_FSYNC(vp,flags,vinfo) \
	(*(*(vp)->v_gnode->gn_ops->vn_fsync))(vp,flags,vinfo)
#define VNOP_FTRUNC(vp,flags,length,vinfo) \
	(*(*(vp)->v_gnode->gn_ops->vn_ftrunc))(vp,flags,length,vinfo)
#define VNOP_GETATTR(vp,vattr,cmd,flag) \
	(*(*(vp)->v_gnode->gn_ops->vn_getattr))(vp,vattr,cmd,flag)
#define VNOP_REVOKE(vp,cmd,flags,vattrp) \
	(*(*(vp)->v_gnode->gn_ops->vn_revoke))(vp,cmd,flags,vattrp)
#define VNOP_HOLD(vp) \
	(*(*(vp)->v_gnode->gn_ops->vn_hold))(vp)
#define VNOP_IOCTL(vp,cmd,arg,flags,chan,ext) \
	(*(*(vp)->v_gnode->gn_ops->vn_ioctl))(vp,cmd,arg,flags,chan,ext)
#define VNOP_LINK(vp,dp,tnm) \
	(*(*(vp)->v_gnode->gn_ops->vn_link))(vp,dp,tnm)
#define VNOP_LOCKCTL(vp,flags,offset,lckdat,cmd,vinfo) \
	(*(*(vp)->v_gnode->gn_ops->vn_lockctl))(vp,flags,offset,lckdat,cmd,vinfo)
#define VNOP_LOOKUP(dp,vpp,nam,flag) \
	(*(*(dp)->v_gnode->gn_ops->vn_lookup))(dp,vpp,nam,flag)
#define VNOP_MAP(vp,addr,flag,fd) \
	(*(*(vp)->v_gnode->gn_ops->vn_map))(vp,addr,flag,fd)
#define VNOP_MKDIR(vp,nam,mode) \
	(*(*(vp)->v_gnode->gn_ops->vn_mkdir))(vp,nam,mode)
#define VNOP_MKNOD(vp,nm,mode,dev) \
	(*(*(vp)->v_gnode->gn_ops->vn_mknod))(vp,nm,mode,dev)
#define VNOP_OPEN(vp,flags,ext,vinfop,exec_data) \
	(*(*(vp)->v_gnode->gn_ops->vn_open))(vp,flags,ext,vinfop,exec_data)
#define VNOP_SELECT(vp,whence,vinfo,mpx) \
	(*(*(vp)->v_gnode->gn_ops->vn_select))(vp,whence,vinfo,mpx)
#define VNOP_RDWR(vp,op,flags,offp,uiop,ext,vinfo,fd) \
	(*(*(vp)->v_gnode->gn_ops->vn_rdwr))(vp,op,flags,offp,uiop,ext,vinfo,fd)
#define VNOP_READDIR(vp,uiop) \
	(*(*(vp)->v_gnode->gn_ops->vn_readdir))(vp,uiop)
#define VNOP_READLINK(vp,uiop) \
	(*(*(vp)->v_gnode->gn_ops->vn_readlink))(vp,uiop)
#define VNOP_RELE(vp) \
	(*(*(vp)->v_gnode->gn_ops->vn_rele))(vp)
#define VNOP_REMOVE(dp,vp,nm) \
	(*(*(dp)->v_gnode->gn_ops->vn_remove))(dp,vp,nm)
#define VNOP_RENAME(vp,dp,nm,tp,tdp,tnm) \
	(*(*(vp)->v_gnode->gn_ops->vn_rename))(vp,dp,nm,tp,tdp,tnm)
#define VNOP_RMDIR(vp,dp,nm) \
	(*(*(vp)->v_gnode->gn_ops->vn_rmdir))(vp,dp,nm)
#define VNOP_SETATTR(vp,op,arg1,arg2,arg3) \
	(*(*(vp)->v_gnode->gn_ops->vn_setattr))(vp,op,arg1,arg2,arg3)
#define VNOP_STRATEGY(vp,bp) \
	(*(*(vp)->v_gnode->gn_ops->vn_strategy))(vp,bp)
#define VNOP_SYMLINK(vp,lnm,tnm) \
	(*(*(vp)->v_gnode->gn_ops->vn_symlink))(vp,lnm,tnm)
#define VNOP_UNMAP(vp,addr) \
	(*(*(vp)->v_gnode->gn_ops->vn_unmap))(vp,addr)
#endif /*KERNEL*/


/* for Sun vnode compatibility, they expect vattr to be in vnode.h */
#ifdef SUN_VNODE
#include <sys/vattr.h>
enum vcexcl	{ NONEXCL, EXCL};	/* (non)excl create (create) */
#endif /*SUN_VNODE*/


/*
 * The exec structure is used for return information passed back
 * from the vn_open and vn_access routines if the mode is exec.
 */
struct exec_data {
	ushort	exec_perm;   	/* file's permission mode */
	ushort	exec_uid;	/* owner's translated uid */
	ushort	exec_gid;	/* owner's translated gid */
};

/* the low order 12 bits of exec_perm are used to return the inode's mode */
/* field, the high order 4 bits are used by distributed services to pass  */
/* information about execute permission between a client and a server */

#define	SEXEC_OK	0x1000		/* server exec permission ok */
#define CEXEC_OK	0x2000		/* client exec permission ok */
#define CLIENT_EXEC_CHK	0x4000		/* check execute permission at the */
					/* client 			   */

/*
 * Convert inode formats to vnode types
 */
extern enum vtype iftovt_tab[];		/* these are located in stat.c */
extern int   vttoif_tab[];

#define IFTOVT(M)	(iftovt_tab[((M) & S_IFMT) >> 12])
#define VTTOIF(T)	(vttoif_tab[(int)(T)])

#define MAKEIMODE(T, M)	(VTTOIF(T) | (M))

#endif
