/* @(#)inode.h	7.11 88/10/06 10:53:51 */

/*
 * The I node is the focus of all
 * file activity in unix. There is a unique
 * inode allocated for each active file,
 * each current directory, each mounted-on
 * file, text file, and the root. An inode is 'named'
 * by its dev/inumber pair. (iget/iget.c)
 * Data, from mode on, is read in
 * from permanent inode on volume.
 */

#ifndef _h_INODE
#define _h_INODE

#include <sys/types.h>
#include <sys/seg.h>
#include <sys/vnode.h>

#define	NADDR	13
#define	NSADDR	(NADDR*sizeof(daddr_t)/sizeof(short))
#define	NFADDR	8
#define	PIPSIZ	NFADDR*BSIZE

#define	NHINO	128	/* must be power of 2 */
#define	IHASH(X)	(&ihead[(int)(X) & (NHINO-1)])

struct	inode {
	/* for AIX, inode contains the gnode. Leave gnode first,
	 * so ITOGP and GTOIP macros work.
	 */
	struct gnode	i_gnode;
	struct inode	*i_forw;	/* hash chain forw */
	struct inode	*i_back;	/* hash chain back */
	struct inode	*i_freef;	/* lru chain forw */
	struct inode	**i_freeb;	/* lru chain back */
	short		i_flag;
	short		i_sflag;	/* s-mode and cache flags */
	ushort 		i_mode;		/* permission bits and file type */
	cnt_t		i_count;	/* reference count */
	off_t		i_size;		/* size of file */
	time_t		i_mtime;	/* server modification time */
	struct filock	*i_filocks;	/* locked region list */
	struct mapnode	*i_mapn;	/* for mapped file */
	dev_t		i_dev;		/* device where inode resides */
	ino_t		i_number;	/* i number, 1-to-1 with device */
	char		i_gen;		/* inode generation number */
	short		i_nlink;	/* directory entries */
	ushort		i_uid;		/* owner */
	ushort		i_gid;		/* group of owner */
	struct {
		union {
			daddr_t i_a[NADDR];	/* if normal file/directory */
						/* for pipes, with select */
			struct {
			daddr_t	i__faddr[NFADDR];	/* fifo block ptrs */
			struct proc *i__frsel;	/* ptr for read select */
			struct proc *i__fwsel;	/* ptr for write select */
			/* except select NOT implemented, takes up space! */
			short	i__frptr;
			short	i__fwptr;
			short	i__frcnt;
			short	i__fwcnt;
			short	i__fflag;
			short	i__fpad;	/* nothing */
			} i__f;			
		} i_p;
		daddr_t	i_l;	/* last logical block read (for read-ahead) */
	} i_blks;
	struct  file_access *i_fa;  /* file access structure pointer */
	time_t		i_atime;	/* cached access time */
	time_t		i_ctime;	/* cached changed time */
};

extern struct inode inode[];	/* The inode table itself */
extern struct locklist locklist[]; /* The lock table itself */
#ifndef NFLOCKS
#define NFLOCKS NINODE
#endif

/* flags */
#define	ILOCK	00001		/* inode is locked */
#define	IUPD	00002		/* file has been modified */
#define	IACC	00004		/* inode access time to be updated */
#define	IMOUNT	00010		/* inode is mounted on */
#define	IWANT	00020		/* some process waiting on lock */
#define	ITEXT	00040		/* inode is pure text prototype */
#define	ICHG	00100		/* inode has been changed */
#define	ISYN	00200		/* do synchronous write for iupdate */
#define	INAMI	00400		/* set when namei() calls iput() */
#define	IZOMB	01000		/* this inode is unavailable. */
#define	IDMA	02000		/* use DMA when possible */
#define	IMAP	04000		/* mapped file */

/* sflags */
#define	IFSWARN	00001		/* client sync for near-full filesystem */
#define	IFSFULL	00002		/* async write failed: full filesystem */
#define	ISASYNC	00004		/* inode s_mode async */
#define	ISFSYNC	00010		/* inode s_mode fullsync */
#define	ISRONLY	00020		/* inode s_mode read-only */
#define	ISSMODE	00034		/* mask for smodes */
#define	FALOCK	00100		/* file access structure list is locked */
#define	FAWANT	00200		/* file access structure list is wanted */

/* modes */
#define	IFMT	0170000		/* type of file */
#define	IFDIR	0040000		/* directory */
#define	IFCHR	0020000		/* character special */
#define	IFBLK	0060000		/* block special */
#define	IFREG	0100000		/* regular */
#define	IFMPC	0030000		/* multiplexed char special - obsolete */
#define	IFMPB	0070000		/* multiplexed block special - obsolete */
#define	IFIFO	0010000		/* fifo special */
#define	IFLNK	0120000		/* symbolic link */
#define	ISUID	0004000		/* set user id on execution */
#define	ISGID	0002000		/* set group id on execution */
#define	ISVTX	0001000		/* save swapped text even after use */
#define	IREAD	0000400		/* read, write, execute permissions */
#define	IWRITE	0000200
#define	IEXEC	0000100

/*
 * iblock is used in bmap()/iclear() to determine where a block index
 * is stored in aindirect block
 */

struct iblock {
	daddr_t	ibk_bn;		/* logical block number */
	int	ibk_off;	/* index into indirect block */
};

#define	i_addr	i_blks.i_p.i_a
#define	i_lastr	i_blks.i_l
#define	i_rdev	i_blks.i_p.i_a[0]
#define i_chan	i_blks.i_p.i_a[1]
#define i_pino	i_blks.i_p.i_a[2]

#define	i_faddr	i_blks.i_p.i_a
#define	i_frptr	i_blks.i_p.i__f.i__frptr
#define	i_fwptr	i_blks.i_p.i__f.i__fwptr
#define	i_frcnt	i_blks.i_p.i__f.i__frcnt
#define	i_fwcnt	i_blks.i_p.i__f.i__fwcnt
#define	i_fflag	i_blks.i_p.i__f.i__fflag
#define	i_frsel	i_blks.i_p.i__f.i__frsel
#define	i_fwsel	i_blks.i_p.i__f.i__fwsel

#define	IFIR	01
#define	IFIW	02
/* define select collision flags */
#define IFRCOL	010		/* read select collision */
#define IFWCOL	020		/* write select collision */
				/* exception select NOT implemented */

/*
 * file access structure, linked to server real inode,
 * used to tie together real inode and client surrogate inode
 */

struct file_access {
	struct file_access *fa_next;	/* file access structure pointer */
	short fa_flag;			/* flags */
	short fa_count;			/* use count */
	short fa_orocnt;		/* open read/only count */
	short fa_orwcnt;		/* open read/write count */
	short fa_txtcnt;		/* executing process count */
	struct node *fa_node;		/* node structure pointer */
	int fa_nid;			/* node id */
	struct inode *fa_sip;		/* surrogate inode pointer */
};

/* flags */
#define	FACCINV	01		/* client cache invalid */

/* return code from faput */
#define	DOSMODEPROC	1	/* indicates that smode processing needs */
				/* to be done */

/* AIX Version 2.2.1 security enhancement */
/*
 * mpx_list used to construct an association table for multiplexed files.
 * see aix_lookup.c
 */
struct mpx_list {
	struct mpx_list *mpx_for;
	struct mpx_list *mpx_bak;
	ino_t           mpx_ino;
	dev_t           mpx_dev;
	dev_t           mpx_rdev;
	caddr_t         mpx_chan;
};
 
#define ISMPX(mode) (((mode) & (IFCHR | ISVTX)) == (IFCHR | ISVTX))
/* TCSEC Division C Class C2 */
 

#define VTOIP(x)	((struct inode *)((x)->v_gnode->gn_data))
#define GTOIP(x)	((struct inode *)(x))
#define ITOGP(x)	((struct gnode *)(x))

#endif
