/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

	Example XVT application that displays "Hello World" or "Goodbye World".
	User can choose the message by double clicking, typing a "G" or an "H", or
	selecting from the Choice menu. User can also choose the font to be used.
	Additional windows can be opened by choosing New on the File menu.
*/
#define INTPTR_CHK 0 /* elim. requirement for INTPTR casts */
#include "xvt.h"								/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"							/* standard XVT menu tags */

typedef struct {
	enum {HELLO, GOODBYE} choice;				/* message to display */
	FONT font;									/* font */
} DOCUMENT;										/* contents of one document */

#ifdef PROTO
STATICFCN void new_doc(WINDOW);
STATICFCN void dispose_doc(DOCUMENT *);
STATICFCN void do_update(WINDOW);
STATICFCN void fix_choice_menu(DOCUMENT *);
STATICFCN void do_menu(MENU_TAG);
#else
STATICFCN void new_doc();
STATICFCN void dispose_doc();
STATICFCN void do_update();
STATICFCN void fix_choice_menu();
STATICFCN void do_menu();
#endif

#define M_CHOICE_HELLO	MAKE_MENU_TAG(MENU3, 1)	/* tags for our Choice menu */
#define M_CHOICE_GDBYE	MAKE_MENU_TAG(MENU3, 2)

#define STARTX			30						/* x coordinate for message */
#define STARTY			40						/* y coordinate for message */

APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"hello",						/* application's name */
	W_PLAIN,						/* type of initial window */
	FALSE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	FALSE,							/* close box on initial window? */
	TRUE,							/* want std. font menu? (includes sizes) */
	TRUE							/* want std. style menu? */
};

/*
	Function to associate a DOCUMENT structure with a new window. Each window
	carries a long word of data that can be set and accessed by the application.
	We use it to hold a pointer to a "document," which for this simple example
	consists of only the message choice and its font and style. More generally,
	a document structure would hold text, a picture, etc.
*/
#ifdef FPROTO
static void new_doc(WINDOW win)
#else
static void new_doc(win)
WINDOW win;								/* window to be initialized */
#endif
{
	DOCUMENT *d;

	if ((d = (DOCUMENT *)malloc(sizeof(DOCUMENT))) == NULL)
		fatal("Out of memory.");
	d->choice = HELLO;
	d->font = normal_font;
	set_app_data(win, PTR_LONG(d));
}

/*
	Function to destroy a document strucutre, to be called when a window is
	closed (assuming that a document appears in only one window).
*/
#ifdef FPROTO
static void dispose_doc(DOCUMENT *d)
#else
static void dispose_doc(d)
DOCUMENT *d;
#endif							/* document to be destroyed */
{
	free((char *)d);
}

/*
	Function called by XVT to initialize application. It selects the font for the
	initial window (std_win), makes sure that it's marked appropriately in the
	Font/Style menu, initializes the window's document, sets a check mark in the
	Choice menu, and enables the New item on the File menu.
*/
BOOLEAN XVTENTRY  appl_init BTCENTRY(void)
{
	set_cur_window(std_win);
	set_font(&big_font, FALSE);
	set_font_menu(&big_font);
	new_doc(std_win);
	menu_check(M_CHOICE_HELLO, TRUE);
	menu_enable(M_FILE_NEW, TRUE);
	note("A note");
	note("another Note");
	return(TRUE);
}

/*
	Function to handle update events. The screen is painted white and then the
	chosen message is drawn inside a rounded rectangle. To fit the shape
	properly, the message is first measured in width and height, and a rectangle
	is initialized that fits the text with a margin all around that's equal to
	the descent. (Nothing magical about the descent - it's just a convenient
	amount that's proportional to the size of the font.)
*/
#ifdef FPROTO
static void do_update(WINDOW win)
#else
static void do_update(win)
WINDOW win;
#endif							/* window to be updated */
{
	RCT rct;
	char *msg;
	int ascent, descent, width, corner;
	DOCUMENT *d = (DOCUMENT *)get_app_data(win);

	get_client_rect(win, &rct);
	set_cpen(&white_cpen);
	set_cbrush(&white_cbrush);
	draw_rect(&rct);
	msg = d->choice == HELLO ? "Hello World!" : "Goodbye World!";
	set_font(&d->font, FALSE);
	get_font_metrics(NULL, &ascent, &descent);
	width = get_text_width(msg, -1);
	set_rect(&rct, STARTX - descent, STARTY - descent, STARTX + width + descent,
	  STARTY + ascent + 2 * descent);
	corner = (rct.bottom - rct.top) / 3;
	set_cpen(&black_cpen);
	draw_roundrect(&rct, corner, corner);
	draw_text(STARTX, STARTY + ascent, msg, -1);
}

/* Function to set check marks on the Choice menu. */
#ifdef FPROTO
static void fix_choice_menu(DOCUMENT *d)
#else
static void fix_choice_menu(d)
DOCUMENT *d;							/* active document */
#endif
{
	menu_check(M_CHOICE_HELLO, d->choice == HELLO);
	menu_check(M_CHOICE_GDBYE, d->choice == GOODBYE);
}

	
/* Function to handle menu commands. */
#ifdef FPROTO
static void do_menu(MENU_TAG cmd)
#else
static void do_menu(cmd)
MENU_TAG cmd;							/* tag identifying menu item */
#endif
{
	char title[50];
	RCT rct;
	static int count = 0;
	WINDOW win;
	DOCUMENT *d;

	switch (cmd) {
	case M_FILE_NEW:
		set_rect(&rct, 100, 100, 300, 200);
		sprintf(title, "Window %d", ++count);
		if ((win = new_window(&rct, title, W_DOC, TRUE, FALSE, FALSE, TRUE)) == NULL_WIN)
			error("Can't create window.");
		new_doc(win);
		break;
	case M_FILE_QUIT:
		terminate();
		break;
	case M_CHOICE_HELLO:
	case M_CHOICE_GDBYE:
		if ((win = get_front_window()) != NULL_WIN) {
			d = (DOCUMENT *)get_app_data(win);
			label:
			d->choice = cmd == M_CHOICE_HELLO ? HELLO : GOODBYE;
			fix_choice_menu(d);
			invalidate_rect(win, NULL);
		}
	}
}

/*
	Main application entry point. After changing the message choice or the font,
	the entire client area of the window is invalidated so as to force an update.
	On a font event (user selected from the Font/Style menu) the font associated
	with the document is changed and the menus themselves are modified to show
	the current font.

	An E_QUIT event is generated by the system when it want to shut down. In
	phase one, each running application is polled as to whether it is willing
	to quit (our little application is, so it calls quit_OK). If the vote was
	unanimously positive, in phase two each application is ordered to quit. If
	any application declined in phase one, phase two is skipped and the system
	stays up. (Not all window systems use this protocol.)
*/
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;								/* window receiving event, if any */
EVENT_PTR ep;								/* pointer to event structure */
#endif
{
	DOCUMENT *d = NULL;

	if (win == NULL_WIN)
		win = get_front_window();
	if (win != NULL_WIN)
		d = (DOCUMENT *)get_app_data(win);
	switch (ep->type) {
	case E_MOUSE_DBL:
		if (d->choice == HELLO)
			d->choice = GOODBYE;
		else
			d->choice = HELLO;
		invalidate_rect(win, NULL);
		break;
	case E_CHAR:
		if (d != NULL) {
			switch (toupper(ep->v.chr.ch)) {
			case 'G':
				d->choice = GOODBYE;
				break;
			case 'H':
				d->choice = HELLO;
			}
			invalidate_rect(win, NULL);
		}
		note("A Character");
		note("Another note about the character");
		break;
	case E_UPDATE:
		do_update(win);
		break;
	case E_ACTIVATE:
		if (ep->v.active && d != NULL) { /* make menu state reflect document */
			set_font_menu(&d->font);
			fix_choice_menu(d);
		}
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag);
		break;
	case E_FONT:
		if (d != NULL) {
			d->font = ep->v.font.font;
			set_font_menu(&ep->v.font.font);
			invalidate_rect(win, NULL);
		}
		break;
	case E_CLOSE:
		if (d != NULL)
			dispose_doc(d);
		close_window(win);
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
	}
}

/* Function called by XVT just before terminating application. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
	/* no cleanup (e.g., files to be removed) needed */
}
