/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

	Example XVT application, showing:
	
	- a standard skeleton for all XVT applications
	- how to use abstract fonts
	- how to create windows of different styles with new_window
	- how to put up a dialog box and organize a callback function
*/
#include "xvt.h"					/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"				/* standard XVT menu tags */
#include "examp.h"					/* this app's menu tags and dialog IDs */

#ifdef PROTO
STATICFCN BOOLEAN XVTENTRY cb_opt BTCENTRY(int, CONTROL_INFO *);
STATICFCN BOOLEAN opt_dialog(void);
STATICFCN void change_num_wins(BOOLEAN);
STATICFCN void build_window(WIN_TYPE);
STATICFCN void do_close(WINDOW);
STATICFCN BOOLEAN save_document(void);
STATICFCN void discard_window(void);
STATICFCN BOOLEAN quit_approved(void);
STATICFCN void do_menu(MENU_TAG, BOOLEAN, BOOLEAN);
STATICFCN void do_update(WINDOW);
#else
STATICFCN BOOLEAN XVTENTRY cb_opt BTCENTRY();
STATICFCN BOOLEAN opt_dialog();
STATICFCN void change_num_wins();
STATICFCN void build_window();
STATICFCN void do_close();
STATICFCN BOOLEAN save_document();
STATICFCN void discard_window();
STATICFCN BOOLEAN quit_approved();
STATICFCN void do_menu();
STATICFCN void do_update();
#endif

/* Required application setup structure. */
APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"examp",						/* application's name */
	W_PLAIN,						/* type of initial window */
	FALSE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	FALSE,							/* close box on initial window? */
	TRUE,							/* want std. font menu? (includes sizes) */
	TRUE							/* want std. style menu? */
};

/*
	Following structure contains the user's current window options,
	changed only when the OK button in the options dialog is clicked.
*/
static struct s_opt {				/* effective window options */
	BOOLEAN close;					/* want close box? */
	BOOLEAN size;					/* want size box? */
	BOOLEAN hscroll;				/* want horizontal scroll bar? */
	BOOLEAN vscroll;				/* want vertical scroll bar? */
} win_opt = {
	FALSE,
	FALSE,
	FALSE,
	FALSE
};
static int opt_dismiss;				/* button that dismissed options dialog */

/*
	Callback function for options dialog box.  While the box is up, the
	temporary options structure tmp_opt reflects the current status of the
	check boxes.  With every call, including the first one, all of the
	checkboxes are set to conform to that structure, whether they need
	setting or not.  Initially, tmp_opt is set from the global structure,
	win_opt.  The options are made permanent by storing them back into
	win_opt only when and if the OK button is clicked.  The global variable
	opt_dismiss provides a way to communicate the user's final action to
	the function that put up the box, opt_dialog.

	A TRUE return from this function means that XVT leaves the box up; a
	FALSE return makes it go away.
*/
#ifdef FPROTO
static BOOLEAN XVTENTRY cb_opt BTCENTRY(int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN XVTENTRY cb_opt BTCENTRY(control_id, cip)
int control_id;						/* ID of control that got activated */
CONTROL_INFO *cip;
#endif
{
	static struct s_opt tmp_opt;

	NOREF(cip);
	opt_dismiss = control_id;
	switch (control_id) {
	case DLG_INIT:
		tmp_opt = win_opt;
		break;
	case DLG_OK:
		win_opt = tmp_opt;
		return(FALSE);
	case DLG_CANCEL:
		return(FALSE);
	case OPT_CLOSE:
		tmp_opt.close = !tmp_opt.close;
		break;
	case OPT_SIZE:
		tmp_opt.size = !tmp_opt.size;
		break;
	case OPT_HSCROLL:
		tmp_opt.hscroll = !tmp_opt.hscroll;
		break;
	case OPT_VSCROLL:
		tmp_opt.vscroll = !tmp_opt.vscroll;
	}
	check_box(OPT_CLOSE, tmp_opt.close);
	check_box(OPT_SIZE, tmp_opt.size);
	check_box(OPT_HSCROLL, tmp_opt.hscroll);
	check_box(OPT_VSCROLL, tmp_opt.vscroll);
	return(TRUE);
}

/*
	Function to put up the options dialog box.  A TRUE return means that
	the user clicked the OK button, and processing should proceed.  A FALSE
	return means that the user bailed out with the Cancel button.
*/
#ifdef FPROTO
static BOOLEAN opt_dialog(void)
#else
static BOOLEAN opt_dialog()
#endif
{
	if (!new_dialog(DLG_MODAL, OPT_DLG, cb_opt, 0L)) {
		error("Can't put up dialog.");
		return(FALSE);
	}
	return(opt_dismiss == DLG_OK);
}

/*
	Function to keep track of the number of open windows, so that the Close
	item on the File menu is enabled only when there is a window to close.
	Note that we set the menu item all the time -- this is easier than
	remembering its current state.
*/
#ifdef FPROTO
static void change_num_wins(BOOLEAN add_one)
#else
static void change_num_wins(add_one)
BOOLEAN add_one;					/* add a window, or subtract one? */
#endif
{
	static int num_wins = 0;

	num_wins += (add_one ? 1 : -1);
	menu_enable(M_FILE_CLOSE, num_wins > 0);
}

/* Function to build a new window according to the user's specifications. */
#ifdef FPROTO
static void build_window(WIN_TYPE wtype)
#else
static void build_window(wtype)
WIN_TYPE wtype;						/* type of window to build */
#endif
{
	WINDOW win;
	RCT rct;

	switch (wtype) {
	case W_DOC:
		set_rect(&rct, 100, 100, 400, 175);
		break;
	case W_PLAIN:
		set_rect(&rct, 10, 100, 100, 200);
		break;
	case W_DBL:
		set_rect(&rct, 180, 100, 260, 200);
		break;
	default:
		fatal("Invalid window type [build_window].");
	}
	if ((win = new_window(&rct, "A Window", wtype, win_opt.size,
	  win_opt.vscroll, win_opt.hscroll, win_opt.close)) == NULL_WIN)
		error("Can't create window.");
	else
		change_num_wins(TRUE);
	caret_on(win, 10, 20);
}

/* Function to close a window. */
#ifdef FPROTO
static void do_close(WINDOW win)
#else
static void do_close(win)
WINDOW win;
#endif
{
	close_window(win);
	change_num_wins(FALSE);
}

/* Fake "save document" function.  Fakes success/failure return, too. */
#ifdef FPROTO
static BOOLEAN save_document(void)
#else
static BOOLEAN save_document()
#endif
{
	switch (ask("Succeed", "Fail", NULL,
	  "Simulating saving of document.")) {
	case RESP_DEFAULT:
		return(TRUE);
	case RESP_2:
		error("A (pretend) write error occurred.");
		return(FALSE);
	}
}

/* Fake "discard window" function. */
#ifdef FPROTO
static void discard_window(void)
#else
static void discard_window()
#endif
{
	note("Pretending to discard window.");
}

/*
	Function to ask the user if quitting is OK.  Normally, it will put up a
	Save/Discard/Cancel dialog box for each unsaved document, but, as we have
	no documents in this example program, it just asks about one "document."
*/
#ifdef FPROTO
static BOOLEAN quit_approved(void)
#else
static BOOLEAN quit_approved()
#endif
{
	switch (ask("Save", "Discard", "Cancel", "Save changes before closing?")) {
	case RESP_DEFAULT:
		if (!save_document())	/* try to save document */
			return(FALSE);
  		/* fall through */
  	case RESP_2:
  		discard_window();		/* discard window */
  		return(TRUE);
  	case RESP_3:
  		return(FALSE);			/* abort quitting procedure */
  	}
}

/* Function to handle all menu commands. */
#ifdef FPROTO
static void do_menu(MENU_TAG cmd, BOOLEAN shift, BOOLEAN control)
#else
static void do_menu(cmd, shift, control)
MENU_TAG cmd;						/* menu tag */
BOOLEAN shift;						/* was shift key down? */
BOOLEAN control;					/* was control key down? */
#endif
{
	WINDOW win;

	NOREF(shift);
	NOREF(control);
	switch (cmd) {
	case M_FILE_CLOSE:
		if ((win = get_front_window()) != NULL_WIN)
			do_close(win);
		break;
	case M_FILE_QUIT:
		if (quit_approved())
			terminate();
		break;
	case M_WIN_DOC:
		if (opt_dialog())
			build_window(W_DOC);
		break;
	case M_WIN_PLAIN:
		build_window(W_PLAIN);
		break;
	case M_WIN_DBL:
		build_window(W_DBL);
	}
}

/*
	Application initialization.  Set font menu and record the existence of
	XVT's initial window, created from the appl_setup structure (std_win).
*/
BOOLEAN XVTENTRY appl_init BTCENTRY()
{
	set_font_menu(&normal_font);
	change_num_wins(TRUE);
	caret_on(std_win, 10, 20);
	return(TRUE);
}

/* Update function -- called for E_UPDATE events. */
#ifdef FPROTO
static void do_update(WINDOW win)
#else
static void do_update(win)
WINDOW win;
#endif
{
	RCT rct;

	set_cur_window(win);
	get_client_rect(win, &rct);
	set_cpen(&hollow_cpen);
	set_cbrush(&white_cbrush);
	draw_rect(&rct);
	draw_text(4, 15, win == get_front_window() ? "Front" : "Not front", -1);
	draw_text(4, 30, "Hello World!", -1);
}

/*
	Main application entry point.  All events are listed, although not
	all are acted on by this application.
*/
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;							/* window */
EVENT *ep;							/* event */
{
	DRAW_CTOOLS tools;
	static int act_counter = 0;

	switch (ep->type) {
	case E_MOUSE_DOWN:
	case E_MOUSE_UP:
	case E_MOUSE_MOVE:
	case E_MOUSE_DBL:
	case E_CHAR:
		break;
	case E_UPDATE:
		do_update(win);
		break;
	case E_ACTIVATE:
		if (ep->v.active) {
			get_draw_ctools(&tools);
			set_font_menu(&tools.font);
		}
		invalidate_rect(win, NULL);
		break;
	case E_KILL_WINDOW:
	case E_VSCROLL:
	case E_HSCROLL:
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag, ep->v.cmd.shift, ep->v.cmd.control);
		break;
	case E_CLOSE:
		do_close(win);
		break;
	case E_SIZE:
		break;
	case E_FONT:
		if ((win = get_front_window()) != NULL_WIN) {
			set_cur_window(win);
			set_font_menu(&ep->v.font.font);
			set_font(&ep->v.font.font, FALSE);
			invalidate_rect(win, NULL);
		}
		break;
	case E_QUIT:
		if (ep->v.query) {
			if (quit_approved())
				quit_OK();
		}
		else
			terminate();
		break;
	default:
		fatal("Invalid event.");
	}
}

/* Application cleanup.  Nothing to do. */
void XVTENTRY appl_cleanup BTCENTRY()
{
}
