/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_utility.c                                                            */
/*                                                                           */
/*  Utility functions for CTOS Sequential Access devices.                    */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  07/07/92  122J.04  D. Gilson     /  Added new msg for invalid dev spec.  */
/*                                      NLS_DEVICE_INVALID_2 PLE 15591773    */
/*  04/29/91  121J.03  P. Johansson  /  Parse device configuration file to   */
/*                                      get parameters for SeqAccessOpen.    */
/*  04/18/91  121J.02  P. Johansson  /  New buffer control parameters in the */
/*                                      device configuration file.           */
/*  02/11/91  121G.01  P. Johansson  /  New case for 'Tape Rewind' command;  */
/*                                      In case device doesn't position to   */
/*                                      BOT after erase to EOM, rewind.      */
/*  12/13/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <ctype.h>
#include <intel80X86.h>
#include <kbd.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

extern int sprintf(char *s, const char *format, ...);

/* Suppress C run-time (only CTOS functionality needed) */

pragma Off(List);
#include <stub.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define Beep
#define BuildFileSpec
#define CheckErc
#define CloseFile
#define CreateFile
#define CSubparams
#define ErrorExit
#define Exit
#define ExpandLocalMsg
#define FieldEdit
#define LookUpNumber
#define LookUpReset
#define LookUpString
#define NlsGetYesNoStrings
#define NlsYesOrNo
#define OpenFile
#define ParseFileSpec
#define PutFrameChars
#define QueryVidBs
#define RgParam
#define RgParamSetSimple
#define SetField
#define SetFieldNumber
#define ULCmpB

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define sdType

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
#include <seq_access.h>
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern void initialize_msgs(void);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern Boolean proceed(void);
extern sdType *standard_msg(unsigned msg_num);
extern vid_only_msg(unsigned nls_msg_index, unsigned signature,
                    sdType nls_parms[], unsigned nls_parms_len);

/* Error return codes used by this module */

#define FsErc
#define NLSErc
#define RqErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern char sbVerRun[];
extern char bsVid[];

/* Function prototypes defined before the functions themselves are declared */

void configure(void);
void copy(void);
void erase(void);
Boolean erase_OK(char *device_spec, unsigned device_spec_len,
                 char *password, unsigned password_len);
void read_configuration(char *device_spec, unsigned device_spec_len,
                        seq_parameters_type *seq_parameters);
void retension(Boolean rewind_only);

pragma Page(1);
/*-----------------------------------------------------------------------------
 This executable code is at present used by four CTOS commands:

	Configure Sequential Access Device
	    Configuration file name

	Tape Copy
	    File from
	    File to
	    [Overwrite OK?]

	Tape Erase
	    [Device name(s) ([QIC])]
	    [Suppress user interaction?]

	Tape Retension
	    [Device name(s) ([QIC])]

	Tape Rewind
	    [Device name(s) ([QIC])]

 The commands are distinguished by the cases 'CF', 'CP, 'ER', 'RT' and 'RW',
 respectively.  Execute the appropriate case and then exit. */

void main(void) {

   sdType nls_parms[] = {         NULL, 0,
                         (void *) &sbVerRun[1], sbVerRun[0]};
   sdType sd_case;

   initialize_msgs();		/* Consult native-language message file */
   RgParam(0, 0, &nls_parms[0]);
   log_msg(NLS_COMMAND_VERSION, 1, nls_parms, sizeof(nls_parms));
   CheckErc(RgParam(0, 1, &sd_case));
   if (sd_case.cb != 2)
      log_msg(NLS_CASE_INVALID, 0, NULL, 0);
   else if (ULCmpB(sd_case.pb, "CF", 2) == 0xFFFF)
      configure();
   else if (ULCmpB(sd_case.pb, "CP", 2) == 0xFFFF)
      copy();
   else if (ULCmpB(sd_case.pb, "ER", 2) == 0xFFFF)
      erase();
   else if (ULCmpB(sd_case.pb, "RT", 2) == 0xFFFF)
      retension(FALSE);
   else if (ULCmpB(sd_case.pb, "RW", 2) == 0xFFFF)
      retension(TRUE);
   else
      log_msg(NLS_CASE_INVALID, 0, NULL, 0);
   Exit();

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Sequential access device configuration uses a secondary menu after the name
 have of the configuration file has been obtained from the executive command
 form:

	Sequential Access Device Parameters
	    [Minimum (or fixed) record size (512)]
	    [Maximum record size (for variable records)]
	    [Block size (fixed records, defaults to record size)]
	    [Recording density code]
	    [Transport speed code]
	    [Unbuffered mode?]
	    [Service buffer size, in bytes]
	    [Total service buffers]
	    [Erase to EOM after close?]
	    [Rewind after close?]

 If these parameters pass simple sanity checks, they are written to the
 configuration file specified. */

private void configure(void) {

#define N_FIELDS 10

   char buffer[PAGE_SIZE], exit_char, string[N_FIELDS][UCHAR_MAX];
   unsigned block_size = PAGE_SIZE, column = 0, config_fh, density = 0,
      erc = ercOK, i, j, line[N_FIELDS], min_record_size = PAGE_SIZE,
      max_record_size = PAGE_SIZE, service_buffer_size = 0, speed = 0,
      total_service_buffers = 0, width;
   Boolean unbuffered = FALSE, erase_on_close = FALSE, rewind_on_close = FALSE;
   void *field_destination[] = {&min_record_size, &max_record_size,
                                &block_size, &speed, &density, &unbuffered,
                                &service_buffer_size, &total_service_buffers,
                                &erase_on_close, &rewind_on_close};
   unsigned field_msg_num[] = {NLS_MIN_RECORD_SIZE_FIELD,
                               NLS_MAX_RECORD_SIZE_FIELD,
                               NLS_BLOCK_SIZE_FIELD, NLS_DENSITY_FIELD,
                               NLS_SPEED_FIELD, NLS_UNBUFFERED_FIELD,
                               NLS_SERVICE_BUFFER_SIZE_FIELD,
                               NLS_TOTAL_SERVICE_BUFFERS_FIELD,
                               NLS_ERASE_ON_CLOSE_FIELD,
                               NLS_REWIND_ON_CLOSE_FIELD};
   enum {NUMERIC, STRING} field_type[] = {NUMERIC, NUMERIC, NUMERIC, NUMERIC,
                                          NUMERIC, STRING, NUMERIC, NUMERIC,
                                          STRING, STRING};
   unsigned param_msg_num[] = {NLS_MIN_RECORD_SIZE, NLS_MAX_RECORD_SIZE,
                               NLS_BLOCK_SIZE, NLS_DENSITY, NLS_SPEED,
                               NLS_UNBUFFERED, NLS_SERVICE_BUFFER_SIZE,
                               NLS_TOTAL_SERVICE_BUFFERS, NLS_ERASE_ON_CLOSE,
                               NLS_REWIND_ON_CLOSE};
   sdType sd_config, *sd_field, sd_nls_yes_no[2], sd_string[N_FIELDS];
   struct {
      char frame;
      char lines;
      char columns;
      char current_line;
      char current_column;
      Boolean cursor_visible;
      Boolean pause_enabled;
      char attribute_mode;
      Boolean literal_mode;
      char reserved[7];
   } vid_state;

   NlsGetYesNoStrings(NULL, sd_nls_yes_no);
   for (i = 0; i < N_FIELDS; i++)
      sd_string[i].pb = string[i];
   CheckErc(RgParam(CONFIG_FILE_PARAM, 0, &sd_config));
   do {
      if (erc == ercNoSuchFile)
         erc = CreateFile(sd_config.pb, sd_config.cb, NULL, 0, 0);
      if (erc == ercOK)
         erc = OpenFile(&config_fh, sd_config.pb, sd_config.cb, NULL, 0,
                        modeModify);
   } while (erc == ercNoSuchFile);
   if (erc != ercOK)
      exit_with_msg(erc, 0);
   for (i = 0; i < N_FIELDS; i++) {
      LookUpReset();
      if ((sd_field = standard_msg(field_msg_num[i])) != NULL)
         if (field_type[i] == NUMERIC)
            LookUpNumber(config_fh, buffer, sizeof(buffer), sd_field->pb,
                         sd_field->cb, field_destination[i]);
         else {
            LookUpString(config_fh, buffer, sizeof(buffer), sd_field->pb,
                         sd_field->cb, sd_string[i].pb, sizeof(string[i]),
                         &sd_string[i].cb);
            NlsYesOrNo(NULL, &sd_string[i], field_destination[i]);
         }
   }
   for (i = 0; i < N_FIELDS; i++)
      if (field_type[i] == NUMERIC) {
         sprintf(sd_string[i].pb, "%u", *((unsigned *) field_destination[i]));
         sd_string[i].cb = strlen(sd_string[i].pb);
      } else if (*((Boolean *) field_destination[i]))
         memcpy(sd_string[i].pb, sd_nls_yes_no[0].pb,
                sd_string[i].cb = sd_nls_yes_no[0].cb);
      else
         memcpy(sd_string[i].pb, sd_nls_yes_no[1].pb,
                sd_string[i].cb = sd_nls_yes_no[1].cb);
   vid_only_msg(NLS_SEQ_DEVICE_PARAMETERS, 0, NULL, 0);
   for (i = 0; i < N_FIELDS; i++) {
      vid_only_msg(param_msg_num[i], 0, NULL, 0);
      QueryVidBs(bsVid, &vid_state);
      if (vid_state.current_column > column)
         column = vid_state.current_column;
   }
   column += 2;
   width = vid_state.columns - column;
   for (i = 0; i < N_FIELDS; i++)
      line[i] = vid_state.current_line - (N_FIELDS - 1 - i);
   for (i = 0; i < N_FIELDS; i++)
      PutFrameChars(vid_state.frame, column, line[i], sd_string[i].pb,
                    sd_string[i].cb);
   i = 0;				/* Start with the first field... */
   do {
      do {
         if ((erc = FieldEdit(vid_state.frame, column, line[i], 0, width,
                              sd_string[i].pb, sd_string[i].cb, width,
                              &sd_string[i].cb, &exit_char)) == ercOK)
            if (field_type[i] == NUMERIC) {
               for (j = 0; j < sd_string[i].cb; j++)
                  if (!isdigit(sd_string[i].pb[j])) {
                     erc = ercBadCharacterInNumber;
                     break;
                  }
               if (erc == ercOK) {
                  sd_string[i].pb[sd_string[i].cb] = 0;
                  if (atol(sd_string[i].pb) > UINT_MAX)
                     erc = ercBadCharacterInNumber;
                  else
                     *((unsigned *) field_destination[i])
                                                       = atoi(sd_string[i].pb);
               }
            } else
               erc = NlsYesOrNo(NULL, &sd_string[i], field_destination[i]);
         if (erc != ercOK)
            Beep();
      } while (erc != ercOK);
      switch (i) {
         case 0:				/* :MinRecordSize: */
            if (min_record_size == 0)
               exit_char = 0xFF;
            break;

         case 1:				/* :MaxRecordSize: */
            if (max_record_size == 0)
               exit_char = 0xFF;
            break;
      }
      switch (exit_char) {
         case UP_ARROW:
            i = (i + N_FIELDS - 1) % N_FIELDS;
            break;

         case DOWN_ARROW:
         case NEXT:
            i = (i + 1) % N_FIELDS;
            break;

         case CANCEL:
            break;

         case FINISH:
         case GO:
            i = 0xFFFF;			/* Innocent until proven guilty... */
            if (min_record_size > max_record_size)
               i = 1;			/* Error in :MaxRecordSize: */
            else if (      min_record_size == max_record_size
                        && (   block_size == 0
                            || block_size % min_record_size != 0)
                     || min_record_size != max_record_size && block_size != 0)
               i = 2;			/* Error in :BlockSize: */
            else if (   service_buffer_size != 0
                     && (   (   block_size == 0
                             && service_buffer_size < max_record_size)
                         || (   block_size != 0
                             && (   service_buffer_size < block_size
                                 || service_buffer_size % block_size != 0))))
               i = 6;			/* Error in :ServiceBufferSize: */
            else if (   service_buffer_size == 0 && total_service_buffers != 0
                     || service_buffer_size != 0 && total_service_buffers == 0)
               i = 7;			/* Error in :TotalServiceBuffers: */
            if (i != 0xFFFF) {		/* Any inconsistency errors? */
               exit_char = 0xFF;	/* Yes, can't leave the loop... */
               Beep();
            }
            break;

         default:
            Beep();
      }
   } while (exit_char != CANCEL && exit_char != GO && exit_char != FINISH);
   if (exit_char == FINISH || exit_char == GO)
      for (i = 0; i < N_FIELDS; i++) {
         LookUpReset();
         if ((sd_field = standard_msg(field_msg_num[i])) != NULL)
            if (field_type[i] == NUMERIC)
               SetFieldNumber(config_fh, buffer, sizeof(buffer), sd_field->pb,
                              sd_field->cb,
                              *((unsigned *) field_destination[i]));
            else
               SetField(config_fh, buffer, sizeof(buffer), sd_field->pb,
                        sd_field->cb, sd_string[i].pb, sd_string[i].cb);
      }
   CloseFile(config_fh);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void copy(void) {

   return;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For all of the device name(s) specified on the Executive command form,
 attempt to open the device MODE_READ and ascertain whether or not a valid
 archive data set has been written on the medium.  If not (or if the user
 answers affirmnatively to an 'Overwrite?' question), close the device, reopen
 it in MODE_MODIFY and then perform an erase operation.  Except for a failure
 on SeqAccessOpen caused by the absence of the Sequential Access service, this
 code will just soldier on through errors and attempt to erase the medium on
 each device it comes to in its turn.  Attempts to solicit operator
 intervention in the case of difficulties are controlled by the setting of the
 '[Suppress user interaction?]' parameter for the command. */

private void erase(void) {

   char device[MAX_DEVICE_LENGTH + 1],
      device_spec[MAX_FILE_SPECIFICATION_LENGTH + 1],
      node[MAX_NODE_LENGTH + 1], password[MAX_PASSWORD_LENGTH + 1];
   unsigned device_len, device_spec_len, erc, i, n_device, node_len,
      password_len, position_len, seq_handle;
   sdType nls_parms[] = {(void *) device_spec, 0,
                         (void *) &erc, sizeof(erc)};
   Boolean noninteractive = FALSE;
   unsigned long residual;
   sdType sd_param;
   seq_parameters_type seq_parameters;

   if (CSubparams(NONINTERACTIVE_PARAM) != 0) {
      CheckErc(RgParam(NONINTERACTIVE_PARAM, 0, &sd_param));
      if (NlsYesOrNo(NULL, &sd_param, &noninteractive) != ercOK) {
         log_msg(NLS_SYNTAX_ERROR, 0, standard_msg(NLS_NONINTERACTIVE),
                 sizeof(sdType));
         exit_with_msg(ercSyntax, NLS_YES_OR_NO_EXPECTED);
      }
   }
   if ((n_device = CSubparams(DEVICE_PARAM)) == 0) {
      CheckErc(RgParamSetSimple(DEVICE_PARAM,
                                standard_msg(NLS_DEFAULT_DATASET)));
      n_device = 1;
   }
   for (i = 0; i < n_device; i++) {
      CheckErc(RgParam(DEVICE_PARAM, i, &sd_param));
      erc = ParseFileSpec(0, sd_param.pb, sd_param.cb, FALSE, node, &node_len,
                          device, &device_len, NULL, NULL, NULL, &position_len,
                          password, &password_len, FALSE, 3);
      if (erc != ercOK || position_len != 0) {
         nls_parms[0].pb = sd_param.pb;
         nls_parms[0].cb = sd_param.cb;
         log_msg(NLS_DEVICE_INVALID_2, 0, nls_parms, sizeof(nls_parms));/*m04*/
         continue;
      }
      memset(device_spec, 0, sizeof(device_spec));
      BuildFileSpec(0, device_spec, &device_spec_len, NULL,
                    sizeof(device_spec), FALSE, node, node_len, device,
                    device_len, NULL, 0, NULL, 0, FALSE, NULL, 0, FALSE, 0);
      nls_parms[0].cb = device_spec_len;
      if (!noninteractive)
         if (!erase_OK(device_spec, device_spec_len, password, password_len))
            continue;
      read_configuration(device_spec, device_spec_len, &seq_parameters);
      erc = SeqAccessOpen(&seq_handle, device_spec, device_spec_len,
                          password, password_len, MODE_MODIFY,
                          &seq_parameters, sizeof(seq_parameters));
      if (erc == ercServiceNotAvail)
         exit_with_msg(ercOK, NLS_NEED_SEQ_ACCESS_SERVICE);
      if (erc != ercOK) {
         log_msg(NLS_CANT_OPEN_DEVICE, 0, nls_parms, sizeof(nls_parms));
         continue;
      }
      log_msg(NLS_ERASING, 0, nls_parms, sizeof(nls_parms));
      erc = SeqAccessControl(seq_handle, CTRL_REWIND, SYNCHRONIZE, &residual);
      if (erc == ercOK)
         erc = SeqAccessControl(seq_handle, CTRL_ERASE_MEDIUM, SYNCHRONIZE,
                                &residual);
      if (erc == ercOK)
         erc = SeqAccessControl(seq_handle, CTRL_REWIND, SYNCHRONIZE,
                                &residual);
      if (erc == ercOK)
         log_msg(NLS_DONE, 0, NULL, 0);
      else if (erc == ercTapeNotReady)
         log_msg(NLS_DEVICE_NOT_READY, 0, nls_parms, sizeof(nls_parms));
      else
         log_msg(NLS_FILE_ERROR, 0, nls_parms, sizeof(nls_parms));
      SeqAccessClose(seq_handle);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For all of the device name(s) specified on the Executive command form,
 attempt to open the device MODE_READ and then perform a retension operation.
 Except for a failure on SeqAccessOpen caused by the absence of the Sequential
 Access service, this code will just soldier on through errors and attempt to
 retension the medium on each device it comes to in its turn.  No attempts are
 made to solicit operator intervention in the case of difficulties, because of
 the lack of a '[Suppress user interaction?]' parameter for the command (and
 hence obstacles when run under the control of Batch). */

private void retension(Boolean rewind_only) {

   char device[MAX_DEVICE_LENGTH + 1],
      device_spec[MAX_FILE_SPECIFICATION_LENGTH + 1],
      node[MAX_NODE_LENGTH + 1], password[MAX_PASSWORD_LENGTH + 1];
   unsigned device_len, device_spec_len, erc, i, n_device, node_len,
      password_len, position_len, seq_handle;
   sdType nls_parms[] = {(void *) device_spec, 0,
                         (void *) &erc, sizeof(erc)};
   unsigned long residual;
   sdType sd_param;
   seq_parameters_type seq_parameters;

   if ((n_device = CSubparams(DEVICE_PARAM)) == 0) {
      CheckErc(RgParamSetSimple(DEVICE_PARAM,
                                standard_msg(NLS_DEFAULT_DATASET)));
      n_device = 1;
   }
   for (i = 0; i < n_device; i++) {
      CheckErc(RgParam(DEVICE_PARAM, i, &sd_param));
      erc = ParseFileSpec(0, sd_param.pb, sd_param.cb, FALSE, node, &node_len,
                          device, &device_len, NULL, NULL, NULL, &position_len,
                          password, &password_len, FALSE, 3);
      if (erc != ercOK || position_len != 0) {
         nls_parms[0].pb = sd_param.pb;
         nls_parms[0].cb = sd_param.cb;
         log_msg(NLS_DEVICE_INVALID_2, 0, nls_parms, sizeof(nls_parms));/*m04*/
         continue;
      }
      memset(device_spec, 0, sizeof(device_spec));
      BuildFileSpec(0, device_spec, &device_spec_len, NULL,
                    sizeof(device_spec), FALSE, node, node_len, device,
                    device_len, NULL, 0, NULL, 0, FALSE, NULL, 0, FALSE, 0);
      nls_parms[0].cb = device_spec_len;
      read_configuration(device_spec, device_spec_len, &seq_parameters);
      erc = SeqAccessOpen(&seq_handle, device_spec, device_spec_len,
                          password, password_len, MODE_READ, &seq_parameters,
                          sizeof(seq_parameters));
      if (erc == ercServiceNotAvail)
         exit_with_msg(ercOK, NLS_NEED_SEQ_ACCESS_SERVICE);
      if (erc != ercOK) {
         log_msg(NLS_CANT_OPEN_DEVICE, 0, nls_parms, sizeof(nls_parms));
         continue;
      }
      log_msg((rewind_only) ? NLS_REWINDING : NLS_RETENSIONING, 0, nls_parms,
              sizeof(nls_parms));
      erc = SeqAccessControl(seq_handle, CTRL_REWIND, SYNCHRONIZE, &residual);
      if (erc == ercOK && !rewind_only) {
         erc = SeqAccessControl(seq_handle, CTRL_RETENSION, SYNCHRONIZE,
                                &residual);
         if (erc == ercOK)
            erc = SeqAccessControl(seq_handle, CTRL_REWIND, SYNCHRONIZE,
                                   &residual);
      }
      if (erc == ercOK)
         log_msg(NLS_DONE, 0, NULL, 0);
      else if (erc == ercTapeNotReady)
         log_msg(NLS_DEVICE_NOT_READY, 0, nls_parms, sizeof(nls_parms));
      else
         log_msg(NLS_FILE_ERROR, 0, nls_parms, sizeof(nls_parms));
      SeqAccessClose(seq_handle);
      if (erc != ercOK)
         ErrorExit(erc);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure is used by the erase function when operating in an interactive
 mode.  It performs n functions: a) the rewind operation is used to verify
 device ready condition (if the device is not ready, the user is prompted to
 correct the condition and retry, skip this device or exit the utility),
 b) the read of the first block of data on the tape is used to ascertain
 whether or not a valid archive data set (either old 12.0 or new 12.1 format)
 exists on the tape (if such an archive exists, the user is again prompted for
 confirmation of the erase).  The procedure returns FALSE or TRUE according to
 whether of not this device should be skipped or have its medium erased. */

private Boolean erase_OK(char *device_spec, unsigned device_spec_len,
                         char *password, unsigned password_len) {

   archive_label_type archive_label;
   unsigned checksum = ARCHIVE_SIGNATURE, erc, seq_handle;
   signed i;
   sdType nls_parms[] = {(void *) device_spec, device_spec_len,
                                  NULL, 0,
                                  NULL, 0};
   unsigned long residual;
   Boolean retry = TRUE;
   sdType *sd_volume_hdr;
   seq_parameters_type seq_parameters;
   vam_volume_hdr_type *vam_volume_hdr;

   read_configuration(device_spec, device_spec_len, &seq_parameters);
   erc = SeqAccessOpen(&seq_handle, device_spec, device_spec_len,
                       password, password_len, MODE_READ, &seq_parameters,
                       sizeof(seq_parameters));
   if (erc != ercOK)
      return(TRUE);	/* Force exposure of the erc in calling procedure */
   do {
      erc = SeqAccessControl(seq_handle, CTRL_REWIND, SYNCHRONIZE, &residual);
      if (erc == ercOK)
         erc = SeqAccessRead(seq_handle, &archive_label,
                             sizeof(archive_label), &residual);
      if (erc == ercTapeNotReady) {
         log_msg(NLS_DEVICE_NOT_READY, 0, nls_parms, sizeof(nls_parms));
         log_msg(NLS_REQUEST_INTERVENTION, 0, NULL, 0);
         retry = proceed();
      }
   } while (erc == ercTapeNotReady && retry);
   SeqAccessClose(seq_handle);	/* Don't leave stale handles lying around */
   if (erc != ercOK)
      return(retry);		/* Can't determine if archive is present */
   sd_volume_hdr = standard_msg(NLS_ARCHIVE_BOV);
   for (i = 0; i < PAGE_SIZE / 2; i++)
      checksum += ((unsigned *) &archive_label)[i];
   if (     checksum == 0
         && ULCmpB(archive_label.description, sd_volume_hdr->pb,
                   sizeof(archive_label.description)) == 0xFFFF) {
      nls_parms[1].pb = (void *) &archive_label.creation_time;
      nls_parms[1].cb = sizeof(archive_label.creation_time);
      log_msg(NLS_CONFIRM_ERASE, 0, nls_parms, sizeof(nls_parms));
      return(proceed());	/* Ask user if we should proceed */
   }
   vam_volume_hdr = (vam_volume_hdr_type *) &archive_label;
   sd_volume_hdr = standard_msg(NLS_OLD_BACKUP_BOV);
   if (ULCmpB(vam_volume_hdr->backup_hdr, sd_volume_hdr->pb,
              sd_volume_hdr->cb) == 0xFFFF) {
      for (i = last(vam_volume_hdr->backup_time); i >= 0; i--)
         if (vam_volume_hdr->backup_time[i] != ' ')
            break;
      nls_parms[1].pb = vam_volume_hdr->backup_time;
      nls_parms[1].cb = i + 1;
      if (vam_volume_hdr->backup_id[last(vam_volume_hdr->backup_id)] == 0)
         i = strlen(vam_volume_hdr->backup_id);
      else {
         for (i = last(vam_volume_hdr->backup_id); i >= 0; i--)
            if (vam_volume_hdr->backup_id[i] != ' ')
               break;
         i++;
      }
      if (vam_volume_hdr->backup_id[0] == '\n')
         memcpy(vam_volume_hdr->backup_id, &vam_volume_hdr->backup_id[1], --i);
      nls_parms[2].pb = vam_volume_hdr->backup_id;
      nls_parms[2].cb = i + 1;
      log_msg(NLS_CONFIRM_ERASE_OLD_FORMAT, 0, nls_parms, sizeof(nls_parms));
      return(proceed());	/* Ask user if we should proceed */
   }
   return(TRUE);		/* Didn't find a header... */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Attempt to open a device configuration file for the Sequential Access
 device.  Search for [Sys]<Sys>XxxxConfig.Sys, where XXX is the name of the
 sequential access device to be used for the archive operations.  If the file
 can be found, update the mode parameters for the subsequent SeqAccessOpen. */

private void read_configuration(char *device_spec, unsigned device_spec_len,
                                seq_parameters_type *seq_parameters) {

   char buffer[PAGE_SIZE], config_spec[MAX_FILE_SPECIFICATION_LENGTH],
      device[MAX_DEVICE_LENGTH], directory[MAX_DIRECTORY_LENGTH],
      filename[MAX_FILENAME_LENGTH], node[MAX_NODE_LENGTH],
      volume[MAX_VOLUME_LENGTH];
   unsigned config_fh = 0, config_spec_len, directory_len, filename_len, i,
      node_len, volume_len;
   Boolean device_at_server;
   unsigned *field_destination[] = {(void *) &seq_parameters->min_record_size,
                                    (void *) &seq_parameters->max_record_size,
                                    (void *) &seq_parameters->block_size,
                                    (void *) &seq_parameters->density,
                                    (void *) &seq_parameters->speed,
                                    (void *) &seq_parameters->unbuffered,
                                    (void *)
                                          &seq_parameters->service_buffer_size,
                                    (void *) &seq_parameters->service_buffers};
   unsigned field_msg_num[] = {NLS_MIN_RECORD_SIZE_FIELD,
                               NLS_MAX_RECORD_SIZE_FIELD,
                               NLS_BLOCK_SIZE_FIELD, NLS_DENSITY_FIELD,
                               NLS_SPEED_FIELD, NLS_UNBUFFERED_FIELD,
                               NLS_SERVICE_BUFFER_SIZE_FIELD,
                               NLS_TOTAL_SERVICE_BUFFERS_FIELD};
   enum {NUMERIC, STRING} field_type[] = {NUMERIC, NUMERIC, NUMERIC, NUMERIC,
                                          NUMERIC, STRING, NUMERIC, NUMERIC};
   char string[255];
   sdType *sd_config, sd_device = {device}, *sd_field, sd_string = {string};

   ParseFileSpec(0, device_spec, device_spec_len, FALSE, node,
                 &node_len, device, &sd_device.cb, NULL, NULL, NULL,
                 NULL, NULL, NULL,
                 FALSE, 3);
   device_at_server = (device[0] == '!');
   if (device_at_server)
      memcpy(device, &device[1], --sd_device.cb);
   sd_config = standard_msg(NLS_SEQ_CONFIG_FILE);
   ExpandLocalMsg(&sd_device, sizeof(sd_device), sd_config->pb,
                  sd_config->cb, filename, sizeof(filename), &filename_len,
                  FALSE);
   ParseFileSpec(0, filename, filename_len, FALSE, NULL, NULL, volume,
                 &volume_len, directory, &directory_len, filename,
                 &filename_len, NULL, NULL, FALSE, 0);
   if (device_at_server && volume_len < MAX_VOLUME_LENGTH) {
      _rmemcpy(&volume[1], volume, volume_len++);
      volume[0] = '!';
   }
   memset(config_spec, 0, sizeof(config_spec));
   BuildFileSpec(0, config_spec, &config_spec_len, NULL, sizeof(config_spec),
                 FALSE, node, node_len, volume, volume_len, directory,
                 directory_len, filename, filename_len, FALSE, NULL, 0,
                 FALSE, 0);
   if (     OpenFile(&config_fh, config_spec, config_spec_len, NULL, 0,
                     modeRead) != ercOK
         && volume[0] != '!' && volume_len < MAX_VOLUME_LENGTH) {
      _rmemcpy(&volume[1], volume, volume_len++);
      volume[0] = '!';
      memset(config_spec, 0, sizeof(config_spec));
      BuildFileSpec(0, config_spec, &config_spec_len, NULL,
                    sizeof(config_spec), FALSE, node, node_len, volume,
                    volume_len, directory, directory_len, filename,
                    filename_len, FALSE, NULL, 0, FALSE, 0);
      OpenFile(&config_fh, config_spec, config_spec_len, NULL, 0, modeRead);
   }
   memset(seq_parameters, 0, sizeof(*seq_parameters));
   for (i = 0; i <= last(field_msg_num); i++) {
      LookUpReset();
      if ((sd_field = standard_msg(field_msg_num[i])) != NULL)
         if (field_type[i] == NUMERIC)
            LookUpNumber(config_fh, buffer, sizeof(buffer), sd_field->pb,
                         sd_field->cb, field_destination[i]);
         else {
            LookUpString(config_fh, buffer, sizeof(buffer), sd_field->pb,
                         sd_field->cb, sd_string.pb, sizeof(string),
                         &sd_string.cb);
            NlsYesOrNo(NULL, &sd_string, field_destination[i]);
         }
   }
   CloseFile(config_fh);

}
