/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  archive_io.c                                                             */
/*                                                                           */
/*  Functions to perform archive medium I/O for the utilities.               */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/19/92  122H.27  T. Tran       /  Confusing pointer casting. Change it.*/
/*  05/04/92  122G.26  D. Gilson     /  Modify calculate_piecemeal_size to   */
/*                                      handle new BNet ercs.                */
/*  05/01/92  122G.25  D. Gilson     /  Fix erc 80 when trying to display a  */
/*                                      tape header that is all screwed up.  */
/*  04/20/92  122G.24  D. Gilson     /  Fix for PLE 15506083 When Sequence # */
/*                                      rolls from xFFFF to 0.               */
/*  03/11/92  122F.23  D. Gilson     /  Minor fix for PLE 15425539           */
/*                                   /  Get erc 9074 when doing multiple     */
/*                                   /  disk volume that span multiple tapes.*/
/*  02/15/92  122F.22  D. Gilson     /  Minor fix for PLE 15419288           */
/*                                   /  Got erc 9035 on mount rq for 2nd tape*/
/*  12/19/91  130E.21  D. Gilson     /  Added multi drive support for backups*/
/*  12/17/91  130E.20  D. Gilson     /  PLE 15373342-problems with 7.X tapes */
/*  12/12/91  130E.19  D. Gilson     /  Fix a problem with m18               */
/*  11/14/91  130D.18  D. Gilson     /  PLE 15312637 Does not report error   */
/*                                   /  if archive disk file is invalid.     */
/*                                   /  Just asks for next archive file.     */
/*  08/19/91  130B.17  P. Johansson  /  Dynamically allocate RSAM buffer for */
/*                                      old format disk datasets (up to half */
/*                                      of available memory).                */
/*  08/05/91  130A.16  P. Johansson  /  Check for block size too big for     */
/*                                      service buffer size if dataset label */
/*                                      dictates a SeqAccessModeSet change.  */
/*  07/19/91  130A.15  P. Johansson  /  Ignore EOM warning on erase medium;  */
/*                                      check that dataset creation time in  */
/*                                      sentinel records matches for dataset.*/
/*  05/24/91  121J.14  C. Naik       /  Bug fix to handle I/O error on disk  */
/*                                      reads correctly.                     */
/*  05/22/91  121J.13  P. Johansson  /  Don't allocate tape EOM recovery     */
/*                                      buffer in unbuffered mode.           */
/*  05/20/91  121J.12  P. Johansson  /  Archive utilties can operate in      */
/*                                      unbuffered mode only if records are  */
/*                                      not blocked.                         */
/*  05/10/91  121J.11  P. Johansson  /  Sometimes 12.0 software does not     */
/*                                      write a filemark at the end of a     */
/*                                      tape in a multivolume set, so change */
/*                                      ercTapeIoError to ercTapeFilemark    */
/*                                      when at EOT; in position_archive(),  */
/*                                      translate SeqAccessStatus error code */
/*                                      to ercOK unless it was busy, offline */
/*                                      or reset.                            */
/*  04/23/91  121J.10  P. Johansson  /  New buffer control parameters in the */
/*                                      device configuration file; fix RSAM  */
/*                                      reads of 12.0 style diskettes so we  */
/*                                      can start after the first diskette.  */
/*  04/16/91  121J.09  P. Johansson  /  Configuration file self-consistency. */
/*  04/11/91  121J.08  P. Johansson  /  Correct piecemealing of 12.0 style   */
/*                                      tapes; save room for up to 64 Kb     */
/*                                      record buffer for 12.0 Flemington;   */
/*                                      enhance messages for positioning;    */
/*                                      check for EOM after positioning to   */
/*                                      [XXX]+ on half-inch tape.
/*  04/03/91  121H.07  P. Johansson  /  Support multiple volume backup to    */
/*                                      [XXX]+ datasets on tape.             */
/*  03/19/91  121H.06  P. Johansson  /  Multivolume 12.0 Flemington tapes.   */
/*  03/07/91  121G.05  P. Johansson  /  Add support for half-inch tape.      */
/*  02/01/91  121F.04  P. Johansson  /  Piecemeal on buffer data recovery.   */
/*  01/18/91  121F.03  P. Johansson  /  Add capability to read old format    */
/*                                      RSAM datasets from disk or floppy.   */
/*  01/15/91  121F.02  P. Johansson  /  For old format archive datasets,     */
/*                                      "blank tape" at EOT same as filemark.*/
/*  01/07/91  121F.01  P. Johansson  /  Detect write-protected tapes during  */
/*                                      open sequence, not during writes;    */
/*                                      allow new tape to be mounted if user */
/*                                      uses CANCEL to deny overwrite.       */
/*  12/13/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocExch
#define AllocMemorySL
#define BuildFileSpec
#define CheckErc
#define ChangeFileLength
#define CloseFile
#define CloseRsFile
#define CreateFile
#define CreateProcess
#define DeallocMemorySL
#define DeleteFile
#define ErrorExit
#define ExpandAreaSL
#define ExpandLocalMsg
#define GetFileStatus
#define GetDateTime
#define LookUpReset
#define LookUpNumber
#define LookUpString
#define NlsULCmpB
#define NlsYesOrNo
#define OpenFile
#define OpenRsFile
#define ParseFileSpec
#define QueryBigMemAvail
#define QueryMemAvail
#define Read
#define ReadRsRecord
#define RgParam
#define Send
#define SetRsLfa
#define ULCmpB
#define Wait
#define Write

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define ProcDescType
#define sdType

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
#include <seq_access.h>
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern void lock_video(unsigned signature);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern Boolean proceed(void);
extern void quicksort(char **key, unsigned n);
extern sdType *standard_msg(unsigned msg_num);
extern void unlock_video(unsigned signature);
extern vid_only_msg(unsigned nls_msg_index, unsigned signature,
                    sdType nls_parms[], unsigned nls_parms_len);

/* Error return codes used by this module */

#define ClErcOrXblkErc
#define FsErc
#define RqErc
#define RxErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern filesys_type filesys;
extern summary_type summary;
extern target_type target;
extern vlpb_type vlpb;

/* Global variables exported by this manuscript */

archive_type archive;

/* Static variables global within this manuscript */

flag is_first_archive_volume = TRUE;							/* m19 */
private archive_sentinel_type archive_sentinel;
private unsigned data_buffer_exch, iob_exch, page_buffer_exch;
private unsigned data_buffer_size = SEGMENT_SIZE - PAGE_SIZE;
static char (*mfd)[PAGE_SIZE] = NULL;
static mfd_entry_type **mfd_entry = NULL;
static unsigned n_mfd_entry = 0;
private void *recovery_buffer;
private unsigned long recovery_buffer_size = 0;
private seq_parameters_type seq_parameters;
private struct {
   Boolean skip_record;
   Boolean fabricate_record;
   unsigned deferred_erc;
   void *block_buffer;
   unsigned block_size;
   void *block_data;
   unsigned block_residual;
   unsigned block_sequence;
   unsigned file_num;
   unsigned record_sequence;
   unsigned record_residual;
} vam;

/* Function prototypes defined before the functions themselves are declared */

iob_type *allocate_iob(unsigned buffer_size);
void archive_io_process(void);
unsigned calculate_piecemeal_size(void);
unsigned checksum(void *data);
Boolean checksum_valid(void *data);
void close_archive_dataset(void);
void close_archive_seq_dataset(Boolean change_medium, Boolean final_close);
void deallocate_iob(iob_type *iob);
void fetch_mfd(unsigned mfd_pages);
mfd_entry_type* find_mfd_entry(char directory[]);
void interpret_archive_command(iob_type *iob);
void open_archive_input_dataset(void);
void open_archive_output_dataset(void);
void open_archive_seq_dataset(void);
void open_sequential_device(Boolean inbound);
unsigned position_archive(void);
void read_archive_configuration(void);
void read_archive_data(void *buffer, unsigned buffer_len, flag is_sentinal);
void read_archive_fhb(void);
unsigned read_archive_label(void);
void read_archive_sentinel(Boolean file_header_sentinel);
unsigned read_archive_rsam_record(void *record, unsigned record_len,
                                  unsigned long *record_residual);
unsigned read_archive_vam_record(void *record, unsigned record_len,
                                 unsigned long *residual);
void release_mfd(unsigned *mfd_pages);
unsigned valid_archive_bov(archive_label_type *archive_label);
unsigned valid_vam_volume_hdr(vam_volume_hdr_type *vam_volume_hdr);
void write_archive_data(void *buffer, unsigned buffer_len);
unsigned write_archive_label(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 The archive I/O process is a co-process to either a backup or restore main
 process.  In the first case, the main process sends Iob's describing blocks
 of data to be written to the archive medium (either disk or tape).  In the
 second case, the archive I/O process initiates buffered read-ahead from the
 archive medium and sends Iob's describing blocks of data to be restored to
 the main process.  Create the process here at a higher priority level than
 the main process (an aid to keep sequential archive media in constant motion)
 and allocate as many buffers for the Iob's as we can get our hands on!  Don't
 forget to reserve memory space for other allocations made by the main process
 (allocation bit map and EOM recovery buffer for a Volume Archive, EOM
 recovery buffer for a Selective Archive, old format tape buffer for a Restore
 Archive and room for the MFD for all). */

void create_archive_io_process(void) {

#define MAX_IOBS 8
#define STACK_SIZE 2048

   unsigned long available_memory = 0,
      desired_memory = filesys.memory_for_structures, min_buffer_memory;
   void *data_buffer;
   unsigned i, max_iobs, vam_buffer_size = 0;
   iob_type *iob = (void *) &iob;
   void *page_buffer = &page_buffer;
   ProcDescType process_descriptor;
   void *stack = &stack;

   memset(&archive_sentinel, 0, sizeof(archive_sentinel));
   CheckErc(AllocExch(&archive.msg_exch));
   CheckErc(AllocExch(&iob_exch));
   CheckErc(AllocExch(&data_buffer_exch));
   CheckErc(AllocExch(&page_buffer_exch));
   CheckErc(ExpandAreaSL(STACK_SIZE, selector_of(stack), &offset_of(stack)));
   memset(stack, 0, STACK_SIZE);
   if (archive.sequential)
      open_sequential_device(!vlpb.write_archive);
   else if (!vlpb.write_archive) {		/* Provide for RSAM */
      archive.rswa = (void *) &archive.rswa;
      CheckErc(ExpandAreaSL(sizeof(*archive.rswa), selector_of(archive.rswa),
                            &offset_of(archive.rswa)));
   }
   if (QueryBigMemAvail(&available_memory) != ercOK)
      if (QueryMemAvail(&available_memory) == ercOK)
         available_memory *= PARAGRAPH_SIZE;
   if (!archive.sequential && !vlpb.write_archive) {	/* Need RSAM buffer? */
      archive.rsam_buffer_size = _min(SEGMENT_SIZE - PAGE_SIZE,
                                      available_memory / 2);
      available_memory -= archive.rsam_buffer_size;
   }
   min_buffer_memory = 2 * (sizeof(iob_type) + PAGE_SIZE
                            + seq_parameters.min_record_size);
   if (archive.sequential)		/* Guesstimate tape buffers... */
      if (vlpb.write_archive) {
         recovery_buffer_size = seq_parameters.service_buffer_pool_size;
         if (!seq_parameters.data_buffer_nonrecoverable)
            recovery_buffer_size += seq_parameters.device_buffer_size;
         recovery_buffer_size = _min(recovery_buffer_size, SEGMENT_SIZE);
      } else
         vam_buffer_size = 0xFE00;	/* In case of 12.0 style tape */
   if (     min_buffer_memory + recovery_buffer_size + vam_buffer_size
         >= available_memory)
      ErrorExit(ercMemoryNotAvail);
   available_memory -= recovery_buffer_size + vam_buffer_size;
   if (desired_memory + min_buffer_memory < available_memory)
      available_memory -= desired_memory;
   max_iobs = available_memory / (sizeof(iob_type) + SEGMENT_SIZE);
   if (max_iobs < 2) {
      max_iobs = 2;
      data_buffer_size = ((available_memory
                           - 2 * (sizeof(iob_type) + PAGE_SIZE)) / 2)
                          & ~(PAGE_SIZE - 1);
   }
   max_iobs = _min(max_iobs, MAX_IOBS);
   for (i = 0; i < max_iobs; i++) {
      CheckErc(ExpandAreaSL(sizeof(iob_type), selector_of(iob), &iob));
      CheckErc(Send(iob_exch, iob));
   }
   for (i = 0; i < max_iobs; i++) {
      CheckErc(ExpandAreaSL(PAGE_SIZE, selector_of(page_buffer),
                            &page_buffer));
      CheckErc(Send(page_buffer_exch, page_buffer));
   }
   for (i = 0; i < max_iobs; i++) {
      CheckErc(AllocMemorySL(data_buffer_size, &data_buffer));
      CheckErc(Send(data_buffer_exch, data_buffer));
   }
   if (recovery_buffer_size != 0)		/* Allocate tape EOM buffer? */
      CheckErc(AllocMemorySL(recovery_buffer_size - 1, &recovery_buffer));
   if (archive.rsam_buffer_size != 0)		/* Allocate RSAM buffer? */
      CheckErc(AllocMemorySL(archive.rsam_buffer_size, &archive.rsam_buffer));
   memset(&process_descriptor, 0, sizeof(process_descriptor));
   process_descriptor.pEntry = (Pointer)archive_io_process; 	/* m27 */
   process_descriptor.saData = selector_of(stack);
   process_descriptor.saStack = selector_of(stack);
   process_descriptor.oStackInit = offset_of(stack) + STACK_SIZE;
   process_descriptor.priority = priorityUser - 1;
   CheckErc(AllocExch(&process_descriptor.exchgDefaultResponse));
   CheckErc(CreateProcess(&process_descriptor));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Some sequential access devices (notably the HSB-001 QIC module) have a
 hardware forced rewind when a SeqAccessClose is performed.  This loses our
 position information if we are doing multiple volume backup to tape.  In
 order to avoid this difficulty, the tape is opened only once by the archive
 IO process---as opposed to subsequent "opens" of individual datasets on the
 tape.  Note that after the open we have to see if the default buffers
 provided by the device can accomodate the block size we are using.  If not,
 calculate how many buffers of the correct size can be had, close the device
 and reopen it to force a different buffer allocation. */

private void open_sequential_device(Boolean inbound) {

   unsigned erc;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                   archive.spec[archive.device_num][0]};

   read_archive_configuration();
   archive.piecemeal_size = calculate_piecemeal_size();
   if ((erc = SeqAccessOpen(&archive.handle, 
                            &archive.spec[archive.device_num][1],
                            archive.spec[archive.device_num][0],
                            &archive.password[archive.device_num][1],
                            archive.password[archive.device_num][0],
                            (inbound) ? MODE_READ : MODE_MODIFY,
                            &seq_parameters,
                            sizeof(seq_parameters))) == ercTapeInUse)
      log_msg(NLS_DEVICE_IN_USE, 2, nls_parms, sizeof(nls_parms));
   if (     erc == ercOK
         && (erc = SeqAccessModeQuery(archive.handle, &seq_parameters,
                                      sizeof(seq_parameters))) == ercOK
         && !seq_parameters.unbuffered
         && seq_parameters.block_size > seq_parameters.service_buffer_size
         && (erc = SeqAccessClose(archive.handle)) == ercOK) {
      seq_parameters.service_buffers = seq_parameters.service_buffer_pool_size
                                        / seq_parameters.block_size;
      seq_parameters.service_buffer_size = seq_parameters.block_size;
      if ((erc = SeqAccessOpen(&archive.handle,
                               &archive.spec[archive.device_num][1],
                               archive.spec[archive.device_num][0],
                               &archive.password[archive.device_num][1],
                               archive.password[archive.device_num][0],
                               (inbound) ? MODE_READ : MODE_MODIFY,
                               &seq_parameters,
                               sizeof(seq_parameters))) == ercOK)
         erc = SeqAccessModeQuery(archive.handle, &seq_parameters,
                                  sizeof(seq_parameters));
   }
   if (erc != ercOK)
      exit_with_msg(erc, (erc == ercServiceNotAvail)
                         ? NLS_NEED_SEQ_ACCESS_SERVICE : 0);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Attempt to open a device configuration file for the Sequential Access
 device.  Search for [Sys]<Sys>XxxxConfig.Sys, where XXX is the name of the
 sequential access device to be used for the archive operations.  If the file
 can be found, update the mode parameters for the subsequent SeqAccessOpen. */

private void read_archive_configuration(void) {

   char buffer[PAGE_SIZE], config_spec[MAX_FILE_SPECIFICATION_LENGTH],
      device[MAX_DEVICE_LENGTH], directory[MAX_DIRECTORY_LENGTH],
      filename[MAX_FILENAME_LENGTH], node[MAX_NODE_LENGTH],
      volume[MAX_VOLUME_LENGTH];
   unsigned config_fh = 0, config_spec_len, directory_len, filename_len, i,
      node_len, volume_len;
   Boolean device_at_server;
   unsigned *field_destination[] = {(void *) &seq_parameters.min_record_size,
                                    (void *) &seq_parameters.max_record_size,
                                    (void *) &seq_parameters.block_size,
                                    (void *) &seq_parameters.density,
                                    (void *) &seq_parameters.speed,
                                    (void *) &seq_parameters.unbuffered,
                                    (void *)
                                           &seq_parameters.service_buffer_size,
                                    (void *) &seq_parameters.service_buffers,
                                    (void *) &archive.erase_on_close,
                                    (void *) &archive.rewind_on_close};
   unsigned field_msg_num[] = {NLS_MIN_RECORD_SIZE_FIELD,
                               NLS_MAX_RECORD_SIZE_FIELD,
                               NLS_BLOCK_SIZE_FIELD, NLS_DENSITY_FIELD,
                               NLS_SPEED_FIELD, NLS_UNBUFFERED_FIELD,
                               NLS_SERVICE_BUFFER_SIZE_FIELD,
                               NLS_TOTAL_SERVICE_BUFFERS_FIELD,
                               NLS_ERASE_ON_CLOSE_FIELD,
                               NLS_REWIND_ON_CLOSE_FIELD};
   enum {NUMERIC, STRING} field_type[] = {NUMERIC, NUMERIC, NUMERIC, NUMERIC,
                                          NUMERIC, STRING, NUMERIC, NUMERIC,
                                          STRING, STRING};
   char string[255];
   sdType sd_archive = {device}, *sd_config, *sd_field, sd_string = {string};

   ParseFileSpec(0, &archive.spec[archive.device_num][1],
                 archive.spec[archive.device_num][0], FALSE, node,
                 &node_len, device, &sd_archive.cb, NULL, NULL, NULL,
                 NULL, NULL, NULL,
                 FALSE, 3);
   device_at_server = (device[0] == '!');
   if (device_at_server)
      memcpy(device, &device[1], --sd_archive.cb);
   sd_config = standard_msg(NLS_SEQ_CONFIG_FILE);
   ExpandLocalMsg(&sd_archive, sizeof(sd_archive), sd_config->pb,
                  sd_config->cb, filename, sizeof(filename), &filename_len,
                  FALSE);
   ParseFileSpec(0, filename, filename_len, FALSE, NULL, NULL, volume,
                 &volume_len, directory, &directory_len, filename,
                 &filename_len, NULL, NULL, FALSE, 0);
   if (device_at_server && volume_len < MAX_VOLUME_LENGTH) {
      _rmemcpy(&volume[1], volume, volume_len++);
      volume[0] = '!';
   }
   memset(config_spec, 0, sizeof(config_spec));
   BuildFileSpec(0, config_spec, &config_spec_len, NULL, sizeof(config_spec),
                 FALSE, node, node_len, volume, volume_len, directory,
                 directory_len, filename, filename_len, FALSE, NULL, 0,
                 FALSE, 0);
   if (     OpenFile(&config_fh, config_spec, config_spec_len, NULL, 0,
                     modeRead) != ercOK
         && volume[0] != '!' && volume_len < MAX_VOLUME_LENGTH) {
      _rmemcpy(&volume[1], volume, volume_len++);
      volume[0] = '!';
      memset(config_spec, 0, sizeof(config_spec));
      BuildFileSpec(0, config_spec, &config_spec_len, NULL,
                    sizeof(config_spec), FALSE, node, node_len, volume,
                    volume_len, directory, directory_len, filename,
                    filename_len, FALSE, NULL, 0, FALSE, 0);
      OpenFile(&config_fh, config_spec, config_spec_len, NULL, 0, modeRead);
   }
   for (i = 0; i <= last(field_msg_num); i++) {
      LookUpReset();
      if ((sd_field = standard_msg(field_msg_num[i])) != NULL)
         if (field_type[i] == NUMERIC)
            LookUpNumber(config_fh, buffer, sizeof(buffer), sd_field->pb,
                         sd_field->cb, field_destination[i]);
         else {
            LookUpString(config_fh, buffer, sizeof(buffer), sd_field->pb,
                         sd_field->cb, sd_string.pb, sizeof(string),
                         &sd_string.cb);
            NlsYesOrNo(NULL, &sd_string, field_destination[i]);
         }
   }
   if (     config_fh != 0
         && (   (   seq_parameters.unbuffered
                 && seq_parameters.block_size != PAGE_SIZE)
             || seq_parameters.min_record_size != PAGE_SIZE
             || seq_parameters.max_record_size != PAGE_SIZE
             || seq_parameters.block_size < PAGE_SIZE
             || seq_parameters.block_size
                 % seq_parameters.min_record_size != 0
             ||    seq_parameters.service_buffer_size == 0
                && seq_parameters.service_buffers != 0
             ||    seq_parameters.service_buffer_size != 0
                && (   seq_parameters.service_buffers == 0
                    || seq_parameters.service_buffer_size
                        < seq_parameters.block_size)
             || seq_parameters.service_buffer_size
                 % seq_parameters.block_size != 0))
      exit_with_msg(ercInvalidTapeConfigFile, 0);
   CloseFile(config_fh);
   archive.target_density = seq_parameters.density;
   archive.target_speed = seq_parameters.speed;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When sequential access medium is used and the service is located remotely
 (i.e. at the server or perhaps across the network), we must determine the
 largest size request that may be sent over the intervening transport
 layer(s).  Note that it is NOT relevant to piecemealing if the service is
 located on a remote processor board within the server; this is because the
 sequential access data transfer requests are styled 'dmaRemote' and the
 service, not the ICC, is responsible for transporting data between
 processors.  To reiterate, 'ercXbufTooSmall' is the ONLY error code that
 needs to be examined.  SeqAccessOpen works just fine for this purpose because
 a) the request is routed by device specification (so we know it gets all the
 way to the service) and b) it has a request parameter block that can be made
 arbitrarily large to test the maximum size.  Use binary search techniques to
 quickly zero in on the maximal block size. 
 
 WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING 
 
 Make sure that any changes in this procedure are also reflected in 
 <CTOS.Lib>SeqAccessTBs.plm
 
 */

private unsigned calculate_piecemeal_size(void) {

   unsigned handle, max_piecemeal_size = SEGMENT_SIZE - PAGE_SIZE,
      min_piecemeal_size = PAGE_SIZE,
      piecemeal_size = SEGMENT_SIZE - PAGE_SIZE,
      erc ;

   while (TRUE) {
      erc = SeqAccessOpen(&handle, &archive.spec[archive.device_num][1],
                        archive.spec[archive.device_num][0], NULL,
                        piecemeal_size, 0, NULL, 0);
      if (erc == ercXbufTooSmall || erc == 5002 || erc == 13578) {
         max_piecemeal_size = piecemeal_size;
         piecemeal_size = (piecemeal_size / 2 + min_piecemeal_size / 2)
                           & ~(PAGE_SIZE - 1);
      } else if (max_piecemeal_size - piecemeal_size > PAGE_SIZE) {
         min_piecemeal_size = piecemeal_size;
         piecemeal_size = (piecemeal_size / 2 + max_piecemeal_size / 2
                           + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1);
      } else
         return(piecemeal_size);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The archive I/O process has an outer and two inner loops.  Before a "mode"
 (write or read) has been set, wait for and respond to only Iob's with a
 control type of MODE.  Once writing or reading has been elected, drop into
 one of two inner loops.  For writing, open the archive file for output and
 continue waiting at the exchange to execute command instructions (File
 header, file data, eond of file, end of data, etc.) as they arrive from the
 archive backup process.  If reading, open the archive file for input and
 commence a loop to read data from the file as long as buffers are available
 (and EOF has not been reached).  This data is parsed into file header, file
 data, end of file, etc. messages attached to Iob's and passed back to the
 principal archive restore process.  At completion, close the archive file and
 resume waiting for a MODE command. */

private void archive_io_process(void) {

   char ctrl;
   unsigned erc;
   iob_type *iob;

   while (TRUE) {
      if ((erc = Wait(archive.msg_exch, &iob)) != ercOK)
         exit_with_msg(erc, 0);
      if (iob->ctrl == MODE) {
         if (iob->mode.write) {
            deallocate_iob(iob);
            archive.inbound = FALSE;
            archive.file_num = 0;
            if (archive.sequential)
               open_archive_seq_dataset();
            else
               open_archive_output_dataset();
            do {
               if ((erc = Wait(archive.msg_exch, &iob)) != ercOK)
                  exit_with_msg(erc, 0);
               interpret_archive_command(iob);
               ctrl = iob->ctrl;
               deallocate_iob(iob);
            } while (ctrl != EOD);
         } else {
            deallocate_iob(iob);
            archive.inbound = TRUE;
            if (archive.sequential)
               open_archive_seq_dataset();
            else
               open_archive_input_dataset();
            do {
               if (archive.old_format)
                  read_archive_fhb();
               else
                  read_archive_sentinel(TRUE);
               if (archive_sentinel.record_type != EOD) {
                  while (target.size > 0) {
                     iob = allocate_iob(SEGMENT_SIZE - PAGE_SIZE);
                     if (target.size < iob->available)
                        iob->available = target.size;
                     read_archive_data(iob->data, iob->available, FALSE);
                     iob->ctrl = DATA;
                     Send(target.msg_exch, iob);
                     target.size -= iob->available;
                  }
                  if (vlpb.file_trailer)
                     read_archive_sentinel(FALSE);
               }
            } while (archive_sentinel.record_type != EOD);
            if (archive.sequential)
               close_archive_seq_dataset(FALSE, TRUE);
            else
               close_archive_dataset();
         }
         iob = allocate_iob(0);
         iob->ctrl = SYNC;
         Send(target.msg_exch, iob);
      } else
         deallocate_iob(iob);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Opening sequential (tape) datasets involves access to the correct device,
 positioning of the medium at the correct file location and verification of
 the presence or absence of a dataset label (either 12.0 style or new 12.1
 style).  After gaining access to the device through the Sequential Access
 service, the medium is positioned to the requested file position (0 through
 N, or +).  If the dataset is being opened for reading or if the file position
 is not 0xFFFF (indicating the "append" mode for writing) and deletion of an
 existing dataset has not been automatically authorized by the user, an
 attempt is made to read a valid dataset label from the medium.  If a non-zero
 error code is returned, the need for user interaction is indicated (a
 descriptive message has already been displayed to the user) and the 'prompt'
 variable is set.  Otherwise, the dataset is opened OK and if we are archiving
 data a dataset label is written.  Throughout this whole sequence, the archive
 co-process is locked from using the video. */

private void open_archive_seq_dataset(void) {

   unsigned erc = ercOK;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
   			 (void *) &erc, sizeof(erc),
   			 (void *) &archive.sequence, sizeof(archive.sequence)};
   static Boolean prompt = FALSE;
   static Boolean last_vol = FALSE;

   if (archive.sequence++ != 0) {	/* Request next tape in sequence... */
      archive.target_position = 0;	/* ...but change [XXX]+ to [XXX]0 */
      /*(archive.device_num_last < 2) because 0 or 1 params could be entered*/
      prompt |= ( (!archive.first_volume) && (archive.device_num_last < 2) )
                || last_vol;
   } 
   else
      if (!archive.inbound) 			  /* Write to Tape m23*/
         prompt &= (archive.current_position == 0xFFFF); /* m23 */

   lock_video(2);
   do {
      if (prompt) {
         if (vlpb.noninteractive)
            exit_with_msg(ercOperatorIntervention, NLS_CANT_MOUNT_ARCHIVE);
         else {
            vid_only_msg(NLS_MOUNT_ARCHIVE_TAPE, 2, nls_parms,
                         sizeof(nls_parms));
            if (!proceed())
               exit_with_msg(ercOperatorIntervention, 0);
            if (archive.sequence == 0)				/* m23 */
               archive.current_position = 0xFFFF;/*Unknown after tape change*/
         }
      }
      archive.block_size = seq_parameters.block_size;	/* First attempt... */
      if ((erc = position_archive()) != ercOK)
         exit_with_msg(erc, 0);
      if (     archive.inbound
            || (archive.target_position == 0 && !vlpb.overwrite_OK))
         erc = read_archive_label();
      if (erc != ercOK)
         prompt = TRUE;			/* Current medium is unacceptable */
      else if (!archive.inbound)	/* OK, but is it a write? */
         erc = write_archive_label();
   } while (erc != ercOK);
   unlock_video(2);

   if (archive.device_num == archive.device_num_last) last_vol = TRUE;
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Archive datasets on sequential access medium are located either at the
 beginning of the medium (designated by names such as [XXX] or [XXX]0), at the
 end of data on the medium (designated by a name such as [XXX]+) or are one of
 the N datasets in between differentiated by file marks (designated by a name
 such as [XXX]n).  Commence by rewinding the medium and then, for the latter
 two cases, position forward through file marks until the sought after ending
 condition or an error is encountered.  Note that 1/2" tape devices (as
 indicated by the medium density code) cannot detect blank medium conditions
 and the "logical" end of the record medium is denoted by two consecutive
 filemarks---if found, we reverse over one and can commence the new data file
 at that location.  Also note that 'ercTapeBlank' must be converted to a
 normal error return code when we are positioning to [QIC]+ or [DDS]+ (default
 devices). */

private unsigned position_archive(void) {

   unsigned erc, i, msg_num, seq_status;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
                         (void *) &archive.target_position,
                                  sizeof(archive.target_position),
                         (void *) &erc, sizeof(erc)};
   unsigned long residual;
   Boolean half_inch, retry;

   half_inch = (   seq_parameters.density == NRZI_800
                || seq_parameters.density == PE_1600
                || seq_parameters.density == PE_3200
                || seq_parameters.density == GCR_6250);
   do {
      for (i = 0; i < 10; i++)
         if ((erc = SeqAccessStatus(archive.handle, &seq_status,
                                    sizeof(seq_status),
                                    &residual)) == ercTapeStatusUnavailable)
            return(erc);
         else if (erc != ercTapeReset)
            break;
      if (     erc != ercTapeNotReady && erc != ercTapeBusy
            && erc != ercTapeReset) {
         erc = ercOK;		/* No other status is of interest (e.g. EOD) */
         if (!archive.inbound && (seq_status & SEQ_WRITE_PROTECT))
            erc = ercTapeWriteProtected;
         else if (   archive.current_position != 0xFFFF
                  && archive.current_position == archive.target_position) {
            if (half_inch && (erc = SeqAccessControl(archive.handle,
                                                     CTRL_SCAN_FILEMARK, -1,
                                                     &residual)) != ercOK)
               archive.current_position = archive.target_position = 0xFFFF;
         } else if ((erc = SeqAccessControl(archive.handle, CTRL_REWIND,
                                            SYNCHRONIZE,
                                            &residual)) == ercOK) {
            archive.current_position = 0;
            if (archive.target_position != 0)	/* Go to EOD or to Nth file */
               if (half_inch && archive.target_position == 0xFFFF) {
                  if ((erc = SeqAccessControl(archive.handle,
                                              CTRL_SCAN_FILEMARK, 2,
                                              &residual)) == ercOK)
                     if (    (erc = SeqAccessStatus(archive.handle,
                                                    &seq_status,
                                                    sizeof(seq_status),
                                                    &residual))
                           == ercTapeStatusUnavailable)
                        return(erc);
                     erc = (seq_status & SEQ_EOM) ? ercTapeEomWarning
                            : SeqAccessControl(archive.handle,
                                               CTRL_SCAN_FILEMARK, -1,
                                               &residual);
               } else {
                  while (   (erc = SeqAccessControl(archive.handle,
                                                    CTRL_SCAN_FILEMARK, 1,
                                                    &residual)) == ercOK
                         && ++archive.current_position
                               < archive.target_position)
                     ;
                  if (     archive.target_position == 0xFFFF
                        && erc == ercTapeBlank
                        && !archive.inbound
                        && SeqAccessStatus(archive.handle, &seq_status,
                                           sizeof(seq_status),
                                           &residual) !=
                                                      ercTapeStatusUnavailable)
                     erc = (seq_status & SEQ_EOM) ? ercTapeEomWarning : ercOK;
               }
         }
      }
      if (erc != ercOK) {
         if (     erc == ercTapeNotReady || erc == ercTapeBusy
               || erc == ercTapeReset)
            msg_num = NLS_DEVICE_NOT_READY;
         else if (erc == ercTapeWriteProtected)
            msg_num = NLS_WRITE_PROTECTED;
         else if (erc == ercTapeEomWarning)
            msg_num = NLS_MEDIUM_FULL;
         else if (erc == ercTapeBlank || erc == ercTapeOverflow)
            msg_num = NLS_NO_ARCHIVE_LABEL;
         else
            msg_num = NLS_POSITIONING_ERROR;
         if (vlpb.noninteractive) {
            log_msg(msg_num, 2, nls_parms, sizeof(nls_parms));
            retry = FALSE;
         } else {
            vid_only_msg(msg_num, 2, nls_parms, sizeof(nls_parms));
            vid_only_msg(NLS_REQUEST_INTERVENTION, 2, NULL, 0);
            retry = proceed();
         }
      }
   } while (erc != ercOK && retry);
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure is called for any mounted tape before either writing or
 reading.  For writing, an attempt is made to find either style of volume
 label at the beginning of the tape.  If one is found, the user may prompted
 (in the procedures that follow) for overwrite permission.  If no label is
 found (because of a blank tape or because the data doesn't conform to the
 label format), the error code is translated to OK to grant permission to
 write one the tape.  For reading, both label formats are checked on the first
 volume, only.  Once a valid dataset is mounted and reading has started, the
 second and subsequent volumes must have the same style of label as the
 first.  Note that the display of the "sequence number error" message must be
 performed in the format-specific procedures (the sequence number is in
 different locations) but that all other label errors are shown to the user in
 this procedure.  Also note that the tape usually needs to be repositioned
 when writing, but only if some data transfer has taken place while searching
 for the label. */

private unsigned read_archive_label(void) {

   archive_label_type archive_label;
   unsigned erc;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
                         (void *) &archive.target_position,
                                  sizeof(archive.target_position),
                         (void *) &archive.block_size,
                                  sizeof(archive.block_size)};
   Boolean reposition = FALSE;
   unsigned long residual;

   reposition = (   (erc = SeqAccessRead(archive.handle, &archive_label,
                                         sizeof(archive_label),
                                         &residual)) != ercTapeBlank
                 && !archive.inbound);
   if (erc == ercOK) {
      if (archive.first_volume || !archive.inbound) {
         if (     (erc = valid_archive_bov(&archive_label)) == ercOK
               && archive.inbound
               && archive.block_size != seq_parameters.block_size) {
            archive.current_position = 0xFFFF;	/* Repositioning is needed */
            if ((erc = position_archive()) == ercOK) {
               seq_parameters.block_size = archive.block_size;
               erc = SeqAccessModeSet(archive.handle, &seq_parameters,
                                      sizeof(seq_parameters));
               SeqAccessModeQuery(archive.handle, &seq_parameters,
                                     sizeof(seq_parameters));
               if (erc == ercOK)
                  erc = SeqAccessRead(archive.handle, &archive_label,
                                      sizeof(archive_label), &residual);
            }
         } else if (erc == ercInvalidTapeReel)
            erc = valid_vam_volume_hdr((vam_volume_hdr_type *) &archive_label);
      } else if (archive.old_format)
         erc = valid_vam_volume_hdr((vam_volume_hdr_type *) &archive_label);
      else
         erc = valid_archive_bov(&archive_label);
   } else if (   (   erc == ercTapeRecordTruncated
                  || erc == ercInvalidTapeRecordSize)
              && sizeof(archive_label) - HALF_INCH_LABEL_SIZE == residual) {
      erc = valid_vam_volume_hdr((vam_volume_hdr_type *) &archive_label);
      if (erc == ercOK && archive.inbound) {
         seq_parameters.block_size = archive.block_size;
         if ((erc = SeqAccessModeSet(archive.handle, &seq_parameters,
                                     sizeof(seq_parameters))) == ercOK)
            SeqAccessModeQuery(archive.handle, &seq_parameters,
                               sizeof(seq_parameters));
      }
   }
   if (erc == ercInvalidTapeParams) {
      if (archive.block_size > seq_parameters.service_buffer_size)
         log_msg(NLS_SERVICE_BUFFER_TOO_SMALL, 2, nls_parms,
                 sizeof(nls_parms));
      exit_with_msg(ercInvalidTapeParams, 0);
   }
   if (archive.inbound) {	/* Was the label absent? */
      if (erc != ercOK && erc != ercInvalidTapeSequence)
         vid_only_msg(NLS_NO_ARCHIVE_LABEL, 2, nls_parms, sizeof(nls_parms));
   } else {
      if (erc != ercDontOverwriteTape)
         erc = ercOK;	/* Any label errors mean "go ahead and write" */
      if (reposition) {	/* There may have been tape motion to detect label */
         archive.current_position = 0xFFFF;	/* So, this is unknown... */
         position_archive();
      }
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 512 bytes of data have been read from the beginning of what may be an archive
 dataset.  The procedure below compares signature and checksum information
 for the tentative dataset label and if it is a valid label proceeds to check
 sequencing information (for reads) or overwrite permission (for writes). */

private unsigned valid_archive_bov(archive_label_type *archive_label) {

   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
                         (void *) &archive_label->creation_time,
                                  sizeof(archive_label->creation_time),
                         (void *) &archive_label->sequence,
                                  sizeof(archive_label->sequence),
                         (void *) &archive.target_position,
                                  sizeof(archive.target_position)};
   sdType *sd_bov = standard_msg(NLS_ARCHIVE_BOV);

   if (     !checksum_valid(archive_label)
         || ULCmpB(archive_label->description, sd_bov->pb,
                   sizeof(archive_label->description)) != 0xFFFF)
      return(ercInvalidTapeReel);
   if (archive.inbound)
      if (archive_label->sequence == archive.sequence) {
         if (archive.first_volume) {
            summary.count[TOTAL_FILES] = archive_label->total_files;
            summary.count[PROCESSED] = archive_label->file_num;
            archive.block_size = (  archive_label->block_size == 0
                                 || archive_label->block_size % PAGE_SIZE != 0)
                                       ? PAGE_SIZE : archive_label->block_size;
            archive.creation_time = archive_label->creation_time;
            log_msg(NLS_RESTORING_FROM, 2, nls_parms, sizeof(nls_parms));
            archive.first_volume = FALSE;
         }
         return(ercOK);
      } else {
         vid_only_msg(NLS_ARCHIVE_SEQUENCE_ERROR, 2, nls_parms,
                      sizeof(nls_parms));
         return(ercInvalidTapeSequence);
      }
   else if (vlpb.overwrite_OK)
      return(ercOK);
   else {
      vid_only_msg(NLS_ARCHIVE_SEQ_DATASET_EXISTS, 2, nls_parms,
                   sizeof(nls_parms));
      return((proceed()) ? ercOK : ercDontOverwriteTape);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Well, the label read from the beginning of the dataset was not recognizable
 as a 12.1 style label.  Maybe this is a tape produced by 12.0 software.
 There is no checksum information for the old labels, just see if an expected
 string is present in the expected location.  If so, it's a valid label --- go
 ahead and check the sequence information (if we're reading) or confirm
 overwrite permission (if we're writing). */

private unsigned valid_vam_volume_hdr(vam_volume_hdr_type *vam_volume_hdr) {

   unsigned erc, i, sequence;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
                         (void *) vam_volume_hdr->backup_time, 0,
                         (void *) &sequence, sizeof(sequence),
                         (void *) &archive.target_position,
                                  sizeof(archive.target_position)};

   sdType *sd_volume_hdr = standard_msg(NLS_OLD_BACKUP_BOV);

   if (ULCmpB(vam_volume_hdr->backup_hdr, sd_volume_hdr->pb,
              sd_volume_hdr->cb) != 0xFFFF)
      if (ULCmpB(vam_volume_hdr->backup_hdr, sd_volume_hdr->pb,
                 3) != 0xFFFF)
         return(ercInvalidTapeReel);
   for (i = last(vam_volume_hdr->backup_time); i >= 0; i--)
      if (vam_volume_hdr->backup_time[i] != ' ')
         break;
/* nls_parms[1].cb = i + 1;							  was  m25 */
   nls_parms[1].cb = (i < 255) ? i + 1 : 256;			/* m25 */
   if (vam_volume_hdr->backup_id[last(vam_volume_hdr->backup_id)] == 0)
      i = strlen(vam_volume_hdr->backup_id);
   else {
      for (i = last(vam_volume_hdr->backup_id); i >= 0; i--)
         if (vam_volume_hdr->backup_id[i] != ' ')
            break;
      i++;
   }
   if (vam_volume_hdr->backup_id[0] == '\n')
      memcpy(vam_volume_hdr->backup_id, &vam_volume_hdr->backup_id[1], --i);
   if (archive.inbound)
      if (     (sequence = atoi(vam_volume_hdr->backup_sequence))
            == archive.sequence) {
         if (archive.first_volume) {
            archive.old_format = TRUE;
            vam.block_size = atoi(vam_volume_hdr->backup_block_size);
            if (     seq_parameters.density == NRZI_800
                  || seq_parameters.density == PE_1600
                  || seq_parameters.density == PE_3200
                  || seq_parameters.density == GCR_6250)
               archive.block_size = vam.block_size;
            else
               summary.count[TOTAL_FILES] = vam_volume_hdr->total_files;
            if (     vam.block_buffer == NULL
                  && (erc = AllocMemorySL(vam.block_size,
                                          &vam.block_buffer)) != ercOK)
               exit_with_msg(erc, 0);
            log_msg(NLS_RESTORING_FROM_OLD_FORMAT, 2, nls_parms,
                    sizeof(nls_parms));
         }
         return(ercOK);
      } else {
         vid_only_msg(NLS_ARCHIVE_SEQUENCE_ERROR, 2, nls_parms,
                      sizeof(nls_parms));
         return(ercInvalidTapeSequence);
      }
   else if (vlpb.overwrite_OK)
      return(ercOK);
   else {
      vid_only_msg(NLS_OLD_FORMAT_DATASET_EXISTS, 2, nls_parms,
                   sizeof(nls_parms));
      return((proceed()) ? ercOK : ercDontOverwriteTape);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Create a 12.1 style archive dataset label and write it at the beginning of
 the dataset.  Besides sequencing information, date time stamps, information
 necessary to configure the Sequential Access service device, etc. the dataset
 label contains some text information that may be helpful to users who need to
 know what software and version created the tape. */

private unsigned write_archive_label(void) {

   archive_label_type archive_label;
   unsigned erc;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
                         (void *) &archive.current_position,
                                  sizeof(archive.current_position)};
   sdType sd_command_name, *sd_msg;
   extern char sbVerRun[];

   if (     seq_parameters.density != NRZI_800
         && seq_parameters.density != PE_1600
         && seq_parameters.density != PE_3200
         && seq_parameters.density != GCR_6250)
      log_msg(NLS_CREATING_SEQ_DATASET, 2, nls_parms, sizeof(nls_parms));
   seq_parameters.density = archive.target_density;
   seq_parameters.speed = archive.target_speed;
   if ((erc = SeqAccessModeSet(archive.handle, &seq_parameters,
                               sizeof(seq_parameters))) != ercOK)
      return(erc);
   SeqAccessModeQuery(archive.handle, &seq_parameters, sizeof(seq_parameters));
   memset(&archive_label, 0, sizeof(archive_label));
   archive_label.signature = ARCHIVE_SIGNATURE;
   archive_label.record_type = BOV;
   sd_msg = standard_msg(NLS_ARCHIVE_BOV);
   memcpy(archive_label.description, sd_msg->pb, sd_msg->cb);
   archive_label.sequence = archive.sequence;
   archive_label.total_files = summary.count[TOTAL_FILES];
   archive_label.file_num = archive.file_num;
   archive_label.block_size = archive.block_size;
   if (archive.creation_time == 0)
      GetDateTime(&archive.creation_time);
   archive_label.creation_time = archive.creation_time;
   memset(archive_label.oem_text, ' ', sizeof(archive_label.oem_text));
   if ((sd_msg = standard_msg(NLS_OEM_TEXT)) != NULL)
      memcpy(archive_label.oem_text, sd_msg->pb,
             _min(sd_msg->cb, sizeof(archive_label.oem_text)));
   memset(archive_label.creator, ' ', sizeof(archive_label.creator));
   RgParam(0, 0, &sd_command_name);
   memcpy(archive_label.creator, sd_command_name.pb,
          _min(sd_command_name.cb, sizeof(archive_label.creator)));
   memset(archive_label.version, ' ', sizeof(archive_label.version));
   memcpy(archive_label.version, &sbVerRun[1],
          _min(sbVerRun[0], sizeof(archive_label.version)));
   archive_label.checksum = checksum(&archive_label);
   write_archive_data(&archive_label, sizeof(archive_label));
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When an output archive dataset is closed, de facto it is the last logical
 dataset on the tape.  For some media (e.g. QIC and DDS) no special action
 need be taken: the entire tape was erased simultaneously with writing the
 first file on the tape and one filemark is sufficient to terminate the data.
 With 1/2" tape however, the special token of two consecutive filemarks is
 required to signal the logical end of tape.  For both input and output
 datasets, don't change the medium position (i.e. rewind) unless a medium
 change has been indicated or this is the last dataset to be processed by the
 program.  In the last case, we can release the device with a SeqAccessClose,
 also. */

private void close_archive_seq_dataset(Boolean change_medium,
                                       Boolean final_close) {

   unsigned erc = ercOK, filemarks;
   unsigned long residual;

   filemarks = (   seq_parameters.density == NRZI_800
                || seq_parameters.density == PE_1600
                || seq_parameters.density == PE_3200
                || seq_parameters.density == GCR_6250) ? 2 : 1;
   if (     !archive.inbound
         && (erc = SeqAccessControl(archive.handle, CTRL_WRITE_FILEMARK,
                                    filemarks,
                                    &residual)) == ercTapeEomWarning)
      erc = ercOK;
   if (erc == ercOK) {
      archive.current_position++;
      if (change_medium)								/* ...m23 */
         archive.current_position = 0xFFFF; /* pos unknown after tape change*/
   }													/* ...m23 */
   archive.target_position++;
   if (change_medium || final_close) {
      if (     !archive.inbound && archive.erase_on_close && erc == ercOK
            && (   (erc = SeqAccessControl(archive.handle, CTRL_ERASE_MEDIUM,
                                           SYNCHRONIZE,
                                           &residual)) == ercTapeCommandIllegal
                || erc == ercTapeEomWarning))
         erc = ercOK;
      if (change_medium && erc == ercOK)
         SeqAccessControl(archive.handle, CTRL_UNLOAD, SYNCHRONIZE, &residual);
      else if (final_close && erc == ercOK) {
         if (archive.rewind_on_close)
            SeqAccessControl(archive.handle, CTRL_REWIND, SYNCHRONIZE,
                             &residual);
         erc = SeqAccessClose(archive.handle);
      }
   }
   if (erc != ercOK)
      exit_with_msg(erc, 0);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void open_archive_output_dataset(void) {

   unsigned erc;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
   			 (void *) &erc, sizeof(erc)};
   static Boolean prompt = FALSE;

   if (++archive.sequence >= 100) {
      if (archive.sequence == 100)
         nls_parms[0].cb = ++archive.spec[archive.device_num][0];
      archive.spec[archive.device_num][archive.spec[archive.device_num][0] -2] 
                  = (archive.sequence / 100) | '0';
   }
   archive.spec[archive.device_num][archive.spec[archive.device_num][0] - 1]
               = ((archive.sequence / 10) % 10) | '0';
   archive.spec[archive.device_num][archive.spec[archive.device_num][0]]
               = (archive.sequence % 10) | '0';
   archive.handle = archive.lfa = archive.size = 0;
   lock_video(2);
   while (archive.handle == 0) {	/* Continue until a file exists */
      if (prompt) {
         vid_only_msg(NLS_MOUNT_ARCHIVE_DISK, 2, nls_parms, sizeof(nls_parms));
         if (!proceed())
            exit_with_msg(ercOperatorIntervention, 0);
      }
      erc = CreateFile(&archive.spec[archive.device_num][1],
                       archive.spec[archive.device_num][0],
                       &archive.password[archive.device_num][1],
                       archive.password[archive.device_num][0], 0);
      if (erc == ercOK || erc == ercFileAlreadyExists) {
         if (erc == ercFileAlreadyExists && !vlpb.overwrite_OK)
            if (vlpb.noninteractive)
               exit_with_msg(ercFileAlreadyExists, NLS_ARCHIVE_DATASET_EXISTS);
            else {
               vid_only_msg(NLS_ARCHIVE_DATASET_EXISTS, 2, NULL, 0);
               if (!proceed())
                  exit_with_msg(erc, 0);
            }
         if (prompt)
            vid_only_msg(NLS_CREATING_ARCHIVE, 2, NULL, 0);
         erc = OpenFile(&archive.handle, &archive.spec[archive.device_num][1],
                        archive.spec[archive.device_num][0],
                        &archive.password[archive.device_num][1],
                        archive.password[archive.device_num][0], modeModify);
         if (erc == ercOK)
            erc = ChangeFileLength(archive.handle, ARCHIVE_SIZE);
         if (   (erc == ercOK || erc == ercDiskFull)
             && (erc = GetFileStatus(archive.handle, FILE_STATUS_EOF,
                                     &archive.size,
                                     sizeof(archive.size))) == ercOK) {
            if (MIN_ARCHIVE_SIZE > archive.size) {
               DeleteFile(archive.handle);
               archive.handle = 0;
               if (vlpb.noninteractive)
                  exit_with_msg(ercDiskFull, NLS_DISK_FULL);
               else if (prompt)
                  vid_only_msg(NLS_DISK_FULL, 2, NULL, 0);
               else
                  prompt = TRUE;
            }
         } else {
            DeleteFile(archive.handle);
            archive.handle = 0;
         }
      }
      if (erc != ercOK)
         if (vlpb.noninteractive) {
            log_msg(NLS_CANT_CREATE_ARCHIVE_DATASET, 2, nls_parms,
                    sizeof(nls_parms));
            exit_with_msg(erc, 0);
         } else if (prompt)
            vid_only_msg(NLS_CANT_CREATE_ARCHIVE_DATASET, 2, nls_parms,
                         sizeof(nls_parms));
         else
            prompt = TRUE;
   }
   if (prompt)
      vid_only_msg(NLS_DONE, 2, NULL, 0);
   unlock_video(2);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void open_archive_input_dataset(void) {

   unsigned erc = 0xFFFF, xfer_count;
   sdType nls_parms[] = {(void *) &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
   			 (void *) &erc, sizeof(erc)};
   char rsam_record[PAGE_SIZE + 4];
   static Boolean prompt = FALSE;

   if (++archive.sequence >= 100) {
      if (archive.sequence == 100)
         nls_parms[0].cb = ++archive.spec[archive.device_num][0];
      archive.spec[archive.device_num][archive.spec[archive.device_num][0]- 2]
                  = (archive.sequence / 100) | '0';
   }
   archive.spec[archive.device_num][archive.spec[archive.device_num][0] - 1]
               = ((archive.sequence / 10) % 10) | '0';
   archive.spec[archive.device_num][archive.spec[archive.device_num][0]]
               = (archive.sequence % 10) | '0';
   archive.handle = archive.lfa = 0;
   lock_video(2);
   while (erc != ercOK) {
      if (prompt) {
         vid_only_msg(NLS_MOUNT_ARCHIVE_DISK, 2, nls_parms, sizeof(nls_parms));
         if (!proceed())
            exit_with_msg(ercOperatorIntervention, 0);
      }
      if (archive.first_volume) {		/* Have to try both formats? */
         if (     (erc = OpenRsFile(archive.rswa,
                                    &archive.spec[archive.device_num][1],
                                    archive.spec[archive.device_num][0],
                                    &archive.password[archive.device_num][1],
                                    archive.password[archive.device_num][0],
                                    modeRead,
                                    archive.rsam_buffer,
                                    archive.rsam_buffer_size)) == ercOK
               && (erc = ReadRsRecord(archive.rswa, &rsam_record,
                                      sizeof(rsam_record),
                                      &xfer_count)) == ercOK) {
            archive.old_format = TRUE;		/* OK, it's an RSAM file */
            if ((erc = SetRsLfa(archive.rswa, 0)) != ercOK)
               exit_with_msg(erc, 0);
         } else if ((erc = OpenFile(&archive.handle,
                                  &archive.spec[archive.device_num][1],
                                  archive.spec[archive.device_num][0],
                                  &archive.password[archive.device_num][1],
                                  archive.password[archive.device_num][0],
                                  modeRead)) == ercOK)
            archive.old_format = archive.first_volume = FALSE;	/* New style */
      } else if (archive.old_format)
         erc = OpenRsFile(archive.rswa,
                          &archive.spec[archive.device_num][1],
                          archive.spec[archive.device_num][0],
                          &archive.password[archive.device_num][1],
                          archive.password[archive.device_num][0],
                          modeRead,
                          archive.rsam_buffer, archive.rsam_buffer_size);
      else
         erc = OpenFile(&archive.handle,
                        &archive.spec[archive.device_num][1],
                        archive.spec[archive.device_num][0],
                        &archive.password[archive.device_num][1],
                        archive.password[archive.device_num][0], modeRead);
      if (erc != ercOK)
         if (vlpb.noninteractive) {
            log_msg(NLS_CANT_OPEN_ARCHIVE_DATASET, 2, nls_parms,
                    sizeof(nls_parms));
            exit_with_msg(erc, 0);
         } else if (prompt)
            vid_only_msg(NLS_CANT_OPEN_ARCHIVE_DATASET, 2, nls_parms,
                         sizeof(nls_parms));
         else
            prompt = TRUE;
   }
   unlock_video(2);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Be parsimonious with disk space, and shorten the original file allocation if
 we didn't use all of it.  Then, just use the appropriate close operation for
 the file.  As a safety precaution, clear the handle. */

private void close_archive_dataset(void) {

   unsigned erc = ercOK;

   if (!archive.inbound && archive.lfa < archive.size)
      erc = ChangeFileLength(archive.handle, archive.lfa);
   if (erc == ercOK)
      erc = (archive.old_format) ? CloseRsFile(archive.rswa)
                                 : CloseFile(archive.handle);
   if (erc != ercOK)
      exit_with_msg(erc, 0);
   archive.handle = 0;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void interpret_archive_command(iob_type *iob) {

   switch (iob->ctrl) {
      case BOF:
         archive.file_num++;
         write_archive_data(iob->base, iob->available);
         memcpy(&archive_sentinel, iob->base, sizeof(archive_sentinel));
         break;

      case DATA:
         write_archive_data(iob->base, iob->available);
         break;

      case EOF:
         if (vlpb.file_trailer)
            write_archive_data(&archive_sentinel, sizeof(archive_sentinel));
         break;

      case EOD:
         write_archive_data(iob->base, iob->available);
         if (archive.sequential)
            close_archive_seq_dataset(FALSE, iob->mode.final_close);
         else
            close_archive_dataset();
         break;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void write_archive_data(void *buffer, unsigned buffer_len) {

   unsigned erc, i, j, seq_status, xfer_actual, xfer_size;
   unsigned long buffer_residual, checkpoint_residual, recovery_len,
      recovery_residual, residual;

   if (vlpb.file_trailer) {
      j = buffer_len / 2;
      for (i = 0; i < j; i++)
         archive_sentinel.file_checksum += ((unsigned *) buffer)[i];
   }
   while (buffer_len > 0) {
      if (archive.sequential) {
         erc = SeqAccessWrite(archive.handle, buffer,
                              xfer_size = _min(buffer_len,
                                               archive.piecemeal_size),
                              &residual);
         if (residual <= xfer_size) {
            xfer_actual = xfer_size - residual;
            residual = 0;
         } else {
            xfer_actual = 0;
            residual -= xfer_size;
         }
      } else
         erc = Write(archive.handle, buffer, buffer_len, archive.lfa,
                     &xfer_actual);
      offset_of(buffer) += xfer_actual;
      buffer_len -= xfer_actual;
      archive.lfa += xfer_actual;
      if (erc == ercEOM) {
         close_archive_dataset();
         open_archive_output_dataset();
      } else if (erc == ercTapeEomWarning) {
         recovery_len = 0;
         if (residual > recovery_buffer_size) {
            if (        (erc = SeqAccessStatus(archive.handle, &seq_status,
                                               sizeof(seq_status),
                                               &buffer_residual))
                     != ercTapeStatusUnavailable
                  &&    (erc = SeqAccessCheckpoint(archive.handle,
                                                   &checkpoint_residual))
                     == ercOK)
               if (checkpoint_residual == 0) {
                  xfer_actual = buffer_residual - residual;
                  offset_of(buffer) += xfer_actual;
                  buffer_len -= xfer_actual;
                  archive.lfa += xfer_actual;
               } else
                  erc = ercTapeOverflow;
         } else {
            while (residual > 0) {
               xfer_size = _min(residual, archive.piecemeal_size);
               if (     (erc = SeqAccessRecoverBufferData(archive.handle,
                                                          recovery_buffer,
                                                          xfer_size,
                                                          &recovery_residual))
                     != ercOK)
                  exit_with_msg(erc, 0);
               xfer_actual = (xfer_size - recovery_residual);
               offset_of(recovery_buffer) += xfer_actual;
               residual -= xfer_actual;
               recovery_len += xfer_actual;
            }
            offset_of(recovery_buffer) = 0;	/* Reset after piecemeal */
            erc = SeqAccessDiscardBufferData(archive.handle);
         }
         if (erc != ercOK)
            exit_with_msg(erc, 0);
/*       close_archive_seq_dataset(TRUE, FALSE);		m21... */
         close_archive_seq_dataset(TRUE, 
                      ((archive.device_num_last > 1)	/* m22 was > 0 */
                      && (archive.device_num < (MAX_NUMBER_OF_DEVICES - 1)) ));
         if ( (archive.device_num_last > 1)				/* m22 was > 0 */
            && (archive.device_num < (MAX_NUMBER_OF_DEVICES - 1)) ) {
            archive.device_num++;
            open_sequential_device(FALSE);
         }									/* ...m21 */
         open_archive_seq_dataset();
         while (recovery_len > 0) {
            xfer_size = _min(recovery_len, archive.piecemeal_size);
            if ((erc = SeqAccessWrite(archive.handle, recovery_buffer,
                                      xfer_size, &residual)) != ercOK)
               exit_with_msg(erc, 0);
            else {
               xfer_actual = (xfer_size - residual);
               offset_of(recovery_buffer) += xfer_actual;
               recovery_len -= xfer_actual;
            }
         }
         offset_of(recovery_buffer) = 0;	/* Restore after piecemeal */
      } else if (erc != ercOK)
         exit_with_msg(erc, 0);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void read_archive_sentinel(Boolean file_header_sentinel) {

   unsigned erc;
   iob_type *iob;

   do
      read_archive_data(&archive_sentinel, sizeof(archive_sentinel)
                       ,is_first_archive_volume);					/* m19 */
   while (   archive_sentinel.signature != ARCHIVE_SIGNATURE
          || !checksum_valid(&archive_sentinel)
          || (   archive_sentinel.dataset_creation_time != 0
              && archive_sentinel.dataset_creation_time
                  != archive.creation_time));
   if (is_first_archive_volume) is_first_archive_volume = FALSE;	/* m19 */
   if (file_header_sentinel) {
      iob = allocate_iob(sizeof(archive_sentinel));
      target.size = archive_sentinel.pages_data * PAGE_SIZE;
      memcpy(iob->data, &archive_sentinel, sizeof(archive_sentinel));
   } else {
      iob = allocate_iob(0);
      iob->mode.checksum_error =
                          (archive_sentinel.checksum != archive.file_checksum);
   }
   iob->ctrl = archive_sentinel.record_type;
   archive.file_num++;
   if ((erc = Send(target.msg_exch, iob)) != ercOK)
      exit_with_msg(erc, 0);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void read_archive_fhb(void) {

   archive_fhb_type archive_fhb;
   unsigned erc;
   iob_type *iob;
   mfd_entry_type *mfd_entry;

   read_archive_data(&archive_fhb, sizeof(archive_fhb), FALSE);
   memset(&archive_sentinel, 0, sizeof(archive_sentinel));
   archive_sentinel.signature = ARCHIVE_SIGNATURE;
   if (archive_fhb.fileHeaderPageNum == 0) {	/* Old-style EOD signature */
      archive_sentinel.record_type = EOD;
      archive_sentinel.checksum = checksum(&archive_sentinel);
      iob = allocate_iob(0);
   } else if (archive_fhb.fileHeaderPageNum == 2) {
      fetch_mfd((archive_fhb.endOfFileLfa + PAGE_SIZE - 1) / PAGE_SIZE);
      target.size = 0;
      return;
   } else {					/* Another compressed FHB... */
      archive_sentinel.record_type = BOF;
      archive_sentinel.pages_data = (archive_fhb.endOfFileLfa + PAGE_SIZE - 1)
                                     / PAGE_SIZE;
      memcpy(archive_sentinel.directory, archive_fhb.dirName,
             archive_fhb.dirName[0] + 1);
      if ((mfd_entry = find_mfd_entry(archive_sentinel.directory)) != NULL) {
         memcpy(archive_sentinel.directory, mfd_entry->sbDirectory,
                mfd_entry->sbDirectory[0] + 1);
         memcpy(archive_sentinel.directory_password, mfd_entry->sbPassword,
                mfd_entry->sbPassword[0] + 1);
         archive_sentinel.directory_protection = mfd_entry->defaultAccessCode;
         archive_sentinel.directory_pages = mfd_entry->cPages;
      }
      memcpy(archive_sentinel.filename, archive_fhb.fileName,
             archive_fhb.fileName[0] + 1);
      memcpy(archive_sentinel.file_password, archive_fhb.password,
             archive_fhb.password[0] + 1);
      archive_sentinel.directory_type = archive_fhb.ObjectType;
      archive_sentinel.protection = archive_fhb.accessProtection;
      archive_sentinel.control.suppress_backup = archive_fhb.fNoSave;
      archive_sentinel.control.hidden_file = archive_fhb.fNoDirPrint;
      archive_sentinel.control.prevent_delete = archive_fhb.fNoDelete;
      archive_sentinel.control.read_only = archive_fhb.fReadOnly;
      archive_sentinel.ms_dos_magic = archive_fhb.bDosMagic;
      archive_sentinel.ms_dos_attribute = archive_fhb.bDosAttribute;
      archive_sentinel.object_type = archive_fhb.userObjectType;
      archive_sentinel.creation_date = archive_fhb.creationDT;
      archive_sentinel.modification_date = archive_fhb.modificationDT;
      archive_sentinel.access_date = archive_fhb.accessDT;
      archive_sentinel.expiration_date = archive_fhb.expirationDT;
      archive_sentinel.eof_lfa = archive_fhb.endOfFileLfa;
      memcpy(archive_sentinel.distrix_info, archive_fhb.rgbDistrix,
             sizeof(archive_fhb.rgbDistrix));
      memcpy(archive_sentinel.application_info, archive_fhb.application,
             sizeof(archive_fhb.application));
      archive_sentinel.checksum = checksum(&archive_sentinel);
      iob = allocate_iob(sizeof(archive_sentinel));
      target.size = archive_sentinel.pages_data * PAGE_SIZE;
      memcpy(iob->data, &archive_sentinel, sizeof(archive_sentinel));
   }
   iob->ctrl = archive_sentinel.record_type;
   if ((erc = Send(target.msg_exch, iob)) != ercOK)
      exit_with_msg(erc, 0);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 MFD.Sys needs to be fetched from an old-style archive dataset (so that
 Restore Archive can use it to fabricate new-style archive sentinels from the
 FHB information in combination with the MFD information. */

static void fetch_mfd(unsigned mfd_pages) {

   archive_sentinel_type *archive_sentinel;
   unsigned i, page;
   iob_type *iob;
   static unsigned prev_mfd_pages = 0;

   release_mfd(&prev_mfd_pages);		/* Free memory of prior use */
   prev_mfd_pages = mfd_pages;
   if (AllocMemorySL(sizeof(mfd_entry) * mfd_pages
                      * ((PAGE_SIZE - 1) / sizeof(mfd_entry_type)),
                     &mfd_entry) != ercOK)
      return;
   if (AllocMemorySL(mfd_pages * PAGE_SIZE, &mfd) != ercOK) {
      release_mfd(&prev_mfd_pages);
      return;
   }
   for (page = 0; page < mfd_pages; page++) {
      read_archive_data(mfd[page], sizeof(mfd[page]), FALSE);
      i = 1;				/* Start just past overflow flag */
      while (i < PAGE_SIZE && mfd[page][i] != 0) {
         mfd_entry[++n_mfd_entry - 1] = (mfd_entry_type *) &mfd[page][i];
         i += sizeof(mfd_entry_type);
      }
   }
   quicksort((char **) mfd_entry, n_mfd_entry);
   for (i = 0; i < n_mfd_entry; i++) {
      iob = allocate_iob(sizeof(archive_sentinel_type));
      iob->ctrl = BOF;
      iob->available = sizeof(archive_sentinel_type);
      archive_sentinel = iob->data;
      memset(archive_sentinel, 0, sizeof(*archive_sentinel));
      archive_sentinel->signature = ARCHIVE_SIGNATURE;
      archive_sentinel->record_type = BOF;
      memcpy(archive_sentinel->directory, mfd_entry[i]->sbDirectory,
             mfd_entry[i]->sbDirectory[0] + 1);
      memcpy(archive_sentinel->directory_password, mfd_entry[i]->sbPassword,
             mfd_entry[i]->sbPassword[0] + 1);
      archive_sentinel->directory_protection = mfd_entry[i]->defaultAccessCode;
      archive_sentinel->directory_pages = mfd_entry[i]->cPages;
      archive_sentinel->checksum = checksum(archive_sentinel);
      Send(target.msg_exch, iob);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 A previously obtained MFD is no longer needed.  Return the memory in the same
 order that it was obtained. */

static void release_mfd(unsigned *mfd_pages) {

   if (mfd != NULL) {
      DeallocMemorySL(mfd, *mfd_pages * PAGE_SIZE);
      mfd = NULL;
   }
   if (mfd_entry != NULL) {
      DeallocMemorySL(mfd_entry, sizeof(mfd_entry) * *mfd_pages
                      * ((PAGE_SIZE - 1) / sizeof(mfd_entry_type)));
      mfd_entry = NULL;
   }
   *mfd_pages = 0;
   n_mfd_entry = 0;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Given a directory name, see if it is described in the MFD.Sys currently in
 memory (it is the caller's responsibility to use the fetch_mfd() function at
 appropriate times to ensure that the MFD data in memory corresponds to the
 volume containing the directory).  Note that the directory search is made
 shorter by using the already sorted list of directory names (these names are
 in fact the start of a directory entry).  If the directory entry can be found,
 return a pointer to the entry.  Otherwise, return a null pointer. */

static mfd_entry_type *find_mfd_entry(char directory[]) {

   unsigned i, j;
   static mfd_entry_type *prev_mfd_entry = NULL;

   if (     prev_mfd_entry != NULL
         && directory[0] == prev_mfd_entry->sbDirectory[0]
         && NlsULCmpB(NULL, &directory[1], &prev_mfd_entry->sbDirectory[1],
                      directory[0], &j) == ercOK
         && j == 0xFFFF)
      return(prev_mfd_entry);
   for (i = 0; i < n_mfd_entry; i++)
      if (     directory[0] == mfd_entry[i]->sbDirectory[0]
            && NlsULCmpB(NULL, &directory[1], &mfd_entry[i]->sbDirectory[1],
                         directory[0], &j) == ercOK
            && j == 0xFFFF)
         return(prev_mfd_entry = mfd_entry[i]);
   return(NULL);

}
pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void read_archive_data(void *buffer, unsigned buffer_len,
                               flag is_sentinal) {

   unsigned erc, i, j, xfer_actual, xfer_size;
   unsigned long residual;

   while (buffer_len > 0) {
      if (archive.sequential) {
         if (archive.old_format)
            erc = read_archive_vam_record(buffer, xfer_size = buffer_len,
                                          &residual);
         else
            erc = SeqAccessRead(archive.handle, buffer,
                                xfer_size = _min(buffer_len,
                                                 archive.piecemeal_size),
                                &residual);
         xfer_actual = xfer_size - residual;
      } else if (archive.old_format) {
         erc = read_archive_rsam_record(buffer, xfer_size = buffer_len,
                                        &residual);
         xfer_actual = xfer_size - residual;
      } else
         erc = Read(archive.handle, buffer, buffer_len, archive.lfa,
                    &xfer_actual);
      offset_of(buffer) += xfer_actual;
      buffer_len -= xfer_actual;
      archive.lfa += xfer_actual;
      if (erc != ercOK)
         if (archive.sequential)
            if (erc == ercTapeFileMark) {
/*             close_archive_seq_dataset(TRUE, FALSE);		m21... */
               close_archive_seq_dataset(TRUE, 
                            ((archive.device_num_last > 1)/* m22 was > 0 */
                      && (archive.device_num < (MAX_NUMBER_OF_DEVICES - 1)) ));
               if ( (archive.device_num_last > 1)		/* m22 was > 0 */	
                  && (archive.device_num < (MAX_NUMBER_OF_DEVICES - 1)) ) {
                  archive.device_num++;
                  open_sequential_device(FALSE);
               }									/* ...m21 */
               open_archive_seq_dataset();
            } else
               exit_with_msg(erc, 0);
         else if (erc == ercEOM) 					/* m18 ... */
            if (is_sentinal) 
               exit_with_msg(ercCorruptTape, 0);
            else /* ...m18 */	{
               close_archive_dataset();
               open_archive_input_dataset();
         } else
            exit_with_msg(erc, 0);
   }
   if (vlpb.file_trailer) {
      j = buffer_len / 2;
      for (i = 0; i < j; i++)
         archive_sentinel.file_checksum += ((unsigned *) buffer)[i];
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private unsigned read_archive_vam_record(void *record, unsigned record_len,
                                         unsigned long *record_residual) {

   unsigned erc, status, xfer_count, xfer_size;
   unsigned long residual;
   vam_block_hdr_type *vam_block_hdr;
   vam_record_hdr_type *vam_record_hdr;
   flag dont_fabricate_record = FALSE;				/* m24 */

   *record_residual = record_len;
   do {
      while (vam.block_residual < sizeof(vam_record_hdr_type)) {
         if ((erc = vam.deferred_erc) != ercOK) {
            vam.deferred_erc = ercOK;
            return(erc);		/* We've caught up with reality! */
         }
         vam.block_data = vam.block_buffer;
         vam.block_residual = vam.block_size;
         while (   vam.block_residual != 0
                && (erc = SeqAccessRead(archive.handle, vam.block_data,
                                        xfer_size = _min(vam.block_residual,
                                                       archive.piecemeal_size),
                                        &residual)) == ercOK) {
            vam.block_residual -= xfer_size;
            offset_of(vam.block_data) += xfer_size;
         }
         if (     (erc == ercTapeBlank || erc == ercTapeIoError)
               && SeqAccessStatus(archive.handle, &status, sizeof(status),
                                  &residual) != ercTapeStatusUnavailable
               && (status & SEQ_EOM))
            erc = ercTapeFileMark;
         if (vam.block_residual != 0) {	/* 12.0 format wrote whole blocks! */
            vam.block_residual = 0;	/* Insure reentry into while loop... */
            return((erc == ercOK) ? ercCorruptTape : erc);
         }
         vam.block_data = vam.block_buffer;
         vam.block_residual = vam.block_size;
         vam.deferred_erc = erc;	/* IF there's an error, save it */
         erc = ercOK;		/* Everything's apples, for the moment */
         vam_block_hdr = vam.block_data;
         if (     archive.first_volume
               && vam_block_hdr->signature == OLD_TAPE_MAGIC) {
            vam.block_sequence = vam_block_hdr->block_sequence + 1;
            vam.record_residual = 0;
            offset_of(vam.block_data) += sizeof(vam_block_hdr_type)
                                          + vam_block_hdr->record_residual;
            vam.block_residual -= sizeof(vam_block_hdr_type)
                                   + vam_block_hdr->record_residual;
            vam_record_hdr = vam.block_data;
            vam.file_num = vam_record_hdr->file_num;
            vam.record_sequence = vam_record_hdr->sequence - 1;
            archive.first_volume = FALSE;
         } else if (vam_block_hdr->signature != OLD_TAPE_MAGIC)
            return(ercCorruptTape);
         else if (vam_block_hdr->block_sequence < vam.block_sequence)
            vam.block_residual = 0;		/* Force read of next block */
         else if (   vam_block_hdr->block_sequence != vam.block_sequence++
                  || vam_block_hdr->record_residual != vam.record_residual)
            return(ercCorruptTape);
         else {
            offset_of(vam.block_data) += sizeof(vam_block_hdr_type);
            vam.block_residual -= sizeof(vam_block_hdr_type);
         }
      }
      if (vam.record_residual == 0) {	/* Nothing left of the old record? */
         vam_record_hdr = vam.block_data;
         if (     vam_record_hdr->size != 0
               && vam_record_hdr->size != PAGE_SIZE
               && vam_record_hdr->size != VAM_FHB_SIZE)
            return(ercCorruptTape);	/* Lost in the woods now, give up! */
         if (vam_record_hdr->sequence == 0) {
            vam.file_num = vam_record_hdr->file_num;
            if (vam.record_sequence == 0xFFFF) {	/* m24 */
               dont_fabricate_record = TRUE;		/* m24 */
            }										/* m24 */
            vam.record_sequence = 0;
         } else if (   vam_record_hdr->file_num != vam.file_num
                    || vam_record_hdr->sequence < ++vam.record_sequence)
            return(ercCorruptTape);
         if (vam_record_hdr->sequence == vam.record_sequence) {
            vam.record_residual = vam_record_hdr->size;
            offset_of(vam.block_data) += sizeof(vam_record_hdr_type);
            vam.block_residual -= sizeof(vam_record_hdr_type);
            vam.skip_record = (   record_len == VAM_FHB_SIZE
                               && vam.record_residual != VAM_FHB_SIZE
                               && vam.record_residual != 0);
         }
      }
      vam.fabricate_record = (   vam.record_residual == 0
                              || (   vam.record_sequence == 0
                                  && record_len != vam.record_residual
                                  && !dont_fabricate_record));	/* m24 */
      dont_fabricate_record = FALSE;							/* m24 */
      if (vam.fabricate_record) {
         xfer_count = _min(record_len, PAGE_SIZE);
         memset(record, 0, xfer_count);
         offset_of(record) += xfer_count;
         record_len -= xfer_count;
         *record_residual -= xfer_count;
      } else {
         xfer_count = _min(vam.record_residual, vam.block_residual);
         if (!vam.skip_record) {
            memcpy(record, vam.block_data, xfer_count);
            offset_of(record) += xfer_count;
            record_len -= xfer_count;
            *record_residual -= xfer_count;
         }
         offset_of(vam.block_data) += xfer_count;
         vam.block_residual -= xfer_count;
         vam.record_residual -= xfer_count;
      }
   } while (record_len != 0);
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private unsigned read_archive_rsam_record(void *record, unsigned record_len,
                                          unsigned long *record_residual) {

   unsigned erc, xfer_count;
   static struct {
      Boolean skip_record;
      Boolean fabricate_record;
      unsigned file_num;
      unsigned sequence;
      unsigned residual;
      struct {
         unsigned file_num;
         unsigned sequence;
         char data[PAGE_SIZE];
      } record;
   } rsam = {0, 0, 0, 0, 0};

   *record_residual = record_len;
   do {
      if (rsam.residual == 0) {
         if ((erc = ReadRsRecord(archive.rswa, &rsam.record,
                                 sizeof(rsam.record),
                                 &rsam.residual)) != ercOK)
            return((erc == ercEOF) ? ercEOM : erc);
         if (rsam.residual < sizeof(rsam.record) - sizeof(rsam.record.data))
            return(ercInconsistency);
         rsam.residual -= sizeof(rsam.record) - sizeof(rsam.record.data);
         if (     rsam.residual != PAGE_SIZE
               && rsam.residual != RSAM_FHB_SIZE
               && rsam.residual != 0)
            return(ercInconsistency);
         if (archive.first_volume) {
            rsam.file_num = rsam.record.file_num;
            rsam.sequence = rsam.record.sequence - 1;
            archive.first_volume = FALSE;
         }
      }
      if (rsam.record.sequence == 0) {
         rsam.file_num = rsam.record.file_num;
         rsam.sequence = 0;
      } else if (   rsam.record.file_num != rsam.file_num
                 || rsam.record.sequence < ++rsam.sequence)
         return(ercInconsistency);
      if (rsam.record.sequence == rsam.sequence) {
         rsam.skip_record = (   record_len == RSAM_FHB_SIZE
                             && rsam.residual != RSAM_FHB_SIZE
                             && rsam.residual != 0);
      }
      rsam.fabricate_record = (   rsam.residual == 0
                               || (   rsam.sequence == 0
                                   && record_len != rsam.residual));
      if (rsam.fabricate_record) {
         xfer_count = _min(record_len, PAGE_SIZE);
         memset(record, 0, xfer_count);
         offset_of(record) += xfer_count;
         record_len -= xfer_count;
         *record_residual -= xfer_count;
      } else {
         xfer_count = _min(rsam.residual, record_len);
         if (!rsam.skip_record) {
            memcpy(record, rsam.record.data, rsam.residual);
            offset_of(record) += rsam.residual;
            record_len -= rsam.residual;
            *record_residual -= rsam.residual;
         }
         rsam.residual = 0;
      }
   } while (record_len != 0);
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The archive I/O process is driven by the exchange of Iob's with the main
 (either backup or restore) process.  The functions below provide for Iob's to
 be allocated and returned as needed.  Each Iob has an associated buffer; for
 the sake of efficient memory use, these are segregated into two classes,
 small (PAGE_SIZE or 512 bytes) and large (generally 64 Kb less one page).
 When the data buffers are not in use, they are maintained on free lists that
 are separate from the free list for the Iob's. */

iob_type *allocate_iob(unsigned buffer_size) {

   iob_type *iob = NULL;

   Wait(iob_exch, &iob);
   memset(iob, 0, sizeof(iob_type));
   iob->return_exch = iob_exch;
   if (buffer_size == PAGE_SIZE) {
      Wait(page_buffer_exch, &iob->base);
      iob->size = PAGE_SIZE;
   } else {
      Wait(data_buffer_exch, &iob->base);
      iob->size = data_buffer_size;
   }
   iob->data = iob->base;
   if (archive.inbound)
      iob->available = iob->size;
   return(iob);

}

void deallocate_iob(iob_type *iob) {

   Send((iob->size == PAGE_SIZE) ? page_buffer_exch : data_buffer_exch,
        iob->base);
   Send(iob_exch, iob);

}

/*-----------------------------------------------------------------------------
 A pair of procedures to either calculate or verify the checksum for a page of
 data (usually this is an archive sentinel record). */

private unsigned checksum(void *data) {

   unsigned checksum = -ARCHIVE_SIGNATURE, i;

   for (i = 0; i < PAGE_SIZE / 2; i++)
      checksum -= ((unsigned *) data)[i];
   return(checksum);

}

private Boolean checksum_valid(void *data) {

   unsigned checksum = ARCHIVE_SIGNATURE, i;

   for (i = 0; i < PAGE_SIZE / 2; i++)
      checksum += ((unsigned *) data)[i];
   return(checksum == 0);

}
