/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service_srp_qic.c                                                    */
/*                                                                           */
/*  SRP/XE QIC-02 device procedures for CTOS Sequential Access service.      */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/17/91  121J.06  P. Johansson  /  Don't allow buffered mode to change  */
/*                                      in SeqAccessModeSet.                 */
/*  04/23/91  121J.05  P. Johansson  /  Don't write a filemark if the count  */
/*                                      is zero; WRITE DATA must be reissued */
/*                                      as a new command subsequent to EXCP. */
/*  03/27/91  121H.04  P. Johansson  /  Fixes to checkpoint logic at EOM;    */
/*                                      Fix hang on write at end of tape.    */
/*  02/13/91  121G.03  C.G. Naik     /  READ STATUS not to be saved in Dcb.  */
/*  01/04/91  121F.02  P. Johansson  /  New procedure to return default      */
/*                                      device buffer configuration details. */
/*  12/20/90  121E.01  P. Johansson  /  Corrections to 'map_qic_status()'    */
/*  12/13/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocLdtSlot
#define BuildLdtSlot
#define Delay
#define FProtectedMode
#define Request
#define Respond
#define Send
#define SetTimerInt
#define ShortDelay
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define SrpQIC
#define tpib_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "srp_qic.h"
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void disable(void);
extern void enable(void);
extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern char input(unsigned io_address);
extern unsigned inword(unsigned io_address);
extern void output(unsigned io_address, char value);
extern void outword(unsigned io_address, unsigned value);

/* Error return codes used by this module */

#define ercQicException 9082

#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned own_user_num;
extern unsigned serv_exch;

/* Static variables global within this manuscript */

private Boolean protected_mode;
private dcb_qic_type *dcb;
private unsigned qic_address[] = {MISC_COMMAND_ADDRESS, MISC_COMMAND_ADDRESS,
                                  WRITE_DATA_ADDRESS, WRITE_FILEMARK_ADDRESS,
                                  READ_DATA_ADDRESS, READ_FILEMARK_ADDRESS,
                                  READ_STATUS_ADDRESS, QIC_FLAGS_ADDRESS,
                                  WRITE_BUFFER_ADDRESS, READ_BUFFER_ADDRESS,
                                  QIC_STATUS_ADDRESS, OFFLINE_ADDRESS,
                                  REQ_ADDRESS, RESET_QIC_ADDRESS};

private qic_iob_type *qic_iob;
private unsigned tpib_interval[] = {10, 600, 60, 10, 60, 400, 10};

/* Function prototypes defined before the functions themselves are declared */

void assert_qic_REQ(void);
unsigned issue_qic_command(char command);
unsigned map_qic_status(void);
void poll_qic_interrupt(void);
unsigned qic_ready(void);
char read_qic_flags(void);
unsigned read_qic_status(void);
void reestablish_remote_slot(void);				/* DEBUG */
unsigned reset_qic_firmware(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 The QIC interface board on the SRP is a memory-mapped I/O device.  Establish
 pointers to the different registers in the QIC address space.  Note that a
 different pointer is established in protected mode for each unique QIC
 address (LDT slots are conserved when the same QIC address is remapped) while
 in real mode all the pointers have the same value.  The redundancy of the
 real mode approach was chosen to maximize code sharing between protected and
 real modes once the service is installed.  Also, check to see if the QIC
 interface board is present at the address specified by the installation
 call.  If it's not, we might as well exit now rather than disappoint the user
 later. */

void srp_qic_init(dcb_qic_type *uninitialized_dcb, char qic_slot) {

   unsigned config, erc, nmi_source;
   signed i, j;

   dcb = uninitialized_dcb;	/* No reentrancy worries, one per customer! */
   protected_mode = FProtectedMode();
   qic_iob = (qic_iob_type *) &dcb->own_rq;
   dcb->qic_slot = qic_slot;
   if (protected_mode) {
      dcb->remote_slot = REMOTE_SLOT_REG_386;
      dcb->br0 = BASE_REG0_386;
      for (i = 0; i <= last(qic_address); i ++) {
         for (j = 0; j <= i - 1; j++)
            if (qic_address[i] == qic_address[j]) {
               dcb->qic_memory_map[i] = dcb->qic_memory_map[j];
               break;
            }
         if (dcb->qic_memory_map[i] == NULL) {
            if (     (erc = AllocLdtSlot(&selector_of(dcb->qic_memory_map[i])))
                  != ercOK)
               exit_with_msg(erc, 0);
            if ((erc = BuildLdtSlot(selector_of(dcb->qic_memory_map[i]),
                                    (unsigned long ) (  0xE000
                                                      | qic_address[i]) << 16,
                                    ((   qic_address[i] == WRITE_BUFFER_ADDRESS
                                      || qic_address[i] == READ_BUFFER_ADDRESS)
                                     ? PAGE_SIZE : sizeof(char)) - 1,
                                    0x0092)) != ercOK)
               exit_with_msg(erc, 0);
         }
      }
      output(REMOTE_SLOT_REG_386, qic_slot);
      Delay(1);
      config = inword(CONFIG_REG_386);
      Delay(1);
      disable();
      outword(CONFIG_REG_386, config & ~NMI_ENABLE_386);
      read_qic_flags();
      nmi_source = inword(NMI_REG_386);
      outword(CONFIG_REG_386, config);
      enable();
      if ((~nmi_source & (  NONEXISTENT_MEMORY_386 | BUS_TIMEOUT_386
                          | BUS_ERROR_386)) != 0)
         exit_with_msg(ercNoTapeHardware, 0);
   } else {
      dcb->remote_slot = REMOTE_SLOT_REG;
      dcb->br0 = BASE_REG0;
      for (i = 0; i <= last(dcb->qic_memory_map); i++)
         selector_of(dcb->qic_memory_map[i]) = REMOTE_ADDRESS_SELECTOR;
      output(REMOTE_SLOT_REG, qic_slot);
      disable();
      output(NMI_REG, 0);
      read_qic_flags();
      nmi_source = inword(NMI_REG);
      output(NMI_REG, NMI_ENABLE);
      enable();
      if (~nmi_source & (NONEXISTENT_MEMORY | BUS_TIMEOUT | BUS_ERROR))
         exit_with_msg(ercNoTapeHardware, 0);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function provides device specific information about buffer capacity so
 that SeqAccessOpen may allocate buffers intelligently. */

unsigned srp_qic_defaults(default_info_type *default_info) {

   default_info->device_buffer_size = 1024;
   default_info->recommended_buffers = 8;
   default_info->recommended_buffer_size = 4096;
   default_info->recommended_buffer_pool_size = 32768;
   default_info->recommended_write_buffer_full_threshold = 1024;
   default_info->recommended_read_buffer_empty_threshold = 1024;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Parts of the Dcb are zeroed each time a device is closed, so recreate the
 "request block" in the Dcb that is used to communicate interrupt
 information.  Afterwards, reset the firmware in the QIC interface board and
 return the status that the tape drive is showing. */

unsigned srp_qic_open(dcb_qic_type *dcb) {

   void *data_segment = &dcb;

   dcb->own_rq.s_cnt_info = 4;
   dcb->own_rq.exch_resp = serv_exch;
   dcb->own_rq.user_num = own_user_num;
   dcb->min_record_size = dcb->max_record_size = dcb->block_size = PAGE_SIZE;
   dcb->tpib.interrupt_process = (void (*)(void)) poll_qic_interrupt;
   dcb->tpib.data_segment = selector_of(data_segment);
   dcb->tpib.rq = &dcb->tpib_rq;
   output(dcb->remote_slot, dcb->qic_slot);
   reset_qic_firmware();
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 There is nothing controllable or user configurable about the QIC-02 tape
 backup module.  However, the generic Sequential Access service is expecting
 reasonable answers to be provided in the mode parameters block.  It arrives
 zero filled, which simplifies the job here tremendously! */

unsigned srp_qic_mode_query(dcb_qic_type *dcb,
                             seq_parameters_type *seq_parameters) {

   if (read_qic_status() == ercOK)
      seq_parameters->write_protected = dcb->unit_status.write_protected;
   seq_parameters->density = 0x05;		/* QIC-24 */
   seq_parameters->block_size =			/* Not selectable, 512 bytes */
      seq_parameters->min_record_size = 
      seq_parameters->max_record_size =  PAGE_SIZE;
   seq_parameters->disable_error_correction = TRUE;
   seq_parameters->read_retry_limit =		/* Built into firmware */
      seq_parameters->write_retry_limit = 16;
   seq_parameters->data_buffer_nonrecoverable = TRUE;
   return(ercOK);

}

unsigned srp_qic_mode_set(dcb_qic_type *dcb,
                           seq_parameters_type *seq_parameters) {

   dcb->operating_mode.variable_length = FALSE;
   dcb->operating_mode.buffer_recoverable = FALSE;
   dcb->operating_mode.data_compression = FALSE;
   dcb->block_size = dcb->min_record_size =  dcb->max_record_size =  PAGE_SIZE;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For each of the different SeqAccessControl functions, select the appropriate
 QIC-02 command and establish a timeout limit (in tenths of seconds) long
 enough for the operation to complete.  Note that record spacing has to be
 implemented by QIC-02 read data operations---in this case, the fact that
 'data_transfer' is not set in the Dcb state indicates that data read is to be
 discarded, not transferred to a (nonexistent!) user buffer. */

unsigned srp_qic_ctrl(dcb_qic_type *dcb) {

   char command;
   seq_access_rq_type *rq = dcb->rq;

   switch (dcb->ctrl_function = rq->ctrl_function) {
      case CTRL_REWIND:
      case CTRL_UNLOAD:
         command = REWIND;
         dcb->timeout = 100;
         break;

      case CTRL_RETENSION:
         command = RETENSION;
         dcb->timeout = 200;
         break;

      case CTRL_ERASE_MEDIUM:
         command = ERASE;
         dcb->timeout = 200;
         break;

      case CTRL_WRITE_FILEMARK:
         if (rq->ctrl_qualifier < 0)
            return(ercTapeCommandIllegal);
         command = WRITE_FILEMARK;
         if ((dcb->ctrl_qualifier = rq->ctrl_qualifier) == 0)
            return(ercOK);
         dcb->timeout = 60;
         break;

      case CTRL_SCAN_FILEMARK:
         if (rq->ctrl_qualifier != 1)
            return(ercTapeCommandIllegal);
         command = READ_FILEMARK;
         dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 900;
         break;

      case CTRL_ERASE_GAP:
         return(ercOK);

      default:
         return(ercTapeCommandIllegal);
   }
   return(issue_qic_command(command));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The QIC-02 interface board always maintains current status which is not
 cleared by a reference to it (unlike sense data for SCSI devices).  Thus it
 is always safe to read status information from the QICI board before mapping
 it to "generic" device status. */

unsigned srp_qic_status(dcb_qic_type *dcb) {

   unsigned erc;

   erc = (dcb->state.status_available) ? map_qic_status() : read_qic_status();
   dcb->state.pending_status = dcb->state.status_available = FALSE;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 QIC-02 read and write operations on the SRP are very simple because no DMA
 set-up is involved.  When buffers are empty (for writes), data is copied from
 the Dcb device buffer to the QICI board and the transfer continues
 automatically.  For read operations, data is copied from the QICI board to
 the Dcb device buffer whenever at least one full buffer is available.  Here,
 just establish a generous sixty second time limit and call the command
 initiation procedure.  If the previous command was already a data transfer
 command, a timer pseudo-interrupt is request to await a buffer attention
 interrupt from the QICI.  Otherwise the read or write command is issued to
 the QICI and the timer interrupt is set. */

unsigned srp_qic_io(dcb_qic_type *dcb) {

   dcb->ctrl_function = 0xFFFF;	/* Make sure it's not a real function */
   dcb->timeout = 60;
   return(issue_qic_command((dcb->state.inbound) ? READ_DATA : WRITE_DATA));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure issues QIC-02 commands to the QICI board on the SRP.  Some
 notable points are:  a) the QICI inverts ALL QIC-02 signals, so commands
 written (as well as status read by another procedure) must be one's
 complemented before they are sent,  b) once a READ or WRITE command is
 underway, subsequent READ or WRITE commands do not have to be issued to keep
 the tape streaming---the QICI monitors the buffer empty/full status,
 c) along the same lines, it is not necessary to examine READY when a data
 transfer command is in process---just look at the buffer full bits, and
 d) buffer attention interrupts are provided each time a buffer empties or
 fills (even though no additional READ or WRITE commands were necessary).
 This procedure allows the caller to specify wait for completion, otherwise a
 timer pseudo-interrupt counter is started to watch for the INTERRUPT bit in
 the QIC flags. */

private unsigned issue_qic_command(char command) {

   unsigned erc;

   dcb->state.status_available = FALSE;	/* Status cleared by new command */
   dcb->io_xfer_actual = dcb->io_xfer_count = 0;
   qic_iob->erc_ret = REQUEST_IN_PROGRESS;
   if (command == WRITE_DATA)
      while (   dcb->device_buffer->available - dcb->io_xfer_count != 0
             &&    (read_qic_flags() & QIC_BUFFERS_FULL) != QIC_BUFFERS_FULL) {
         outword(dcb->br0, (protected_mode) ? 0 : WRITE_BUFFER_ADDRESS);
         memcpy(dcb->qic_write_buffer_reg,
                (char *) dcb->device_buffer->data + dcb->io_xfer_count,
                PAGE_SIZE);
         dcb->io_xfer_count += PAGE_SIZE;
      }
   if (     command != dcb->command
         || ((command != READ_DATA) && (command != WRITE_DATA))) {
      while ((erc = qic_ready()) != ercOK)
         if (     erc != ercQicException
               || (   (erc = read_qic_status()) != ercTapeEomWarning
                   && erc != ercOK))
            return(erc);
      outword(dcb->br0, (protected_mode) ? 0 : qic_address[command >> 5]);
      *dcb->qic_command_reg[command >> 5] = ~(dcb->command = command);
   }
   dcb->tpib.intervals = tpib_interval[command >> 5];
   poll_qic_interrupt();
   return(REQUEST_IN_PROGRESS);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The SRP QICI board is not capable of interrupting the processor boards, so we
 must poll it periodically for the interrupt bit (note that INTERRUPT is
 always asserted if EXCEPTION is asserted).  The frequency of the timer pseudo-
 interrupt process below is adjusted for each QIC-02 command type to obtain
 good throughput without excessive consumption of processor cycles.  The
 maximum time allowed for command completion is also tailored to the operation
 (e.g. retensioning operations complete in a matter of minutes, not
 milliseconds).  In the normal course of events, detect the interrupt and send
 a message to the background process so the interrupt may be serviced. */

void poll_qic_interrupt(void) {

   output(dcb->remote_slot, dcb->qic_slot);
   if ((qic_iob->qic_flags = read_qic_flags()) & QIC_INTERRUPT) {
      qic_iob->erc_ret = ercOK;
      Send(qic_iob->exch_resp, qic_iob);
      return;
   }
   SetTimerInt(&dcb->tpib);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 QIC-02 interrupts (detected by the timer pseudo-interrupt polling routine)
 are serviced at process level by this procedure.  A QIC "Iob" has arrived at
 the service exchange and this Iob signals that interrupt processing is
 required.  If we are in a data transfer state, copy data out of the QICI
 buffers (read, only) and update the count of bytes moved to or from the
 device buffer on the Dcb queue.  The only other processing needed is to check
 for exception conditions and obtain QIC-02 status as needed.  Note that on an
 exception condition, not all of the data previously transferred to the QICI
 buffers has necessarily been transferred to the medium---check the two buffer
 full flags to determine how much, if any, has actually been transferred. */

unsigned srp_qic_rq_done(dcb_qic_type *dcb) {

   unsigned erc = qic_iob->erc_ret;
   char garbage_buffer[PAGE_SIZE];
   char qic_flags;

   if (dcb->state.data_transfer) {
      if (dcb->command == READ_DATA) {
         while (   dcb->device_buffer->available - dcb->io_xfer_count != 0
                && (read_qic_flags() & QIC_BUFFERS_FULL)) {
            outword(dcb->br0, (protected_mode) ? 0 : READ_BUFFER_ADDRESS);
            memcpy((char *) dcb->device_buffer->data + dcb->io_xfer_count,
                   dcb->qic_read_buffer_reg, PAGE_SIZE);
            dcb->io_xfer_count += PAGE_SIZE;
         }
      }
      dcb->io_xfer_actual = dcb->io_xfer_count;
   } else if (dcb->command == READ_DATA)	/* This is space records... */
      while (read_qic_flags() & QIC_BUFFERS_FULL) {
         outword(dcb->br0, (protected_mode) ? 0 : READ_BUFFER_ADDRESS);
         memcpy(garbage_buffer, dcb->qic_read_buffer_reg, PAGE_SIZE);
      }
   if (     ((qic_flags = read_qic_flags()) & QIC_EXCEPTION)
         && (!dcb->state.inbound || (qic_flags & QIC_BUFFERS_FULL) == 0)) {
      if (dcb->state.data_transfer && !dcb->state.inbound)
         switch (qic_flags & QIC_BUFFERS_FULL) {
            case QIC_BUFFER_A_FULL:
            case QIC_BUFFER_B_FULL:
               if (dcb->io_xfer_actual > PAGE_SIZE)
                  dcb->io_xfer_actual -= PAGE_SIZE;
               else
                  dcb->io_xfer_actual = 0;
               break;

            case QIC_BUFFERS_FULL:
               if (dcb->io_xfer_actual > 2 * PAGE_SIZE)
                  dcb->io_xfer_actual -= 2 * PAGE_SIZE;
               else
                  dcb->io_xfer_actual = 0;
               break;
         }
      dcb->state.status_available = TRUE;
      dcb->state.pending_status = ((erc = read_qic_status()) != ercOK);
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Synthesize the six QIC-02 status bytes into the virtual status information
 maintained in the Dcb.  After that is done, see if an error return code has
 to be generated unique to the status condition. */

private unsigned map_qic_status(void) {

   unsigned erc = ercOK;

   dcb->soft_errors += ((dcb->qic_status[2] << 8) + dcb->qic_status[3]);
   dcb->underruns += ((dcb->qic_status[4] << 8) + dcb->qic_status[5]);
   dcb->unit_status.reset = ((dcb->qic_status[1] & POWER_UP_RESET) != 0);
   dcb->unit_status.busy = ((dcb->qic_ctrl_lines & QIC_READY) == 0);
   dcb->unit_status.ready = ((dcb->qic_status[0] & NO_CARTRIDGE) == 0);
   dcb->unit_status.write_protected =
         (dcb->unit_status.ready && (dcb->qic_status[0] & WRITE_PROTECT) != 0);
   dcb->unit_status.on_line = ((dcb->qic_status[0] & NOT_ONLINE) == 0);
   dcb->position.beginning_of_medium = ((dcb->qic_status[1] & BOM) != 0);
   dcb->position.end_of_data = ((dcb->qic_status[1] & EOD) != 0);
   dcb->position.end_of_medium = ((dcb->qic_status[0] & EOM) != 0);
   dcb->command_status.file_mark = ((dcb->qic_status[0] & FILEMARK) != 0);
   dcb->command_status.illegal_operation =
                                 ((dcb->qic_status[1] & ILLEGAL_COMMAND) != 0);
   dcb->command_status.recovered_error =
                                    (dcb->qic_status[1] & EIGHT_RETRIES)
                                 && !(dcb->qic_status[0] & UNRECOVERABLE_DATA);
   dcb->command_status.hard_error =
                              ((dcb->qic_status[0] & UNRECOVERABLE_DATA) != 0);
   if (!dcb->unit_status.on_line)
      erc = ercTapeHardwareError;
   else if (dcb->unit_status.reset)
      erc = ercTapeReset;
   else if (dcb->command_status.illegal_operation)
      erc = ercTapeCommandIllegal;
   else if (!dcb->unit_status.ready)
      erc = ercTapeNotReady;
   else if (dcb->qic_status[1] & NO_DATA || dcb->position.end_of_data)
      erc = ercTapeBlank;
   else if (dcb->command_status.file_mark)
      erc = (dcb->ctrl_function == CTRL_SCAN_FILEMARK) ? ercOK
                                                       : ercTapeFileMark;
   else if (dcb->command_status.hard_error)
      erc = ercTapeIoError;
   else if (dcb->position.end_of_medium && !dcb->state.ignore_EOM)
      erc = ercTapeEomWarning;
   else if (dcb->unit_status.write_protected && !dcb->state.inbound)
      erc = ercTapeWriteProtected;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 A collection of small procedures (or is it a small collection of procedures?)
 to perform memory-mapped I/O for the SRP QICI board.  Most of these could
 have been two generic routines, one for reading and one for writing, but this
 is more optimal. */

private void assert_qic_REQ(void) {

   char garbage;

   reestablish_remote_slot();					/* DEBUG */
   outword(dcb->br0, (protected_mode) ? 0 : REQ_ADDRESS);
   garbage = *dcb->qic_REQ_reg;

}

private unsigned qic_ready(void) {

   char qic_flags;
   unsigned i, j;

   reestablish_remote_slot();					/* DEBUG */
   outword(dcb->br0, (protected_mode) ? 0 : QIC_FLAGS_ADDRESS);
   for (i = 1; i <= 10; i++) {
      for (j = 0; j != 0xFFFF; j++) {
         qic_flags = *dcb->qic_flags_reg	/* Invert active low signals */
                                 ^ (QIC_INTERRUPT | QIC_EXCEPTION | QIC_READY);
         if (qic_flags & QIC_EXCEPTION)
            return(ercQicException);
         else if (qic_flags & QIC_READY)
            return(ercOK);
      }
      Delay(1);
   }
   return(ercTapeNotReady);

}

private char read_qic_flags(void) {

   reestablish_remote_slot();					/* DEBUG */
   outword(dcb->br0, (protected_mode) ? 0 : QIC_FLAGS_ADDRESS);
   return(  *dcb->qic_flags_reg	/* Invert the active low signals... */
          ^ (QIC_INTERRUPT | QIC_EXCEPTION | QIC_READY));

}

private unsigned read_qic_status(void) {

   unsigned i;

   reestablish_remote_slot();					/* DEBUG */
   if (qic_ready() == ercTapeNotReady)	/* Need READY or EXCP to read status */
      return(ercTapeNotReady);
   outword(dcb->br0, (protected_mode) ? 0 : READ_STATUS_ADDRESS);
   *dcb->qic_command_reg[READ_STATUS >> 5] = ~(dcb->command = READ_STATUS);
   for (i = 1; i <= 60; i++) {
      Delay(1);			/* Wait circa one minute for interrupt... */
      if (read_qic_flags() & QIC_INTERRUPT) {
         for (i = 0; i <= last(dcb->qic_status); i++) {
            if (qic_ready() != ercOK)
               return(ercTapeStatusUnavailable);
            outword(dcb->br0, (protected_mode) ? 0 : QIC_STATUS_ADDRESS);
            dcb->qic_status[i] = ~(*dcb->qic_status_reg);
            assert_qic_REQ();
         }
         dcb->state.status_available = TRUE;
         return(map_qic_status());
      }
   }
   return(ercTapeTimeout);

}

private unsigned reset_qic_firmware(void) {

   unsigned i;

   reestablish_remote_slot();					/* DEBUG */
   outword(dcb->br0, (protected_mode) ? 0 : RESET_QIC_ADDRESS);
   *dcb->qic_reset_reg = (dcb->command = RESET_QIC_FIRMWARE);
   for (i = 1; i <= 60; i++) {
      Delay(1);			/* Wait circa one minute for interrupt... */
      if (read_qic_flags() & QIC_INTERRUPT)
         read_qic_status();
         ShortDelay(10);
         return(read_qic_status());
   }
   return(ercTapeTimeout);

}

unsigned erroneous_slot = 0;					/* DEBUG */
								/* DEBUG */
static void reestablish_remote_slot(void) {			/* DEBUG */
								/* DEBUG */
   if (input(dcb->remote_slot) != dcb->qic_slot) {		/* DEBUG */
      erroneous_slot++;						/* DEBUG */
      output(dcb->remote_slot, dcb->qic_slot);			/* DEBUG */
   }								/* DEBUG */
								/* DEBUG */
}								/* DEBUG */
