/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service_flem_qic.c                                                   */
/*                                                                           */
/*  Workstation QIC-02 device procedures for CTOS Sequential Access service. */
/*                                                                           */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/17/91  121J.09  P. Johansson  /  Don't allow buffered mode to change  */
/*                                      in SeqAccessModeSet.                 */
/*  05/10/91  121J.08  C.G.Naik      /  Read status twice to work around the */
/*                                      firmware bug on TS-25 Viper drive.   */
/*  04/22/91  121J.07  P. Johansson  /  Don't write a filemark if the count  */
/*                                      is zero; WRITE DATA must be reissued */
/*                                      as a new command subsequent to EXCP. */
/*  04/08/91  121J.06  P. Johansson  /  More TS-25 Viper timing issues---    */
/*                                      wait for READY in 'flem_qic_ctrl()'. */
/*  03/27/91  121H.05  P. Johansson  /  Fixes to checkpoint logic at EOM;    */
/*                                      timing differences for Viper drive.  */
/*  02/25/91  121G.04  P. Johansson  /  Modify to follow San Jose module     */
/*                                      practices wherever applicable.       */
/*  02/01/91  121G.03  K. Keeney     /  Add watchdog for tape timeouts.      */
/*  01/29/91  121G.02  K. Keeney     /  Modified to work.                    */
/*  01/04/91  121F.01  P. Johansson  /  New procedure to return default      */
/*                                      device buffer configuration details. */
/*  12/17/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define CloseRTClock
#define Delay
#define GetModuleId
#define MapBusAddress
#define OpenRTClock
#define PaFromP
#define QueryRequestInfo
#define Request
#define Send
#define SetTimerInt
#define ShortDelay
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define FlemingtonQIC
#define SysConfigType
#define tpib_type

#define last(array) (sizeof(array) / sizeof(*array) - 1)

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "flem_qic.h"
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void disable(void);
extern void enable(void);
extern char input(unsigned io_address);
extern unsigned inword(unsigned io_address);
extern void output(unsigned io_address, char value);
extern void outword(unsigned io_address, unsigned value);
extern void FlemQicISR(void);

/* Error return codes used by this module */

#define RqErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned default_resp_exch;
extern unsigned internal_os_version;
extern unsigned own_user_num;
extern unsigned qic_base_addr;
extern char *qic_control;
extern qic_iob_type *qic_iob;
extern unsigned serv_exch;
extern SysConfigType *sysConfig;

/* Static variables global within this manuscript */

private dcb_qic_type *dcb;
private watchdog_type watchdog = {0, 10, 0, 0, 0, 0xFFFF};
private unsigned tpib_interval[] = {15, 1000, 0, 20, 0, 15, 0};

/* Function prototypes defined before the functions themselves are declared */

unsigned issue_qic_command(char command);
unsigned map_qic_status(void);
void poll_qic_ready(void);
unsigned qic_ready(void);
unsigned read_qic_status(void);
unsigned reset_qic_firmware(void);
unsigned reset_xbus_misr(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 The initialization procedure is called once when the Sequential Access
 service is installed.  The HB-001 hardware does not need any particular
 initialization to eliminate, for example, unwanted interrupts from floating
 I/O signals, so this code doesn't have to program the QIC-02 interface.
 However, the way in which the Flemington module performs DMA is not typical
 for San Jose devices, so extensive set up of the DMA address locations for
 each different processor type is required.  NB: Because only one Flemington
 QIC module is permitted to be attached to the X-Bus at a time, there are no
 reentrancy worries with the use of a static variable ('dcb') to make
 references to the QIC Dcb faster throughout the rest of this manuscript. */

void flem_qic_init(dcb_type *uninitialized_dcb, unsigned base_address) {

   unsigned i, module_id;

   dcb = (dcb_qic_type *) (watchdog.dcb = uninitialized_dcb);
   watchdog.exch_resp = serv_exch;
   qic_base_addr = base_address;		/* Used by assembler ISR... */
   qic_control = &dcb->qic_control;		/* ... */
   qic_iob = (qic_iob_type *) &dcb->own_rq;	/* ...for faster references */
   for (i = 0; i <= last(dcb->qic_reg_addr); i++)
      dcb->qic_reg_addr[i] = (base_address | (2 * i));
   if (sysConfig->HardwareType == B27) {	/* A very strange bird... */
      dcb->mask_dma = 0x04;
      dcb->dma_mode = 0xC0;			/* Cascade mode */
      dcb->dma_xfer_limit = 127 * PAGE_SIZE;	/* 64 Kb less one page */
      dcb->int_vector = 13;
      dcb->dma_count_reg = 0xF802;
      dcb->dma_command_reg = 0xF810;
      dcb->dma_mask_reg = 0xF814;
      dcb->dma_mode_reg = 0xF816;
      dcb->dma_clear_flip_flop = 0xF818;
      if (inword(0xFF2A) < 0x0006)		/* Unmask INT1 interrupts */
         outword(0xFF2A, 0x0006);
      outword(0xFF28, inword(0xFF28) & ~0x20);
   } else {
      dcb->mask_dma = 0x05;			/* All others the same */
      dcb->unmask_dma = 0x01;
      dcb->dma_mode = 0x01;			/* Demand verify mode */
      dcb->dma_xfer_limit = 64 * PAGE_SIZE;	/* 32 Kb */
      GetModuleId(XBUS, 1, &module_id);		/* Tailor to hardware */
      switch (module_id) {
         case CP_001:				/* Also known as a B26 */
            dcb->int_vector = 13;
            dcb->dma_count_reg = 0xF806;
            dcb->dma_command_reg = 0xF810;
            dcb->dma_mask_reg = 0xF814;
            dcb->dma_mode_reg = 0xF816;
            dcb->dma_clear_flip_flop = 0xF818;
            if (inword(0xFF2A) < 0x0006)	/* Unmask INT1 interrupts */
               outword(0xFF2A, 0x0006);
            outword(0xFF28, inword(0xFF28) & ~0x20);
            break;

         case 0x0500:				/* Who knows what this is? */
            dcb->int_vector = 13;
            dcb->dma_count_reg = 0xC006;
            dcb->dma_command_reg = 0xC010;
            dcb->dma_mask_reg = 0xC014;
            dcb->dma_mode_reg = 0xC016;
            dcb->dma_clear_flip_flop = 0xC018;
            break;

         case CP_002:				/* Also known as a B28 */
         case CP_003:				/* Also known as a B38 */
            dcb->int_vector = 82;
            dcb->dma_count_reg = 0xF806;
            dcb->dma_command_reg = 0xF810;
            dcb->dma_mask_reg = 0xF814;
            dcb->dma_mode_reg = 0xF816;
            dcb->dma_clear_flip_flop = 0xF818;
            output(0xFC12, input(0xFC12) & ~0x80);	/* Enable master... */
            output(0xFC02, input(0xFC02) & ~0x04);	/* ...cascade XINTR1 */
            break;

         case SERIES_286I:
         case SERIES_386I:			/* Also known as a B39 */
            dcb->int_vector = 82;
            dcb->slave_dma = TRUE;		/* Must program master DMA */
            dcb->dma_count_reg = 0xF806;
            dcb->dma_command_reg = 0xF810;
            dcb->dma_mask_reg = 0xF814;
            dcb->dma_mode_reg = 0xF816;
            dcb->dma_clear_flip_flop = 0xF818;
            dcb->dma_master_command_reg = 0xF828;
            dcb->dma_master_mask_reg = 0xF82A;
            dcb->dma_master_mode_reg = 0xF83A;
            break;
      }
   }

}

/*-----------------------------------------------------------------------------
 This function provides device specific information about buffer capacity so
 that SeqAccessOpen may allocate buffers intelligently. */

unsigned flem_qic_defaults(default_info_type *default_info) {

   default_info->device_buffer_size = 2048;
   default_info->recommended_buffers = 4;
   default_info->recommended_buffer_size = 8192;
   default_info->recommended_buffer_pool_size = 32768;
   default_info->recommended_write_buffer_full_threshold = 2048;
   default_info->recommended_read_buffer_empty_threshold = 2048;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Information that is in the "erasable" part of the Dcb must be reestablished
 each time the device is opened.  In addition, a real-time clock for the gross
 timing of QIC operations must be established, the X-Bus interrupt mediator
 must be notified that we wish to receive interrupts and the HB-001 interface
 firmware must be reset. */

unsigned flem_qic_open(dcb_qic_type *dcb) {

   unsigned erc, rq_info = 0;
   void *data_segment = &qic_base_addr, *response_rq;
   set_xbus_misr_rq_type set_xbus_misr_rq;

   dcb->own_rq.s_cnt_info = 4;
   dcb->own_rq.exch_resp = serv_exch;
   dcb->own_rq.user_num = own_user_num;
   dcb->min_record_size = dcb->max_record_size = dcb->block_size = PAGE_SIZE;
   dcb->tpib.interrupt_process = (void (*)(void)) poll_qic_ready;
   dcb->tpib.data_segment = selector_of(data_segment);
   dcb->tpib.rq = &dcb->tpib_rq;
   if (watchdog.user_count++ == 0) {
      watchdog.counter = watchdog.counter_reload;
      watchdog.events = 0;
      if ((erc = OpenRTClock(&watchdog)) != ercOK) {
         watchdog.user_count--;
         return(erc);
      }
   }
   memset(&set_xbus_misr_rq, 0, sizeof(set_xbus_misr_rq));
   set_xbus_misr_rq.s_cnt_info = 6;
   set_xbus_misr_rq.n_resp_pb_cb = 1;
   set_xbus_misr_rq.exch_resp = default_resp_exch;
   set_xbus_misr_rq.erc_ret = 0xFFFF;
   if (QueryRequestInfo(set_xbus_misr_rq.rq_code = SET_XBUS_MISR_RQ_CODE,
                        &rq_info, sizeof(rq_info)) != ercOK || rq_info == 0)
      set_xbus_misr_rq.rq_code = OLD_SET_XBUS_MISR_RQ_CODE;
   set_xbus_misr_rq.interrupt_service = (void *) FlemQicISR;
   set_xbus_misr_rq.data_segment = selector_of(data_segment);
   set_xbus_misr_rq.interrupt_handle_ret = &dcb->xbus_handle;
   set_xbus_misr_rq.size_interrupt_handle_ret = sizeof(dcb->xbus_handle);
   if ((erc = Request(&set_xbus_misr_rq)) == ercOK) {
      while (   (erc = Wait(default_resp_exch, &response_rq)) == ercOK
             && &set_xbus_misr_rq != response_rq)
         ;
      if (erc == ercOK)
         erc = set_xbus_misr_rq.erc_ret;
   }
   if (erc == ercOK)
      reset_qic_firmware();
   return((erc == ercServiceNotAvail) ? ercXbifNotInstalled : erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Deassert any control signals to the module, close the real-time clock and
 release the shared X-Bus interrupt. */

unsigned flem_qic_close(dcb_qic_type *dcb) {

   unsigned erc;

   output(dcb->qic_control_reg, dcb->qic_control = 0);
   if (watchdog.user_count == 0)
      erc = reset_xbus_misr();
   else if (   --watchdog.user_count == 0
            && (erc = CloseRTClock(&watchdog)) != ercOK)
      reset_xbus_misr();
   else
      erc = reset_xbus_misr();
   return(erc);

}

private unsigned reset_xbus_misr(void) {

   unsigned erc, rq_info = 0;
   void *response_rq;
   reset_xbus_misr_rq_type reset_xbus_misr_rq;

   memset(&reset_xbus_misr_rq, 0, sizeof(reset_xbus_misr_rq));
   reset_xbus_misr_rq.s_cnt_info = 2;
   reset_xbus_misr_rq.exch_resp = default_resp_exch;
   reset_xbus_misr_rq.erc_ret = 0xFFFF;
   if (QueryRequestInfo(reset_xbus_misr_rq.rq_code = RESET_XBUS_MISR_RQ_CODE,
                        &rq_info, sizeof(rq_info)) != ercOK || rq_info == 0)
      reset_xbus_misr_rq.rq_code = OLD_RESET_XBUS_MISR_RQ_CODE;
   reset_xbus_misr_rq.interrupt_handle = dcb->xbus_handle;
   if ((erc = Request(&reset_xbus_misr_rq)) == ercOK)
      while (   (erc = Wait(default_resp_exch, &response_rq)) == ercOK
             && &reset_xbus_misr_rq != response_rq)
         ;
   if (erc == ercOK)
      erc = reset_xbus_misr_rq.erc_ret;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 There is nothing controllable or user configurable about the QIC-02 tape
 backup module.  However, the generic Sequential Access service is expecting
 reasonable answers to be provided in the mode parameters block.  It arrives
 zero filled, which simplifies the job here tremendously! */

unsigned flem_qic_mode_query(dcb_qic_type *dcb,
                             seq_parameters_type *seq_parameters) {

   if (read_qic_status() == ercOK)
      seq_parameters->write_protected = dcb->unit_status.write_protected;
   seq_parameters->density = 0x05;		/* QIC-24 */
   seq_parameters->block_size =			/* Not selectable, 512 bytes */
      seq_parameters->min_record_size = 
      seq_parameters->max_record_size =  PAGE_SIZE;
   seq_parameters->disable_error_correction = TRUE;
   seq_parameters->read_retry_limit =		/* Built into firmware */
      seq_parameters->write_retry_limit = 16;
   seq_parameters->data_buffer_nonrecoverable = TRUE;
   return(ercOK);

}

unsigned flem_qic_mode_set(dcb_qic_type *dcb,
                           seq_parameters_type *seq_parameters) {

   dcb->operating_mode.variable_length = FALSE;
   dcb->operating_mode.buffer_recoverable = FALSE;
   dcb->operating_mode.data_compression = FALSE;
   dcb->block_size = dcb->min_record_size =  dcb->max_record_size =  PAGE_SIZE;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For each of the different SeqAccessControl functions, select the appropriate
 QIC-02 command and establish a timeout limit (in tenths of seconds) long
 enough for the operation to complete.  Note that record spacing has to be
 implemented by QIC-02 read data operations---in this case, the fact that
 'data_transfer' is not set in the Dcb state indicates that data read is to be
 discarded, not transferred to a (nonexistent!) user buffer. */

unsigned flem_qic_ctrl(dcb_qic_type *dcb) {

   char command;
   unsigned erc;
   seq_access_rq_type *rq;

   rq = dcb->rq;
   switch (dcb->ctrl_function = rq->ctrl_function) {
      case CTRL_REWIND:
      case CTRL_UNLOAD:
         command = REWIND;
         dcb->timeout = 100;
         break;

      case CTRL_RETENSION:
         command = RETENSION;
         dcb->timeout = 200;
         break;

      case CTRL_ERASE_MEDIUM:
         command = ERASE;
         dcb->timeout = 200;
         break;

      case CTRL_WRITE_FILEMARK:
         if (rq->ctrl_qualifier < 0)
            return(ercTapeCommandIllegal);
         command = WRITE_FILEMARK;
         if ((dcb->ctrl_qualifier = rq->ctrl_qualifier) == 0)
            return(ercOK);
         dcb->timeout = 60;
         break;

      case CTRL_SCAN_FILEMARK:
         if (rq->ctrl_qualifier != 1)
            return(ercTapeCommandIllegal);
         command = READ_FILEMARK;
         dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 900;
         break;

      case CTRL_ERASE_GAP:
         return(ercOK);

      default:
         return(ercTapeCommandIllegal);
   }
   if ((erc = qic_ready()) != ercOK)
      return((erc == ercQicException) ? read_qic_status() : erc);
   return(issue_qic_command(command));

}

/*-----------------------------------------------------------------------------
 If the six QIC-02 status bytes are already available because of an earlier
 QIC EXCEPTION condition, it is sufficient to map them into an error return
 code and set the device independent status information in the Dcb.
 Otherwise, issue a READ STATUS command and map its results.  Note that some
 of the drives in the Flemington modules (not sure whether they're WangTek or
 Ciprico drives) raise EXCP if a READ STATUS is attempted while the DMA state
 machine is still active (e.g. after a valid read from a tape which is
 suspended because no additional buffers have been supplied).  In this case
 just read the status a second time. */

unsigned flem_qic_status(dcb_qic_type *dcb) {

   unsigned erc;

   if (dcb->state.status_available)
      erc = map_qic_status();
   else {
      ShortDelay(20);		/* This for Viper in a TS-25 module */
      read_qic_status();	/* Viper in TS-25 needs this done twice! */
      erc = read_qic_status();
   }
   dcb->state.pending_status = dcb->state.status_available = FALSE;
   return(erc);
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Data transfer operations for the Flemington HB-001 module require that the
 DMA circuitry be programmed.  The DMA transfer is controlled by a 24-bit bus
 address and an 8-bit block count.  Additionally, the processor's DMA channels
 have to be appropriately unmasked and the correct modes established---for
 both the master and the cascaded DMA controller as necessary. */

unsigned flem_qic_io(dcb_qic_type *dcb) {

   unsigned long bus_address;
   unsigned erc, garbage;

   if (      dcb->device_buffer->bus_address == 0
          && (erc = (internal_os_version >= 0x0C00)
                     ? MapBusAddress(dcb->io_data = dcb->device_buffer->data,
                                     dcb->device_buffer->size, NO_DMA_FENCE,
                                     &dcb->device_buffer->bus_address,
                                     &garbage)
                     : PaFromP(dcb->io_data = dcb->device_buffer->data, 0,
                               &dcb->device_buffer->bus_address)) != ercOK)
      return(erc);
   dcb->io_xfer_count = _min(dcb->device_buffer->available,
                             dcb->dma_xfer_limit);
   dcb->ctrl_function = 0xFFFF;	/* Make sure it's not a real function */
   dcb->timeout = 60;
   disable();
   outword(dcb->dma_mask_reg, dcb->mask_dma);
   if (dcb->slave_dma) {
      outword(dcb->dma_master_mask_reg, 0x04);
      outword(dcb->dma_master_mode_reg, 0xC0);
      outword(dcb->dma_master_command_reg, 0);
   }
   outword(dcb->dma_clear_flip_flop, 0);
   outword(dcb->dma_mode_reg, dcb->dma_mode);
   outword(dcb->dma_count_reg, 0x00FF);
   outword(dcb->dma_count_reg, 0x00FF);
   outword(dcb->dma_command_reg, 0);
   outword(dcb->dma_mask_reg, dcb->unmask_dma);
   if (dcb->slave_dma)
      outword(dcb->dma_master_mask_reg, 0);
   enable();
   bus_address = dcb->device_buffer->bus_address;
   output(dcb->qic_dma_addr_low_reg, bus_address);
   output(dcb->qic_dma_addr_mid_reg, bus_address >> 8);
   output(dcb->qic_dma_addr_high_reg, bus_address >> 16);
   output(dcb->qic_block_count_reg, ~(dcb->io_xfer_count / PAGE_SIZE));
   if (dcb->state.inbound) {
      dcb->qic_control |= QIC_DMA_WRITE;
      return(issue_qic_command(READ_DATA));
   } else {
      dcb->qic_control &= ~QIC_DMA_WRITE;
      return(issue_qic_command(WRITE_DATA));
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Most of the time, the READY signal is asserted very quickly by the QIC-02
 interface.  Occasionally, though, when a lengthy error recovery or retry
 process is underway in the hardware it may take MUCH longer for READY to be
 present.  The maximum limit of 40 seconds for the secondary wait was arrived
 at empirically with a few error-laden QIC tapes. */

private unsigned qic_ready(void) {

   char qic_ctrl_lines;
   unsigned status_reg, i;

   status_reg = dcb->qic_status_reg;
   for (i = 1; i <= 5000; i++)
      if ((qic_ctrl_lines = input(status_reg)) & QIC_EXCP)
         return(ercQicException);
      else if (qic_ctrl_lines & QIC_READY)
         return(ercOK);
   for (i = 1; i <= 400; i++) {
      Delay(1);
      if ((qic_ctrl_lines = input(status_reg)) & QIC_EXCP)
         return(ercQicException);
      else if (qic_ctrl_lines & QIC_READY)
         return(ercOK);
   }
   return(ercTapeNotReady);

}

private unsigned qic_not_ready(void) {

   char qic_ctrl_lines;
   unsigned status_reg, i;

   status_reg = dcb->qic_status_reg;
   for (i = 1; i <= 5000; i++)
      if ((qic_ctrl_lines = input(status_reg)) & QIC_EXCP)
         return(ercQicException);
      else if ((qic_ctrl_lines & QIC_READY) == 0)
         return(ercOK);
   return(ercTapeHardwareError);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The QIC-02 READ STATUS command is essentially an immediate command because
 its timing is so fast.  Wait in a tight loop until the status bytes are
 available and then read all six.  Subsequently map the status information to
 an error return code and set the generic device status information in the
 Dcb. */

private unsigned read_qic_status(void) {

   unsigned i;

   dcb->state.status_available = FALSE;
   output(dcb->qic_control_reg, dcb->qic_control = QIC_ONLINE);
   ShortDelay(5);
   output(dcb->qic_command_reg, dcb->command = READ_STATUS);
   output(dcb->qic_control_reg, (dcb->qic_control |= QIC_LED) | QIC_REQUEST);
   ShortDelay(5);
   if (qic_ready() != ercOK) {
      output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_LED);
      return(ercTapeStatusUnavailable);
   }
   output(dcb->qic_control_reg, dcb->qic_control);
   ShortDelay(5);
   for (i = 0; i <= last(dcb->qic_status); i++) {
      if (qic_ready() != ercOK) {
         output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_LED);
         return(ercTapeStatusUnavailable);
      }
      dcb->qic_status[i] = input(dcb->qic_data_reg);
      output(dcb->qic_control_reg, dcb->qic_control | QIC_REQUEST);
      if (qic_not_ready() != ercOK) {
         output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_LED);
         return(ercTapeStatusUnavailable);
      }
      output(dcb->qic_control_reg, dcb->qic_control);
   }
   output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_LED);
   dcb->state.status_available = TRUE;
   return(map_qic_status());

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The Flemington HB-001 interface firmware is normally reset when a
 SeqAccessOpen is performed or if a catastrophic error condition (e.g. I/O
 operation timeout) occurs.  After performing the reset (and awaiting the
 interrupt that signals its completion) read the QIC-02 status TWICE to clear
 the power-on reset bit from the status bytes.  At other times, this bit is
 considered abnormal. */

private unsigned reset_qic_firmware(void) {

   unsigned i;

   dcb->command = RESET_QIC_FIRMWARE;
   dcb->state.status_available = FALSE;
   output(dcb->qic_control_reg, 0);	/* Security measure... */
   output(dcb->qic_block_count_reg, 0);	/* Unlatch (previous) interrupt */
   output(dcb->qic_control_reg, (dcb->qic_control = (QIC_LED | QIC_ONLINE))
                                | QIC_RESET);
   ShortDelay(2);	/* Leave RESET asserted for a short time */
   output(dcb->qic_control_reg, dcb->qic_control);
   Delay(25);		/* Allow 2.5 seconds for RESET confidence check */
   for (i = 1; i <= 10; i++)
      if (input(dcb->qic_status_reg) & QIC_EXCP) {
         read_qic_status();	/* Read status once to clear reset bit... */
         ShortDelay(10);
         return(read_qic_status());	/* ...then return real status */
      } else
         Delay(1);		/* Wait up to one more second for EXCP... */
   output(dcb->qic_block_count_reg, 0);	/* Remove (potential) interrupt */
   output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_LED);
   return(ercTapeHardwareError);	/* Bogosity!  EXCP never asserted! */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The QIC-02 interface has the peculiarity that if a data transfer command is
 being issued after the successful completion of an identical data transfer
 command, it is not necessary to present the command and assert REQuest---just
 reenabling DMA and the interrupts suffices to start the command.  Otherwise,
 the command is written to the QIC-02 interface and REQuest is raised to
 advise the firmware to look at the command.  Note that the interrupt service
 routine (in a separate assembler language module) lowers REQuest after a
 COMMAND ACCEPTED interrupt is generated. */

private unsigned issue_qic_command(char command) {

   dcb->state.status_available = FALSE;	/* Status cleared by new command */
   dcb->io_xfer_actual = 0;
   qic_iob->erc_ret = REQUEST_IN_PROGRESS;
   qic_iob->command_accepted = FALSE;
   qic_iob->interrupt = FALSE;
   dcb->tpib.intervals = tpib_interval[command >> 5];
   if (command == READ_DATA || command == WRITE_DATA)
      if (command == dcb->command)	/* Reprise of preceding command? */
         output(dcb->qic_control_reg,	/* Yes, DMA is already enabled! */
                dcb->qic_control |= (QIC_LED | QIC_INT_ENABLE));
      else {
         output(dcb->qic_command_reg, dcb->command = command);
         output(dcb->qic_control_reg,
               (dcb->qic_control |= (QIC_LED | QIC_INT_ENABLE)) | QIC_REQUEST);
      }
   else {
      output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_DMA_ENABLE);
      output(dcb->qic_block_count_reg, 0);	/* Reset TRANSFER COMPLETE */
      output(dcb->qic_command_reg, dcb->command = command);
      output(dcb->qic_control_reg,
             (dcb->qic_control |= (QIC_LED | QIC_INT_ENABLE)) | QIC_REQUEST);
   }
   return(REQUEST_IN_PROGRESS);
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 After the REQuest line has been asserted to let the QIC-02 interface know
 that a command has been presented, a COMMAND ACCEPTED interrupted is expected
 in the normal course of events.  This procedure examines the command that has
 been started in order to either enable the DMA circuitry for data transfer
 commands or start a timer pseudo-interrupt process to watch for command
 completion.  In the first case, the command is expected to eventually
 interrupt again with TRANSFER COMPLETED. */
 
void qic_command_accepted_interrupt(void) {

   if (dcb->command == READ_DATA || dcb->command == WRITE_DATA)
      output(dcb->qic_control_reg, dcb->qic_control |= QIC_DMA_ENABLE);
   else
      SetTimerInt(&dcb->tpib);

}

/*-----------------------------------------------------------------------------
 The Flemington QIC module provides a COMMAND ACCEPTED interrupt, but no
 command completed interrupt.  For the immediate commands, such as READ
 STATUS, this poses no problem since we may perform a (very) brief wait
 in-line after the command is accepted.  For commands that involve data
 transfer, a TRANSFER COMPLETED interrupt is the answer (in addition to the
 grainier, catastrophic error timer in case the command never finishes!).  But
 for commands that involve tape motion but no data transfer, there is no
 choice but to periodically poll the QIC status register to see if READY has
 been asserted.  Once the READY condition is detected, send a message to the
 background process so the device may be serviced. */

void poll_qic_ready(void) {

   char qic_ctrl_lines;

   disable();
   if ( (  (qic_ctrl_lines = input(dcb->qic_status_reg))
         & (TRANSFER_COMPLETE | QIC_READY | QIC_EXCP)) == 0) {
      enable();
      SetTimerInt(&dcb->tpib);
   } else if (  (qic_ctrl_lines & (TRANSFER_COMPLETE | QIC_READY | QIC_EXCP))
            == QIC_READY) {
      output(dcb->qic_control_reg,
             dcb->qic_control &= ~(QIC_LED | QIC_INT_ENABLE));
      enable();
      if (qic_iob->erc_ret == REQUEST_IN_PROGRESS) {
         qic_iob->qic_ctrl_lines = qic_ctrl_lines;
         qic_iob->erc_ret = ercOK;
         Send(qic_iob->exch_resp, qic_iob);
      }
   } else
      enable();

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 If data transfer had been in progress, read the block count register in
 the HB-001 module to determine the transfer residual and calculate the actual
 quantity of data transferred accordingly (this is used by the higher level,
 device independent procedure that called us).  Also, if an exception
 condition occurred, read the QIC-02 status avaialable from the interface. */

unsigned flem_qic_rq_done(dcb_qic_type *dcb) {

   unsigned erc, residual;

   dcb->timeout = 0;			/* OK or in error, it's still done */
   if ((erc = qic_iob->erc_ret) != ercOK) {
      dcb->state.status_available = FALSE;	/* No QIC-02 status... */
      dcb->state.pending_status = TRUE;		/* ..but an error is pending */
   } else {
      if (dcb->state.data_transfer) {	/* Determine actual transfer amount */
         residual = ~inword(dcb->qic_block_count_reg) * PAGE_SIZE;
         dcb->io_xfer_actual = (residual > dcb->io_xfer_count)
                               ? 0 : dcb->io_xfer_count - residual;
      }
      if (qic_iob->qic_ctrl_lines & QIC_EXCP) {	/* Status available? */
         dcb->state.status_available = TRUE;
         dcb->state.pending_status = ((erc = read_qic_status()) != ercOK);
      }
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The watchdog procedure is periodically activated by a timer request block.
 Because the counter reload field is set to 10 and the timer has a granularity
 of 100 milliseconds, the 'events' field holds the number of seconds that have
 elapsed since the last invocation.  This value is used to decrement the
 timeout field in the Dcb, which is also expressed in one second units.
 If time has run out, interrupts are disabled to guard the critical section
 with the interrupt service routine and the QIC-02 interface firmware is
 reset.  After this reset, the timeout error code is inserted into the Iob
 which is then sent to the service exchange for the normal completion
 processing.  While the QIC device is open by a user, the watchdog is always
 running (even if no QIC operations are in progress). */

void flem_qic_watchdog(watchdog_type *watchdog) {

   if (dcb->timeout > watchdog->events)	/* Time still left on the clock? */
      dcb->timeout -= watchdog->events;	/* Yes, just decrement it */
   else if (dcb->timeout != 0) {	/* Is the timer enabled? */
      dcb->timeout = 0;			/* Not any more! */
      disable();			/* Critical section with ISR... */
      if (qic_iob->interrupt)
         enable();			/* Whew! Got in just under the wire */
      else {
         output(dcb->qic_control_reg, dcb->qic_control &=
                                 ~(QIC_LED | QIC_INT_ENABLE | QIC_DMA_ENABLE));
         enable();			/* OK now 'cause interrupts disabled */
         reset_qic_firmware();		/* Reset known state of the world */
         qic_iob->erc_ret = ercTapeTimeout;
         Send(qic_iob->exch_resp, qic_iob);	/* Cause "done" code to run */
      }
   }
   watchdog->events = 0;		/* Ensure a subsequent reactivation */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Synthesize the six QIC-02 status bytes into the virtual status information
 maintained in the Dcb.  After that is done, see if an error return code has
 to be generated unique to the status condition. */

private unsigned map_qic_status(void) {

   unsigned erc = ercOK;

   dcb->soft_errors += ((dcb->qic_status[2] << 8) + dcb->qic_status[3]);
   dcb->underruns += ((dcb->qic_status[4] << 8) + dcb->qic_status[5]);
   dcb->unit_status.reset = ((dcb->qic_status[1] & POWER_UP_RESET) != 0);
   dcb->unit_status.busy = ((dcb->qic_ctrl_lines & QIC_READY) == 0);
   dcb->unit_status.ready = ((dcb->qic_status[0] & NO_CARTRIDGE) == 0);
   dcb->unit_status.write_protected =
         (dcb->unit_status.ready && (dcb->qic_status[0] & WRITE_PROTECT) != 0);
   dcb->unit_status.on_line = ((dcb->qic_status[0] & NOT_ONLINE) == 0);
   dcb->position.beginning_of_medium = ((dcb->qic_status[1] & BOM) != 0);
   dcb->position.end_of_data = ((dcb->qic_status[1] & EOD) != 0);
   dcb->position.end_of_medium = ((dcb->qic_status[0] & EOM) != 0);
   dcb->command_status.file_mark = ((dcb->qic_status[0] & FILEMARK) != 0);
   dcb->command_status.illegal_operation =
                                 ((dcb->qic_status[1] & ILLEGAL_COMMAND) != 0);
   dcb->command_status.recovered_error =
                                    (dcb->qic_status[1] & EIGHT_RETRIES)
                                 && !(dcb->qic_status[0] & UNRECOVERABLE_DATA);
   dcb->command_status.hard_error =
                              ((dcb->qic_status[0] & UNRECOVERABLE_DATA) != 0);
   if (!dcb->unit_status.on_line)
      erc = ercTapeHardwareError;
   else if (dcb->unit_status.reset)
      erc = ercTapeReset;
   else if (dcb->command_status.illegal_operation)
      erc = ercTapeCommandIllegal;
   else if (!dcb->unit_status.ready)
      erc = ercTapeNotReady;
   else if (dcb->qic_status[1] & NO_DATA || dcb->position.end_of_data)
      erc = ercTapeBlank;
   else if (dcb->command_status.file_mark)
      erc = (dcb->ctrl_function == CTRL_SCAN_FILEMARK) ? ercOK
                                                       : ercTapeFileMark;
   else if (dcb->command_status.hard_error)
      erc = ercTapeIoError;
   else if (dcb->position.end_of_medium && !dcb->state.ignore_EOM)
      erc = ercTapeEomWarning;
   else if (dcb->unit_status.write_protected && dcb->mode == MODE_MODIFY)
      erc = ercTapeWriteProtected;
   return(erc);
}
