/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service.c                                                            */
/*                                                                           */
/*  Kernel and principal service loop for CTOS Sequential Access service.    */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/22/91  121J.11  P. Johansson  /  Force buffered mode if request       */
/*                                      blocks are not on local processor.   */
/*  05/20/91  121J.10  P. Johansson  /  Correct errors in variable length    */
/*                                      record support.                      */
/*  05/17/91  121J.09  P. Johansson  /  Unbuffered support returns! Also,    */
/*                                      correct erroneous discrimination     */
/*                                      of watchdog/resume requests.         */
/*  05/10/91  121J.08  P. Johansson  /  For SDV, only, do not support        */
/*                                      unbuffered mode (ercNotImplemented). */
/*  05/08/91  121J.07  P. Johansson  /  Avoid deadlock when only one buffer  */
/*                                      is provided.                         */
/*  04/29/91  121J.06  P. Johansson  /  Suspended requests handled so as to  */
/*                                      accomodate Cipher/Pertec 1/2" tape.  */
/*  04/05/91  121H.05  P. Johansson  /  Check 'cant_deinstall' flag in Dcb.  */
/*  03/27/91  121H.04  P. Johansson  /  Fixes to checkpoint logic at EOM.    */
/*  02/19/91  121G.03  P. Johansson  /  Changes to support Pertec on SRP.    */
/*  02/01/91  121F.02  P. Johansson  /  Support for device-specific watchdog */
/*                                      procedures dispatched generically;   */
/*                                      implement SeqAccessDiscardBufferData.*/
/*  01/04/91  121F.01  P. Johansson  /  Changed 'version_info' structure;    */
/*                                      Don't copy data to buffers on        */
/*                                      SeqAccessWrite if write-protected;   */
/*                                      Use NlsULCmpB in deinstall request.  */
/*  12/17/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <ctype.h>
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define	Check
#define ForwardRequest
#define FSrp
#define GetProcInfo
#define GetPStructure
#define NlsULCmpB
#define ParseFileSpec
#define QueryNodeName
#define Request
#define Respond
#define Send
#define ServeRq
#define SetPartitionLock
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define GetpStructureCase
#define sdType
#define Syscom
#define SysConfigType
#define UCBType

pragma Off(List);
#include <ctosTypes.h>
#include <seq_service.h>
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern buffer_descriptor_type *allocate_buffer(unsigned buffer_size);
extern void copy_client_buffer(dcb_type *dcb);
extern void deallocate_buffer(buffer_descriptor_type *buffer_descriptor);
extern unsigned (*device_buffer[])(dcb_type *dcb);
extern unsigned (*device_close[])(dcb_type *dcb);
extern unsigned (*device_ctrl[])(dcb_type *dcb);
extern unsigned (*device_defaults[])(default_info_type *default_info);
extern unsigned (*device_io[])(dcb_type *dcb);
extern unsigned (*device_mode_query[])(dcb_type *dcb,
                                      seq_parameters_type *seq_parameters);
extern unsigned (*device_mode_set[])(dcb_type *dcb,
                                     seq_parameters_type *seq_parameters);
extern unsigned (*device_open[])(dcb_type *dcb);
extern unsigned (*device_rq_done[])(dcb_type *dcb);
extern unsigned (*device_status[])(dcb_type *dcb);
extern void (*device_watchdog[])(watchdog_type *watchdog);
extern void flush_client_buffer(dcb_type *dcb);
extern void flush_device_buffer(dcb_type *dcb);
extern void log_erc(unsigned erc);
extern void purge_buffers(dcb_type *dcb);
extern void recover_device_buffer(dcb_type *dcb);
extern void update_device_buffer(dcb_type *dcb);

/* Error return codes used by this module */

#define KernelErc
#define FsErc
#define RqErc
#define SvrErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* Request codes used by this module */

pragma Off(List);
#include <rq.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned default_resp_exch;
extern unsigned own_user_num;
extern unsigned save_rq_info[];
extern char sbVerRun[];
extern unsigned serv_exch;
extern unsigned seq_access_rq_code[];
extern unsigned seq_access_rq_code_last;
extern SysConfigType *sysConfig;
extern unsigned term_exch;

/* Global variables exported by this manuscript */

unsigned n_dcb = 0;
dcb_type *dcb_array[MAX_SEQ_DEVICES] = {NULL, NULL, NULL, NULL};

/* Static variables global within this manuscript */

/* Function prototypes defined before the functions themselves are declared */

unsigned find_dcb(seq_access_open_rq_type *rq, unsigned *i);
unsigned long calculate_residual(dcb_type *dcb);
unsigned seq_access_termination(seq_access_term_rq_type *rq);
unsigned seq_access_checkpoint(dcb_type *dcb);
unsigned seq_access_close(dcb_type *dcb);
unsigned seq_access_ctrl(dcb_type *dcb);
unsigned seq_access_deinstall(seq_access_deinstall_rq_type *rq);
unsigned seq_access_discard(dcb_type *dcb);
unsigned seq_access_io(dcb_type *dcb);
unsigned seq_access_mode(dcb_type *dcb);
unsigned seq_access_open(seq_access_open_rq_type *rq, unsigned service_code);
unsigned seq_access_recover(dcb_type *dcb);
dcb_type *seq_access_rq_done(seq_access_rq_type *rq);
unsigned seq_access_status(dcb_type *dcb);
unsigned seq_access_version(seq_access_version_rq_type *rq);
unsigned update_route_table(dcb_type *dcb, char operation);

pragma Page(1);
/*-----------------------------------------------------------------------------
 The main service loop for the Sequential Access service.  Wait at the service
 exchange for either a) new requests arriving from users, b) responses to
 requests issued by the service itself or c) "watchdog" timer request blocks.
 "How do we tell them apart?" you may well ask...it's not foolproof against
 deliberate malice, but for the most part an examination of the request code
 or user number of the request block is sufficient.  For Sequential Access
 requests, dispatch them to the appropriate function according to the local
 service code.  For returning requests that have received their response,
 match them with the appropriate device control block and use the state
 information there to resume whatever processing was underway. */

void seq_access(void) {

   dcb_type *dcb;
   static unsigned **localServiceCode;
   unsigned erc, i, rq_code, service_code;
   seq_access_rq_type *rq;
   seq_access_term_rq_type *term_rq;

   GetPStructure(lGetprgprgLSC, 0, &localServiceCode);
   while (TRUE) {
      if ((erc = Wait(serv_exch, &rq)) != ercOK)
         log_erc(erc);
      else if (rq->rq_code == 0xFFFF) {		/* Real-time clock watchdog? */
         if (     ((watchdog_type *) rq)->user_count != 0
               && (dcb = ((watchdog_type *) rq)->dcb) != NULL
               && device_watchdog[dcb->device_class] != 0)
            device_watchdog[dcb->device_class]((watchdog_type *) rq);
      } else if (rq->user_num == own_user_num) {
         if (rq->rq_code == 0xFFFE) {		/* Resume suspended request? */
            dcb = ((rq_type *) rq)->param[0].pb;
            rq = dcb->rq;
            service_code = dcb->service_code;
            dcb->state.rq_active = FALSE;
            goto resume_rq;
         } else if ((dcb = seq_access_rq_done(rq)) != NULL) {
            if (dcb->state.term_pending) {
               Send(term_exch, NULL);		/* Sentinel token */
               Wait(term_exch, &term_rq);	/* Minor code optimization */
               while (term_rq != NULL) {	/* Examine all terminations */
                  if (     term_rq->user_num == dcb->user_num
                        && !dcb->state.data_transfer) {
                     dcb->state.term_pending = FALSE;	/* This Dcb is idle */
                     if (term_rq->term_erc != ercSwapping)
                        seq_access_close(dcb);	/* Close it if not a swap */
                     if (--term_rq->active_dcbs == 0)
                        Respond(term_rq);	/* Last cow has come home! */
                  } else
                     Send(term_exch, term_rq);
                  Wait(term_exch, &term_rq);
               }
            }
            if (dcb->state.rq_pending) {
               dcb->state.rq_pending = FALSE;
               rq = dcb->rq;
               service_code = dcb->service_code;
               goto resume_rq;
            }
         }
      } else {
         erc = ercNotImplemented;	/* In case of unrecognized request */
         rq_code = rq->rq_code;		/* Optimize references */
         for (i = 0; i <= seq_access_rq_code_last; i++)
            if (rq_code == seq_access_rq_code[i]) {
               service_code =
                            (localServiceCode[rq_code >> 12])[rq_code & 0xFFF];
               if (service_code & VALIDATE_HANDLE) {
                  if (     (rq->seq_handle &= 0x9FFF) == 0
                        || rq->seq_handle > n_dcb
                        || (dcb = dcb_array[rq->seq_handle - 1]) == NULL
                        || dcb->user_num != rq->user_num) {
                     erc = ercInvalidTapeHandle;
                     break;
                  }
                  if (dcb->state.rq_active || dcb->state.rq_pending) {
                     erc = ercTapeOperationOutstanding;
                     break;
                  }
                  dcb->service_code = service_code;
                  dcb->rq = rq;
               }
resume_rq:     switch (service_code >> 8) {
                  case 0:
                     erc = seq_access_version(
                                            (seq_access_version_rq_type *) rq);
                     break;

                  case 1:
                     erc = seq_access_open((seq_access_open_rq_type *) rq,
                                           service_code);
                     break;

                  case 2:
                     erc = seq_access_close(dcb);
                     break;

                  case 3:
                     erc = seq_access_mode(dcb);
                     break;

                  case 4:
                     erc = seq_access_ctrl(dcb);
                     break;

                  case 5:
                     erc = seq_access_status(dcb);
                     break;

                  case 6:
                     erc = seq_access_io(dcb);
                     break;

                  case 7:
                     erc = seq_access_checkpoint(dcb);
                     break;

                  case 8:
                     erc = seq_access_recover(dcb);
                     break;

                  case 9:
                     erc = seq_access_discard(dcb);
                     break;

                  case 10:
                     erc = seq_access_deinstall(
                                          (seq_access_deinstall_rq_type *) rq);
                     break;

                  case 11:
                     erc = seq_access_termination(
                                               (seq_access_term_rq_type *) rq);
                     break;

                  default:
                     erc = ercNotImplemented;
                     break;
               }
               break;	/* Exit request code validation loop...it was found! */
            }
         if (erc == REQUEST_IN_PROGRESS) {
            if (service_code & VALIDATE_HANDLE)
               dcb->state.rq_active = !dcb->state.rq_pending;
         } else  {
            rq->erc_ret = erc;
            Respond(rq);
         }
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 A completed request (or, in the case of QIC-02 and Pertec, a "pseudo-
 request") has arrived at the service exchange and been recognized by the fact
 that its user number matches our own.  If the address of the completed
 request matches one of the private request block areas in the Dcb's, dispatch
 the request to the device specific code for processing.  Once the device
 handler has completed its work, return to perform generic processing of the
 request.  The main service loop that called this function likes to know
 whether or not any waiting requests have a chance of resuming, so return the
 address of the Dcb or NULL according to whether or not we did some useful (and
 potentially liberating) work. */

private dcb_type *seq_access_rq_done(seq_access_rq_type *rq) {

   dcb_type *dcb;
   unsigned erc, i;

   for (i = 0; i <= n_dcb; i++) {
      if (rq == (void *) &dcb_array[i]->own_rq) {
         dcb = dcb_array[i];
         if (     (erc = device_rq_done[dcb->device_class](dcb))
               == REQUEST_IN_PROGRESS)
            return(NULL);
         if (dcb->state.data_transfer) {
            update_device_buffer(dcb);
            if (dcb->device_buffer == NULL && dcb->state.rq_active)
               if (dcb->operating_mode.variable_length) {
                  if (dcb->client_buffer != NULL)
                     copy_client_buffer(dcb);
               } else
                  while (dcb->rq_residual != 0 && dcb->client_buffer != NULL)
                     copy_client_buffer(dcb);
            if (dcb->device_buffer == NULL)
               dcb->state.data_transfer = dcb->state.ignore_EOM = FALSE;
            else if (dcb->state.pending_status || dcb->state.quiet_io) {
               dcb->state.data_transfer = dcb->state.ignore_EOM = FALSE;
               if (dcb->state.inbound || !dcb->operating_mode.buffered)
                  flush_device_buffer(dcb);
            } else if (   (erc = device_io[dcb->device_class](dcb))
                       == REQUEST_IN_PROGRESS)
               erc = ercOK;
            else {
               dcb->state.pending_status = TRUE;
               dcb->state.data_transfer = dcb->state.ignore_EOM = FALSE;
               if (dcb->state.inbound)
                  flush_device_buffer(dcb);
            }
            if (dcb->state.rq_active) {
               if (dcb->operating_mode.variable_length) {
                  if (dcb->client_buffer != NULL)
                     copy_client_buffer(dcb);
               } else
                  while (dcb->rq_residual != 0 && dcb->client_buffer != NULL)
                     copy_client_buffer(dcb);
               if (     dcb->rq_residual == 0
                     || dcb->state.pending_status
                     || (   dcb->operating_mode.variable_length
                         && dcb->rq_residual != dcb->rq->data_len)) {
                  dcb->state.rq_active = FALSE;
                  if (     (*dcb->rq->residual = dcb->rq_residual) == 0
                        && !dcb->operating_mode.variable_length)
                     dcb->rq->erc_ret = ercOK;
                  else {
                     dcb->state.pending_status = FALSE;
                     *dcb->rq->residual += calculate_residual(dcb);
                     dcb->rq->erc_ret = (erc == ercOK && dcb->rq_residual < 0)
                                              ? ercInvalidTapeRecordSize : erc;
                  }
                  Respond(dcb->rq);
               }
            }
         } else {
            if ((dcb->service_code & RESIDUAL) && dcb->rq->residual != NULL)
               *dcb->rq->residual = calculate_residual(dcb);
            dcb->state.rq_active = FALSE;
            dcb->state.pending_status = FALSE;
            dcb->rq->erc_ret = erc;
            Respond(dcb->rq);
         }
         return((dcb->state.data_transfer) ? NULL : dcb);
      }
   }
   return(NULL);		/* Didn't match anything in progress! */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The SeqAccessVersion function provides a number of pieces of handy
 information to the caller.  It is usually invoked before the device is
 opened, since some of the information returned can be used to optimize buffer
 management for the device.  Static information includes the version and
 revision levels of the Sequential Access service as well as a text string
 identifying the version.  Then, device specific procedures are called to
 return information about the size of the buffers contained within the device,
 the recommended overall size of the Sequential Access buffer poll for optimal
 performance with this device and the write buffer full and read buffer empty
 thresholds for the device (these, too, can affect throughput/performance). */

private unsigned seq_access_version(seq_access_version_rq_type *rq) {

   unsigned erc, i;
   version_info_type version_info;

   if ((erc = find_dcb((seq_access_open_rq_type *) rq, &i)) != ercOK)
      return(erc);		/* Device name is not recognized here... */
   if (rq->version_info_ret == NULL || rq->version_info_max == 0)
      return(ercOK);
   memset(&version_info, 0, sizeof(version_info));
   version_info.version = 12;
   version_info.revision = 1;
   version_info.devices = n_dcb;
   memcpy(version_info.version_text, sbVerRun, sbVerRun[0]);
   QueryNodeName(version_info.node, sizeof(version_info.node));
   for (i = 0; i < n_dcb; i++)
      memcpy(version_info.device[i], dcb_array[i]->device_name,
             dcb_array[i]->device_name[0] + 1);
   memcpy(rq->version_info_ret, &version_info,
         _min(rq->version_info_max, sizeof(version_info)));
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The SeqAccessOpen procedure has four principal functions: a) parsing of the
 device name supplied by the user to either forward it immediately to the
 server (explicit [!XXXX] routing), otherwise search for the name in our own
 Dcb tables or, as a last resort, forward the request to the server if the
 name is not found; b) sanity checks of the open mode, some of the mode
 parameters and, of course, whether or not the device is already opened;
 c) buffer allocations (if in buffered mode or if on the SRP and the opener is
 a remote user) and buffer size checks; and d) device-specific initializations
 and sanity checks appropriate when a device is opened (these last by an
 indirect procedure call to the external device-specific code).  If all of
 these tests are passed, a sequence access handle is returned for use in
 subsequent operations.  Note the kluge of the verify bits at the end to make
 cluster routing work --- this code is necessary because "global handle" style
 routing has not yet been implemented in workstation CTOS, only SRP CTOS.
 When global handle routing is made uniform in all CTOS, the code below will
 most likely interact unfavorably with the operating system! */

private unsigned seq_access_open(seq_access_open_rq_type *rq,
                                 unsigned service_code) {

   buffer_descriptor_type *buffer;
   dcb_type *dcb;
   unsigned erc, i, seq_handle;
   seq_parameters_type seq_parameters;
   default_info_type default_info;

   if ((erc = find_dcb(rq, &seq_handle)) != ercOK)
      return(erc);		/* Device name is not recognized here... */
   dcb = dcb_array[seq_handle];
   if (rq->access_mode != MODE_READ && rq->access_mode != MODE_MODIFY)
      return(ercInvalidTapeMode);
   if (dcb->user_num != 0)
      return(ercTapeInUse);
   memset(&seq_parameters, 0, sizeof(seq_parameters));
   if (rq->mode_params != NULL) {
      memcpy(&seq_parameters, rq->mode_params, rq->mode_params_len);
      if (rq->user_num >> 10 != own_user_num >> 10)
         seq_parameters.unbuffered = FALSE;
   }
   if ((erc = device_open[dcb->device_class](dcb)) == REQUEST_IN_PROGRESS) {
      dcb->rq = (seq_access_rq_type *) rq;
      dcb->service_code = service_code;	/* Necessary to be able to try again */
      return(REQUEST_IN_PROGRESS);
   }
   if (erc == ercOK) {
      default_info.dcb = dcb;
      dcb->operating_mode.buffered = !seq_parameters.unbuffered;
      if (     device_mode_set[dcb->device_class] == 0
            || (      (erc = device_mode_set[dcb->device_class](dcb,
                                                              &seq_parameters))
                   == ercOK
                &&    (erc = device_defaults[dcb->device_class](&default_info))
                   == ercOK)) {
         dcb->mode = rq->access_mode;
         dcb->user_num = rq->user_num;
         dcb->operating_mode.variable_length =
                                (dcb->min_record_size != dcb->max_record_size);
         if (     seq_parameters.service_buffers == 0
               || seq_parameters.service_buffer_size == 0) {
            seq_parameters.service_buffers =  default_info.recommended_buffers;
            seq_parameters.service_buffer_size =
                                          default_info.recommended_buffer_size;
         }
         if (     dcb->operating_mode.buffered
               && (   (   dcb->operating_mode.variable_length
                       && seq_parameters.service_buffer_size
                           < dcb->max_record_size)
                   || (   !dcb->operating_mode.variable_length
                       && seq_parameters.service_buffer_size
                           < dcb->block_size)))
            erc = ercInvalidTapeParams;
         else {
            dcb->state.inbound = (dcb->mode == MODE_READ);
            if (dcb->operating_mode.buffered) {
               buffer = (dcb->state.inbound)
                              ? (buffer_descriptor_type *) &dcb->device_buffer
                              : (buffer_descriptor_type *) &dcb->client_buffer;
               for (i = 0; i < seq_parameters.service_buffers; i++)
                  if ((buffer->next =
                           allocate_buffer(seq_parameters.service_buffer_size))
                        != NULL)
                     buffer = buffer->next;
                  else {
                     erc = ercNoTapeBuffer;
                     break;
                  }
            }
         }
      }
   }
   if (erc != ercOK)
      seq_access_close(dcb);
   else if (sysConfig->ClusterConfiguration == 2)		/* Server? */
      *rq->seq_handle_ret = 0x06000 | (seq_handle + 1);
   else					/* No, don't kluge the "verify" code */
      *rq->seq_handle_ret = seq_handle + 1;
   return(erc);

}

/*-----------------------------------------------------------------------------
 This procedure is shared by both SeqAccessVersion and SeqAccessOpen, which
 are given a device name in their requests and have to match it to a Dcb.  Pay
 attention to explicit [!XXX] routing to the server and implicit routing to
 the server if the name is not recognized and we are not the server.  Note that
 implicit routing to the server is performed by the request code itself if a
 Sequential Access service is not installed locally. */

private unsigned find_dcb(seq_access_open_rq_type *rq, unsigned *dcb_index) {

   char device[MAX_DEVICE_LENGTH];
   unsigned device_len, erc, i, j, password_len;
   char password[MAX_PASSWORD_LENGTH];

   if (     rq->device_name == NULL || rq->device_name_len == 0
         || ParseFileSpec(0, rq->device_name, rq->device_name_len, FALSE, NULL,
                          NULL, device, &device_len, NULL, NULL, NULL, NULL,
                          password, &password_len, FALSE, 3) != ercOK)
      return(ercInvalidTapeSpecification);
   if (device_len > 0 && device[0] == '!') {		/* [!XXXX] routing? */
      if (sysConfig->ClusterConfiguration == 1)		/* OK, up the cable! */
         return(((erc = ForwardRequest(CLUSTER_AGENT_SERV_EXCH, rq)) == ercOK)
                ? REQUEST_IN_PROGRESS : erc);
      memcpy(device, &device[1], --device_len);		/* Strip the '!' */
   }
   for (i = 0; i < n_dcb; i++)
      if (     dcb_array[i]->device_name[0] == device_len
            && NlsULCmpB(NULL, &dcb_array[i]->device_name[1],
                         device, device_len, &j) == ercOK
            && j == 0xFFFF) {
         *dcb_index = i;
         return(ercOK);
      }
   if (sysConfig->ClusterConfiguration != 1)	/* Can it be forwarded? */
      return(ercInvalidTapeSpecification);	/* No, the buck stops here */
   return(((erc = ForwardRequest(CLUSTER_AGENT_SERV_EXCH, rq)) == ercOK)
          ? REQUEST_IN_PROGRESS : erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 If buffered data transfer operations are in progress (either look-ahead reads
 or else writes of buffered data), wait for data transfer to quiesce before
 closing the Dcb. */

private unsigned seq_access_close(dcb_type *dcb) {

   unsigned erc = ercOK;
   buffer_descriptor_type *buffer;

   if (dcb->state.data_transfer) {
      dcb->state.quiet_io = (dcb->state.inbound);
      dcb->state.rq_pending = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   while ((buffer = dcb->client_buffer) != NULL) {
      dcb->client_buffer = buffer->next;
      deallocate_buffer(buffer);
   }
   while ((buffer = dcb->device_buffer) != NULL) {
      dcb->device_buffer = buffer->next;
      deallocate_buffer(buffer);
   }
   if (device_close[dcb->device_class] != 0)
      erc = device_close[dcb->device_class](dcb);
   memset(&dcb->erasable_part, 0, _offsetof(dcb_type, device_specific)
          - _offsetof(dcb_type, erasable_part));
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The operational characteristics of sequential access devices may be
 controlled (within the limits of the individual device capabilities) by a set
 of "mode" parameters which may be either queried (SeqAcessModeQuery) or set
 (SeqAccessModeSet and SeqAccessOpen).  Most of the fields in the mode
 parameters have to be interpreted within the context of the particular
 device, so this is left to the device-specific routines that are called.  The
 notable exceptions are buffered vs. unbuffered mode and the characteristics
 of the buffer pool maintained by the Sequential Access service for the
 device.  The former, if FALSE, disables not only the buffered operating mode
 for the device but also eliminates the use of a buffer pool by the service---
 all transfers are made directly to and from the request block data buffers.
 The later, describing the buffer pool may be set only at the time the device
 is opened.  Here it may only be reported, not modified. */

private unsigned seq_access_mode(dcb_type *dcb) {

   buffer_descriptor_type *buffer;
   unsigned erc;
   seq_parameters_type seq_parameters;

   if (dcb->rq->data == NULL || dcb->rq->data_len == 0)
      return(ercOK);
   if (dcb->state.data_transfer) {	/* Data transfer not yet quiescent? */
      dcb->state.rq_pending = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   memset(&seq_parameters, 0, sizeof(seq_parameters));
   if (dcb->service_code & INBOUND) {
      buffer = dcb->client_buffer;	/* OK, sum the buffer size(s) */
      while (buffer != NULL) {
         seq_parameters.service_buffers++;
         seq_parameters.service_buffer_pool_size +=
                           (seq_parameters.service_buffer_size = buffer->size);
         buffer = buffer->next;
      }
      buffer = dcb->device_buffer;
      while (buffer != NULL) {
         seq_parameters.service_buffers++;
         seq_parameters.service_buffer_pool_size +=
                           (seq_parameters.service_buffer_size = buffer->size);
         buffer = buffer->next;
      }
      seq_parameters.write_buffer_threshold = dcb->write_buffer_full_threshold;
      seq_parameters.read_buffer_threshold = dcb->read_buffer_empty_threshold;
      seq_parameters.suppress_default_mode_on_open =
                                            dcb->suppress_default_mode_on_open;
      erc = device_mode_query[dcb->device_class](dcb, &seq_parameters);
      memcpy(dcb->rq->data, &seq_parameters, dcb->rq->data_len);
   } else {
      memcpy(&seq_parameters, dcb->rq->data, dcb->rq->data_len);
      erc = device_mode_set[dcb->device_class](dcb, &seq_parameters);
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Control operations (space, write filemarks, rewind, etc.) are suspended if
 data is awaiting transfer between Sequential Access service buffers and the
 peripheral device.  If no such transfer is pending, the control function is
 allowed to proceed (even though there may still be data in the device's own
 buffers). */

private unsigned seq_access_ctrl(dcb_type *dcb) {

   unsigned erc;

   switch (dcb->rq->ctrl_function) {
      case CTRL_ERASE_MEDIUM:
      case CTRL_WRITE_FILEMARK:
      case CTRL_ERASE_GAP:
         if (dcb->mode == MODE_READ)
            return(ercTapeReadOnly);
         if (dcb->unit_status.write_protected)
            return(ercTapeWriteProtected);
         if (dcb->client_buffer != NULL)
            flush_client_buffer(dcb);
         if (     !dcb->state.data_transfer && !dcb->state.pending_status
               && dcb->device_buffer != NULL)
            dcb->state.data_transfer =
                                  (   (erc = device_io[dcb->device_class](dcb))
                                   == REQUEST_IN_PROGRESS);
         break;
   }
   if (dcb->state.data_transfer) {	/* Data transfer not yet quiescent? */
      dcb->state.rq_pending = dcb->state.ignore_EOM = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   if (     dcb->state.inbound
         && dcb->rq->ctrl_function == CTRL_REWIND)
      dcb->state.pending_status = dcb->state.status_available = FALSE;
   else if (dcb->state.pending_status)	/* Any error conditions present? */
      return(device_status[dcb->device_class](dcb));
   switch (dcb->rq->ctrl_function) {
      case CTRL_REWIND:
      case CTRL_UNLOAD:
      case CTRL_RETENSION:
      case CTRL_SCAN_FILEMARK:
      case CTRL_SPACE_RECORD:
         purge_buffers(dcb);	/* All of these clear any look-aheads */
   }
   return(device_ctrl[dcb->device_class](dcb));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Current status is obtained from the device by this function.  If available
 status exists (because of a previous pending status condition), that status
 is reported without any fresh status inquiries made of the device.  Once
 status is reported, it is marked unavailable and subsequent SeqAccessStatus
 operations always cause a fresh interrogation of the device.  Available
 status is also cleared when any data transfer or medium positioning command
 is initiated for the device.  The status kept in the Dcb is mapped to a
 common format for return to the caller. */

private unsigned seq_access_status(dcb_type *dcb) {

   unsigned erc, synthesized_status = 0;
   seq_access_rq_type *rq;

   rq = dcb->rq;			/* Optimize pointer dereferences */
   if (rq->residual == NULL)
      return(ercInvalidTapeRequest);
   if (rq->size_residual != sizeof(*rq->residual))
      return(ercReturnDataAreaTooSmall);
   if (dcb->state.data_transfer) {	/* Data transfer not yet quiescent? */
      dcb->state.rq_pending = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   if ((erc = device_status[dcb->device_class](dcb)) != REQUEST_IN_PROGRESS) {
      if (dcb->unit_status.reset)
         synthesized_status |= SEQ_RESET;
      if (dcb->unit_status.on_line)
         synthesized_status |= SEQ_ON_LINE;
      if (dcb->position.beginning_of_medium)
         synthesized_status |= SEQ_BOM;
      if (dcb->position.end_of_medium)
         synthesized_status |= SEQ_EOM;
      if (dcb->unit_status.ready)
         synthesized_status |= SEQ_READY;
      if (dcb->unit_status.busy)
         synthesized_status |= SEQ_BUSY;
      if (dcb->unit_status.write_protected)
         synthesized_status |= SEQ_WRITE_PROTECT;
      if (rq->data != NULL) {
         memset(rq->data, 0, rq->data_len);
         memcpy(rq->data, &synthesized_status,
                _min(rq->data_len, sizeof(synthesized_status)));
      }
      *rq->residual = calculate_residual(dcb);
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Data transfer operations SeqAccessRead and SeqAccessWrite are normally
 buffered, that is they receive an 'ercOK' response as soon as the data has
 been transferred between the client data space and the queue of client
 buffers kept by the sequential access service.  If no data transfers are in
 progress, the appropriate function is invoked to commence a data transfer
 with the peripheral.  If error status is pending for the device and all of
 the client data has been transferred, the error status is ignored---a
 subsequent Sequential Access operation will reveal the pending status.
 Otherwise, the status is mapped to an error return code (a subsequent
 SeqAccessStatus operation must be issued by the user to obtain detailed
 status information).. */

private unsigned seq_access_io(dcb_type *dcb) {

   buffer_descriptor_type *buffer;
   unsigned erc = ercOK;
   seq_access_rq_type *rq;

   rq = dcb->rq;			/* Optimize pointer dereferences */
   if (     (   dcb->operating_mode.variable_length
             && (   (rq->data_len != 0 && rq->data_len < dcb->min_record_size)
                 || rq->data_len > dcb->max_record_size))
         || (   !dcb->operating_mode.variable_length
             && (rq->data_len % dcb->min_record_size != 0))
         || rq->residual == NULL)
      return(ercInvalidTapeRequest);
   if (rq->size_residual != sizeof(*rq->residual))
      return(ercReturnDataAreaTooSmall);
   if (offset_of(rq->data) & 0x0001)
      return(ercOddAlignment);		/* Buffer must be word aligned */
   if (!(dcb->service_code & INBOUND) && dcb->unit_status.write_protected)
      return(ercTapeWriteProtected);
   if (dcb->service_code & INBOUND) {
      if (!dcb->state.inbound)
         if (calculate_residual(dcb) == 0) {	/* OK to reverse IFF idle */
            dcb->state.inbound = TRUE;
            while ((buffer = dcb->client_buffer) != NULL) {
               dcb->client_buffer = buffer->next;
               buffer->next = dcb->device_buffer;
               dcb->device_buffer = buffer;
            }
         } else
            return(ercResidualData);
   } else if (dcb->mode == MODE_READ)
      return(ercTapeReadOnly);			/* No writes if 'mr' */
   else if (dcb->state.inbound)
      if (calculate_residual(dcb) == 0) {	/* OK to reverse IFF idle */
         dcb->state.inbound = FALSE;
         while ((buffer = dcb->device_buffer) != NULL) {
            dcb->device_buffer = buffer->next;
            buffer->next = dcb->client_buffer;
            dcb->client_buffer = buffer;
         }
      } else
         return(ercResidualData);
   if (rq->data == NULL || rq->data_len == 0)
      return(ercOK);
   dcb->rq_slot_id = rq->rt_code;
   dcb->rq_data = rq->data;
   dcb->rq_residual = rq->data_len;
   if (dcb->operating_mode.buffered)
      if (dcb->operating_mode.variable_length) {
         if(dcb->client_buffer != NULL)
            copy_client_buffer(dcb);
      } else
         while (dcb->rq_residual != 0 && dcb->client_buffer != NULL)
            copy_client_buffer(dcb);
   else {
      dcb->unbuffered.base = dcb->unbuffered.data = dcb->rq_data;
      dcb->unbuffered.available = dcb->unbuffered.size = dcb->rq_residual;
      dcb->device_buffer = &dcb->unbuffered;
   }
   if (     !dcb->state.data_transfer && !dcb->state.pending_status
         && dcb->device_buffer != NULL)
      dcb->state.data_transfer = (   (erc = device_io[dcb->device_class](dcb))
                                  == REQUEST_IN_PROGRESS);
   if (     dcb->rq_residual == 0
         || (   dcb->operating_mode.variable_length
             && dcb->rq_residual != rq->data_len))
      erc = ercOK;
   else if (dcb->state.pending_status) {
      dcb->rq_residual += calculate_residual(dcb);
      erc = device_status[dcb->device_class](dcb);
   } else if (erc == ercOK)
      erc = REQUEST_IN_PROGRESS;
   *rq->residual = dcb->rq_residual;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 SeqAccessCheckpoint is a NOP for devices opened in read mode (there is only
 look-ahead data in the buffer!).  When writing to the medium, flush any
 partially filled buffers and then wait if necessary for all buffered data to
 be written or an error to be encountered that prevents its transfer to the
 medium.  Once the data transfer state is quiescent, sum the counts of
 unwritten data present in the buffer(s) to determine the residual count
 returned.  If the residual is nonzero and there is no error status (this is
 not expected!), fabricate a non-zero return code. */

private unsigned seq_access_checkpoint(dcb_type *dcb) {

   unsigned erc = ercOK;
   seq_access_rq_type *rq;

   rq = dcb->rq;			/* Optimize pointer dereferences */
   if (rq->residual == NULL)
      return(ercInvalidTapeRequest);
   if (rq->size_residual != sizeof(*rq->residual))
      return(ercReturnDataAreaTooSmall);
   *rq->residual = 0;
   if (dcb->state.inbound)
      return(ercOK);
   if (dcb->client_buffer != NULL)	/* Any partially filled buffers? */
      flush_client_buffer(dcb);
   if (     !dcb->state.data_transfer && !dcb->state.pending_status
         && dcb->device_buffer != NULL)
      dcb->state.data_transfer = (   (erc = device_io[dcb->device_class](dcb))
                                  == REQUEST_IN_PROGRESS);
   if (dcb->state.data_transfer) {	/* Data transfer not yet quiescent? */
      dcb->state.rq_pending = dcb->state.ignore_EOM = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   if (dcb->state.pending_status) {	/* Any error conditions present? */
      *rq->residual = calculate_residual(dcb);
      erc = device_status[dcb->device_class](dcb);
   }
   return((*rq->residual != 0 && erc == ercOK) ? ercResidualData : erc);
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Data not yet transferred to the medium may be recovered from the device's
 buffers and/or the Sequential Access service's buffers at any time that data
 transfer operations are quiescent.  The most often encountered case is at
 early-warning end-of-medium, when all of the unwritten data needs to be
 called back before filemark(s) are written and new medium is mounted.  The
 buffer recovery order is always FIFO, commencing with the device buffers and
 proceeding to the service's own buffers.  Any applications that wish to make
 used of the LIFO recover order supported by some SCSI devices must control
 the device directly through the SCSI Manager and NOT use the Sequential
 Access service. */

private unsigned seq_access_recover(dcb_type *dcb) {

   unsigned erc = ercOK;
   seq_access_rq_type *rq;

   rq = dcb->rq;			/* Optimize pointer dereferences */
   if (rq->data == NULL || rq->data_len == 0 || rq->residual == NULL)
      return(ercInvalidTapeRequest);
   if (rq->size_residual != sizeof(*rq->residual))
      return(ercReturnDataAreaTooSmall);
   dcb->rq_slot_id = rq->rt_code;
   dcb->rq_data = rq->data;
   dcb->rq_residual = rq->data_len;
   if (dcb->state.inbound)
      return(ercTapeCommandIllegal);
   if (offset_of(rq->data) & 0x0001)
      return(ercOddAlignment);		/* Buffer must be word aligned */
   if (dcb->state.data_transfer) {
      dcb->state.rq_pending = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   if (dcb->residual != 0)		/* Any data in the device buffers? */
      erc = device_buffer[dcb->device_class](dcb);
   else
      while (dcb->rq_residual != 0 && dcb->device_buffer != NULL)
         recover_device_buffer(dcb);
   *rq->residual = dcb->rq_residual;
   return(erc);
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Data not yet transferred to the medium may be discarded from the Sequential
 Access service's buffers at any time that data transfer operations are
 quiescent.  The most often encountered case is at early-warning
 end-of-medium, when unwritten data that is still available in the client's
 buffers simply needs to be discarded before filemark(s) are written and new
 medium is mounted. */

private unsigned seq_access_discard(dcb_type *dcb) {

   if (dcb->state.inbound)
      return(ercTapeCommandIllegal);
   if (dcb->state.data_transfer) {
      dcb->state.rq_pending = TRUE;
      return(REQUEST_IN_PROGRESS);
   }
   if (dcb->residual != 0)		/* Any data in the device buffers? */
      return(ercDeviceBufferNotEmpty);	/* Too bad --- we can't discard it! */
   else {
      purge_buffers(dcb);		/* Our buffers we control, though */
      return(ercOK);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The Sequential Access service may be deinstalled piecemeal, device by
 device.  A separate deinstallation request is required for each device to be
 ousted.  The device is conveniently "forgotten" if it is not in use at the
 time of the request.  In addition, if no more devices are under the control
 of the Sequential Access service, the steps necessary before the partition
 may be vacated are taken and this condition (eligible to be vacated) is
 communicated to the requestor by returning a nonzero partition handle (user
 number). */

private unsigned seq_access_deinstall(
                                  seq_access_deinstall_rq_type *deinstall_rq) {

   dcb_type *dcb = NULL;
   char *device_name;
   unsigned device_name_len, erc, i, j;
   rq_type *rq;

   *deinstall_rq->ph = 0;		/* Assume we can't be vacated */
   device_name = deinstall_rq->device_name;
   device_name_len = deinstall_rq->device_name_len;
   for (i = 0; i < n_dcb; i++)		/* Search for requested device */
      if (     dcb_array[i]->device_name[0] == device_name_len
            && NlsULCmpB(NULL, &dcb_array[i]->device_name[1],
                         device_name, device_name_len, &j) == ercOK
            && j == 0xFFFF) {
         dcb = dcb_array[i];
         break;
      }
   if (dcb == NULL)			/* Addressee unknown! */
      return(ercInvalidTapeSpecification);
   if (dcb->user_num != 0)		/* Device is open by a user */
      return(ercTapeInUse);
   if (dcb->cant_deinstall)		/* Can the service go away? */
      return(ercServiceCantBeDeinstalled);
   if (FSrp(NULL) && (erc = update_route_table(dcb,
                                               ROUTE_TABLE_DELETE)) != ercOK)
      return(erc);
   memset(dcb->device_name, 0, sizeof(dcb->device_name));
   memset(dcb->device_password, 0, sizeof(dcb->device_password));
   for (i = 0; i < n_dcb; i++)		/* Any devices still available? */
      if (dcb_array[i]->device_name[0] != 0)
         return(ercOK);			/* Still other devices being served */
   for (i = 0; i <= seq_access_rq_code_last; i++)	/* Un-serve requests */
      ServeRq(seq_access_rq_code[i], save_rq_info[i]);
   do {					/* Purge any last minute requests */
      erc = Check(serv_exch, &rq);
      if (erc == ercOK) {
         rq->erc_ret = ercServiceNotAvail;
         Respond(rq);
      } else if (erc != ercNoMessage)
         log_erc(erc);
   } while (erc != ercNoMessage);
   erc = SetPartitionLock(0);		/* Unlock so we may vacate */
   if (erc == ercOK)			/* Final hurdle passed? */
      *deinstall_rq->ph = own_user_num;	/* Yes, give caller tool to oust us */
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 In order for routing control on the SRP (in this case, route by "device") to
 correctly deliver SeqAccessOpen and SeqAccessVersion requests to the correct
 processor within th SRP, the master routing table maintained by the Master FP
 must be updated with the names of the devices we propose to serve.  And, in a
 similar fashion, names must be removed from the Master FP's tables when
 devices are deinstalled. */

unsigned update_route_table(dcb_type *dcb, char operation) {

   unsigned erc, garbage;
   void *response_rq;
   update_route_table_rq_type *update_route_table_rq;

   memset(update_route_table_rq = (update_route_table_rq_type *) &dcb->own_rq,
          0, sizeof(dcb->own_rq));
   update_route_table_rq->s_cnt_info = 2;
   update_route_table_rq->n_req_pb_cb = 2;
   update_route_table_rq->exch_resp = default_resp_exch;
   update_route_table_rq->erc_ret = 0xFFFF;
   update_route_table_rq->rq_code = UPDATE_ROUTE_TABLE_RQ_CODE;
   update_route_table_rq->operation = operation;
   GetProcInfo(&update_route_table_rq->slot_id, &garbage, &garbage, NULL, 0,
               &garbage);
   update_route_table_rq->device_name = &dcb->device_name[1];
   update_route_table_rq->device_name_len = dcb->device_name[0];
   erc = Request(update_route_table_rq);
   if (erc == ercOK)
      do
         erc = Wait(default_resp_exch, &response_rq);
      while (erc == ercOK && update_route_table_rq != response_rq);
   if (erc == ercOK)
      erc = update_route_table_rq->erc_ret;
   if (erc == ercBadDevSpec)
      erc = ercDeviceAlreadyMounted;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The termination, abort and quiet user for swap requests are all combined into
 one in order to save request codes!  They may be distinguished by examining
 the erc at offset 12 in the request block: ercAbort (8200) signals an abort
 request, ercOK (normal termination) or ercOperatorIntervention (4) indicate a
 termination request while ercSwapping (37) is for the quiet user I/O
 request.  In actual fact, ANY erc other than ercAbort or ercSwapping are
 considered to signal a vanilla termination request.  After all, ercOK and
 ercOperatorIntervention aren't the only ways to cause the demise of a
 program. */

private unsigned seq_access_termination(seq_access_term_rq_type *term_rq) {

   dcb_type *dcb;
   unsigned i, user_num;

   user_num = term_rq->user_num;	/* Reduce segment register loads */
   term_rq->active_dcbs = 0;		/* Innocent until proven guilty */
   for (i = 0; i < n_dcb; i++)
      if (user_num == dcb_array[i]->user_num) {
         dcb = dcb_array[i];
         if (dcb->state.rq_pending)	 {	/* Pending requests have... */
            dcb->state.rq_pending = FALSE;	/* ...not yet started */
            dcb->rq->erc_ret = term_rq->term_erc;
            Respond(dcb->rq);
         }
         if (dcb->state.data_transfer || dcb->state.rq_active) {
            term_rq->active_dcbs++;	/* Buffered operations in progress */
            dcb->state.term_pending = TRUE;
         } else if (term_rq->term_erc != ercSwapping)
            seq_access_close(dcb);	/* OK to close if not just a swap! */
      }
   if (term_rq->active_dcbs == 0)
      return(ercOK);
   Send(term_exch, term_rq);		/* Pocket the termination request... */
   return(REQUEST_IN_PROGRESS);		/* ...until buffered I/O finishes */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When data transfer operations terminate unexpectedly with an exception
 condition, the user must be informed of the residuum of data still in the
 device's and/or the service's buffers.  This procedure examines all the
 buffers and sums the amount of available data that is found. */

private unsigned long calculate_residual(dcb_type *dcb) {

   buffer_descriptor_type *buffer;
   unsigned long residual;

   residual = dcb->residual;
   buffer = dcb->client_buffer;		/* OK, sum the residual count(s) */
   while (buffer != NULL) {
      residual += (dcb->state.inbound)
                      ? buffer->available : (buffer->size - buffer->available);
      buffer = buffer->next;
   }
   buffer = dcb->device_buffer;
   while (buffer != NULL) {
      residual += (!dcb->state.inbound)
                      ? buffer->available : (buffer->size - buffer->available);
      buffer = buffer->next;
   }
   return(residual);

}
