%if (0) then (
Module: IntraSegHeap.asm

Public interfaces exported by this module:

	IntraSegHeapAlloc: 	  (sgHeap, cb, ppNodeRet, fNoWait)	ercType
	IntraSegHeapDeAlloc:  (pNode)							ercType
	IntraSegHeapInit: 	  (sgHeap, cbHeap, exchLock)		ercType
	IntraSegHeapInitStats:(sgHeap, cbHeap, exchLock, pStats)ercType

Description:

This is a boundary tag heap with a singly-linked free list to optimize allocation.  The free list is maintained in ascending address order.  Allocation is first fit.  Deallocation coalesces adjacent free areas.  Each node of the heap is bounded on both sides by a tag which contains the (self-inclusive) size of the node and a bit telling whether the node is free or in-use.

The LSB of the tag is the free/in-use bit for the node.  The bit is set when the node is free.  The high 15 bits of the tag are the relative offset to the next node's leading tag (i.e offset current tag + (tag and 0fffeh) = offset next tag).  The low bit of the tag can be used as the free bit because all nodes and the heap itself are word aligned.

A heap with 2 nodes looks like this:

            ____________________   ____________________
           |                    | |                    |
           v                    | v                    |
	| header | tag0 | node0 | tag0 | tag1 | node1 | tag1 | free or end-of-heap|
				  |                    ^ |                    ^
				  |____________________| |____________________|

)fi

;
; Heap Header description
;
oExchWait			EQU 0		;exchange to wait at upon allocation failure
ocWait				EQU 2		;count of waiters
opMsg				EQU 4		;convenient place for Wait's pMsgRet
oibHeapMax			EQU 8		;limit of heap (offset of last byte)
oFreeHead			EQU 10		;offset of first free area, 0 if none (or tail)
oStats				EQU 12		;14 byte stats block
pStats				EQU 26		;address of stats block, optionally
								;kept outside of segment
sHeader				EQU 30

; stats block
ocFree				EQU 0		;current # of free nodes (fragmentation)
ocFreeMax			EQU 2		;max free nodes ever (worst fragmentation)
osAlloc				EQU 4		;bytes currently allocated
osAllocMax			EQU 6		;max bytes ever allocated
ocAllocFail			EQU 8		;count of allocation failures
ocAllocWait			EQU 10		;count of allocation waits
oHeapSize			EQU 12		;initial heap size


;
; Other Stuff
;
mFree				EQU 1		;mask free bit in tag
sNodeOverhead 		EQU 4		;bytes needed in addition to user space (tags)
sNodeSmallestPiece	EQU 20		;don't split free block if result less than this
sSmallestHeap		EQU sHeader+sNodeSmallestPiece

ercBase				EQU 1601
ercHeapTooSmall		EQU ercBase+1
ercHeapFull			EQU ercBase+2
ercBogusNode		EQU ercBase+3


EXTRN Send: FAR
EXTRN Wait: FAR


IntraSegHeap SEGMENT BYTE PUBLIC 'Code'

ASSUME CS:IntraSegHeap

;
;
; IntraSegHeapInitStats: (sgHeap, cbHeap, exch, pStatsBlock) ercType
;
; Initialize a heap which starts at sgHeap:0 for cbHeap bytes.  The exch is
; the exchange number to wait at upon Alloc failure.
;
;
PUBLIC IntraSegHeapInitStats
IntraSegHeapInitStats PROC FAR

ASSUME DS: Nothing
ASSUME ES: Nothing

pStatsBlock EQU DWORD PTR [BP+6]
exch		EQU WORD PTR [BP+10]
cbHeap		EQU WORD PTR [BP+12]
sgHeap		EQU WORD PTR [BP+14]
sArgs		EQU 10

	PUSH  BP
	MOV   BP, SP

	MOV   SI, cbHeap
	CMP   SI, sSmallestHeap
	JB    InitBadHeapSize

	MOV   AX, sgHeap
	MOV   ES, AX

; Initialize heap header size, exch, etc
	XOR   DI, DI
	XOR   AX, AX
	MOV   CX, sHeader
	CLD
	REP   STOSB

	CMP   SI, 0FFFFh
	JE    InitSetibLast
	TEST  SI, 1
	JZ    InitStoI
	DEC   SI
InitsToi:
	DEC   SI
InitSetibLast:
	MOV   ES:[oibHeapMax], SI
	MOV   BX, exch
	MOV   ES:[oExchWait], BX
	MOV   BX, 1
$MOD386
	MOV   EDX, [pStatsBlock]
	MOV   ES:[pStats], EDX
	LFS   DI, DWORD PTR ES:[pStats]
	MOV   FS:[DI+ocFree], BX
	MOV   FS:[DI+ocFreeMax], BX
	MOV   DX, cbHeap
	MOV   FS:[DI+oHeapSize], DX
$MOD286

; Initialize free list
	MOV   BX, sHeader+2			;free link addresses next link (not node tag) 
	MOV   ES:[oFreeHead], BX
	MOV   ES:[BX], AX			;0 marks end of free list

; Set free area tags
	MOV   AX, SI				;ibHeapLast
	SUB   AX, sHeader-2			;subtract header size and set free bit
	MOV   ES:[sHeader], AX
	MOV   ES:[SI-1], AX


	XOR   AX, AX
InitRetAX:
	POP   BP
	RET   sArgs

InitBadHeapSize:
	MOV   AX, ercHeapTooSmall
	JMP   InitRetAX

IntraSegHeapInitStats ENDP


;
;
; IntraSegHeapInit: (sgHeap, cbHeap, exch) ercType
;
; Initialize a heap which starts at sgHeap:0 for cbHeap bytes.  The exch is
; the exchange number to wait at upon Alloc failure.;
;
PUBLIC IntraSegHeapInit
IntraSegHeapInit PROC FAR

ASSUME DS: Nothing
ASSUME ES: Nothing

exch	EQU WORD PTR [BP+6]
cbHeap	EQU WORD PTR [BP+8]
sgHeap	EQU WORD PTR [BP+10]
sArgs	EQU 6

	PUSH  BP
	MOV   BP, SP
	PUSH  sgHeap 
	PUSH  cbHeap
	PUSH  exch
; pStats points to heap header
	PUSH  sgHeap	;pStats.sa
	PUSH  oStats	;pStats.ra
	CALL  IntraSegHeapInitStats
	POP   BP
	RET   sArgs

IntraSegHeapInit ENDP


;
;
; IntraSegHeapAlloc: (sgHeap, cb, ppNodeRet, fNoWait) ercType
;
; Allocate a node of cb bytes from the given heap, return a pointer to the
; user portion of the node.  Algorithm is first fit.  If allocation fails and
; fNoWait is asserted, return ercHeapFull; otherwise (not fNoWait) wait for
; sufficient free space to become available.
;
;
PUBLIC IntraSegHeapAlloc
IntraSegHeapAlloc PROC FAR

ASSUME DS: Nothing
ASSUME ES: Nothing

fNoWait		EQU BYTE PTR [BP+6]
ppNodeRet	EQU DWORD PTR [BP+8]
cb			EQU WORD PTR [BP+12]
sgHeap		EQU WORD PTR [BP+14]
sArgs		EQU 10

	PUSH  BP
	MOV   BP, SP
	PUSH  DS

	MOV   AX, sgHeap
	MOV   DS, AX

	MOV   CX, cb
	ADD   CX, sNodeOverhead

	CMP   CX, sNodeSmallestPiece
	JAE   AllocRoundToWord
	MOV   CX, sNodeSmallestPiece
	JMP   AllocGo
AllocRoundToWord:
	TEST  CX, 1
	JZ    AllocGo
	INC   CX

AllocGo:						;DS=sgHeap, CX=cb+overhead (free bit off)
	PUSHF
	CLI							;DISABLE
AllocRetry:						;come here from AllocWait
	MOV   DI, oFreeHead
	MOV   BX, DS:[DI]

; CX = size needed
; DS:BX = addr of free node.oFreeNext
AllocTest:
	OR    BX, BX
	JZ    AllocTestWait
	
	CMP   CX, DS:[BX-2]
	JBE   AllocFound
	MOV   DI, BX
	MOV   BX, DS:[BX]
	JMP   AllocTest

AllocTestWait:
	LES   SI, DWORD PTR DS:[pStats]
	TEST  fNoWait, 1
	JNZ   AllocNoWait
	INC   WORD PTR ES:[SI+ocWait]  ;inc count of waiters (DeAlloc will dec)
	POPF						;ENABLE
	JMP   AllocWait
AllocNoWait:
	POPF						;ENABLE
	INC   WORD PTR ES:[SI+ocAllocFail]
	MOV   AX, ercHeapFull
	JMP   AllocRetAX

AllocFound:
	MOV   AX, DS:[BX]			;oFreeNext
	SUB   BX, 2					;address tag
	MOV   SI, DS:[BX]			;tag
	DEC   SI					;remove free bit
	MOV   DX, SI
	SUB   DX, CX

; DS:BX = @freeNode.tag
; AX = oFreeNext
; DI = oFreePrev
; SI = size this free node
; CX = requested size
; DX = count bytes left
	CMP   DX, sNodeSmallestPiece
	JA    SplitNode

; Take whole node, remove from free list, update tags
	MOV   DS:[DI], AX			;freePrev.oFreeNext = freeNode.oFreeNext
	MOV   DS:[BX], SI			;set leading tag
	MOV   DS:[BX][SI-2], SI		;set trailing tag
	MOV   CX, SI
	LES   SI, DWORD PTR DS:[pStats]
	DEC   ES:WORD PTR [SI+ocFree]
	JMP   ReturnP

SplitNode:
; Mark new in-use node
	MOV   SI, CX
	MOV   DS:[BX], CX
	MOV   DS:[BX][SI-2], CX

; Make new free piece & thread to free list
	INC   DX					;set free bit
	MOV   DS:[BX][SI], DX		;set leading tag
	MOV   DS:[BX][SI+2], AX		;newFree.oFreeNext = freeNode.oFreeNext
	LEA   AX, [BX][SI+2]
	MOV   DS:[DI], AX			;freePrev.oFreeNext = .newFree.oFreeNext
	ADD   SI, DX
	MOV   DS:[BX][SI-3], DX		;set trailing tag
	LES   SI, DWORD PTR DS:[pStats]

ReturnP:
; DS:BX = @node.tag
; CX = tag (free bit off = size)
; ES:SI = @stats
	ADD   CX, ES:[SI+osAlloc]
	MOV   ES:[SI+osAlloc], CX
	CMP   CX, ES:[SI+osAllocMax]
	JBE   AllocStatsCurrent
	MOV   ES:[SI+osAllocMax], CX
AllocStatsCurrent:
	POPF						;ENABLE
	LEA   AX, [BX+2]
	LES   BX, ppNodeRet
	MOV   ES:[BX], AX
	MOV   ES:[BX+2], DS

	XOR   AX, AX

AllocRetAX:
	POP   DS
	POP	  BP
	RET	  sArgs

AllocWait:
; Heap is full and caller will wait for a node to become free
; DS = sgHeap
; CX = size of node needed
; cWait inc'd, interrupts enabled, caller's DS on stack
; ES:SI = @stats
	POP   AX					;caller's DS
	PUSH  AX
	PUSH  DS					;save...
	PUSH  CX
	INC   WORD PTR ES:[SI+ocAllocWait]

	PUSH  DS:[oExchWait]
	PUSH  DS
	PUSH  opMsg
	MOV   DS, AX				;restore caller's DS while waiting (debugging)
	CALL  Wait					;DeAlloc will Send to us

	POP   CX					;restore...
	POP   DS
	OR    AX, AX
	JNZ   AllocRetAX			;wait failed?!

	PUSHF
	CLI							;DISABLE
	CMP   WORD PTR DS:[ocWait], 0
	JNE   AllocGoose
	JMP   AllocRetry

AllocGoose:
	DEC   WORD PTR DS:[ocWait]	;dec # waiters: waking up 1
	POPF						;ENABLE
	PUSH  DS					;save...
	PUSH  CX

	PUSH  DS:[oExchWait]
	PUSH  0
	PUSH  0FFFEh				;distinguishes Alloc msg
	CALL  Send					;notify next waiter of free heap space

	POP   CX					;restore...
	POP   DS
	OR    AX, AX
	PUSHF
	CLI							;DISABLE
	JNZ   AllocGooseFailed
	JMP   AllocRetry

AllocGooseFailed:
	INC   WORD PTR DS:[ocWait]	;send failed, someone still waiting
	JMP   AllocRetry

IntraSegHeapAlloc ENDP


;
;
; IntraSegHeapDeAlloc: (pNode) ercType
;
; Free a node previously Alloc'd, return ercBogusNode if any corrupt boundary
; tags are encountered.  Freed node is joined with adjacent free nodes (if any)
; and is inserted on the free list in ascending address order.  If any waiters
; malinger, goose 'em.
;
;
PUBLIC IntraSegHeapDeAlloc
IntraSegHeapDeAlloc PROC FAR

ASSUME DS: Nothing
ASSUME ES: Nothing

pNode	EQU DWORD PTR [BP+6]
raNode	EQU WORD PTR [BP+6]
saNode  EQU WORD PTR [BP+8]
sArgs	EQU 4

	PUSH  BP
	MOV   BP, SP
	PUSH  DS

	LDS   BX, pNode
	SUB   BX, 2					;DS:BX=@node.tag

; Verify valid boundary tags and prepare to mark free
	MOV   SI, DS:[BX]
	TEST  SI, mFree
	JNZ   BogusNode				;free bit already set!
	CMP   SI, DS:[BX][SI-2]
	JNE   BogusNode             ;trailing tag <> leading tag!
	MOV   AX, SI
	INC   AX					;set free bit
; Mark node free
; DS:BX = @node.tag
; SI = count
; AX = tag (free)
	LES   DI, DWORD PTR DS:[pStats]
	SUB   ES:[DI+osAlloc], SI
	PUSHF
	CLI							;DISABLE
	MOV   DS:[BX], AX
	MOV   DS:[BX][SI-2], AX

; Test for adjacent free nodes to join up with, update free list
	CMP   BX, sHeader
	JBE   FindOrdinal			;first in heap, no predecessor node
	MOV   DI, DS:[BX-2]
	TEST  DI, mFree
	JZ    FindOrdinal			;predecessor node not free

; Join with predecessor node
; DS:BX = @node.tag
; SI = count
; AX = tag (free)
; DI = predecessor.tag (free)
	DEC   DI					;tag -> count
	SUB   BX, DI				;DS:BX=@prevNode.tag
	JC    BogusFreeNode
	MOV   AX, DS:[BX]
	DEC   AX					;tag -> count
; AX = prevNode leading count
; DI = prevNode trailing count
; SI = current node count
; DS:BX = @prevNode.tag
	CMP   AX, DI
	JNE   BogusFreeNode			;predecessor's leading tag <> trailing tag!
	ADD   AX, SI
	MOV   SI, AX
	INC   AX					;count -> tag
	MOV   DS:[BX], AX			;mark joined node tags...
	MOV   DS:[BX][SI-2], AX
	JMP   JoinSuccessor			;freed node is now on free list, too

BogusFreeNode:
	POPF						;ENABLE
BogusNode:
	MOV   AX, ercBogusNode
	JMP   DeAllocRetAX

FindOrdinal:
; Find our ordinal on the free list & insert there
; DS:BX = @node.tag
; AX = node.tag (free)
	MOV   DI, oFreeHead
	MOV   SI, DS:[DI]
TestOrdinal:
	OR    SI, SI
	JZ    FoundOrdinal			;end of free list
	CMP   BX, SI
	JB    FoundOrdinal			;found our place
	MOV   DI, SI
	MOV   SI, DS:[SI]
	JMP   TestOrdinal

FoundOrdinal:
; DS:BX = @node.tag
; DI = oFreePrev
; SI = oFreeNext
	MOV   DS:[BX+2], SI			;link to free list...
	LEA   CX, [BX+2]
	MOV   DS:[DI], CX
	LES   DI, DWORD PTR DS:[pStats]
	INC   WORD PTR ES:[DI+ocFree]
	JMP   SHORT JoinSuccessor2

JoinSuccessor:
	LES   DI, DWORD PTR DS:[pStats]
JoinSuccessor2:
; DS:BX = @node.tag, node is on free list
; AX = node.tag (free)
; ES:DI = @stats
	MOV   SI, AX
	DEC   SI					;remove free bit (SI=count)
	LEA   CX, [BX][SI]
	CMP   CX, DS:[oibHeapMax]
	JAE   DeAllocRetOk			;node is last in heap, no successor
	MOV   DX, DS:[BX][SI]
	TEST  DX, mFree
	JZ    DeAllocRetOk			;successor not free

	MOV   CX, DS:[BX][SI+2]		;node.oFreeNext = successorNode.oFreeNext...
	MOV   DS:[BX+2], CX
	DEC   WORD PTR ES:[DI+ocFree]
	DEC   DX					;remove free bit
	ADD   AX, DX
	MOV   SI, AX
	MOV   DS:[BX], AX			;mark joined node tags...
	MOV   DS:[BX][SI-3], AX

DeAllocRetOk:
	MOV   AX, ES:[DI+ocFree]
	CMP   AX, ES:[DI+ocFreeMax]
	JBE   DeAllocStatsCurrent
	MOV   ES:[DI+ocFreeMax], AX	;heap fragmentation getting worse
DeAllocStatsCurrent:	
	CMP   WORD PTR DS:[ocWait], 0
	JNE   DeAllocGoose
	POPF						;ENABLE
	XOR   AX, AX	
DeAllocRetAX:
	POP   DS
	POP	  BP
	RET	  sArgs

DeAllocGoose:
	DEC   WORD PTR DS:[ocWait]	;dec # waiters: waking up 1
	POPF						;ENABLE

	PUSH  DS:[oExchWait]
	PUSH  0
	PUSH  0FFFFh				;distinguishes Alloc msg
	CALL  Send					;notify first waiter of free heap space
	OR    AX, AX
	JZ    DeAllocRetAX

	INC   WORD PTR DS:[ocWait]	;send failed, someone still waiting
	XOR   AX, AX
	JMP   DeAllocRetAX

IntraSegHeapDeAlloc ENDP

IntraSegHeap ENDS

END

;
; Modification history:
;
;	10/7/88  JM created
;	3/28/89  MTR Make heap non-blocking, add fNoWait parm to IntraSegHeapAlloc,
;				 add free list, add stats, and a few other minor embellishments.
;	02/06/93 JM support statistics gathering external to the heap.

