;*****************************************************************************
; WsLph_p.asm
; Workstation Line Protocol Handler 
; This module contains code that is hardware specific for the NGen.
;*****************************************************************************

%set(CTOSp,1)
%SET(Debug,0)
%SET(NGen,1)
%SET(Aws,0)
%SET(Iws,0)

;*****************************************************************************
;  PUBLIC PROCEDURES
;*****************************************************************************
PUBLIC BitStartWrite
PUBLIC SigIsr
PUBLIC IsrNull, IsrRxSpec, IsrExStat	;Dummy procedures to resolve externals

;*****************************************************************************
;  LITERALS
;*****************************************************************************
apa 			EQU 80h		;All Parties Address
idle 			EQU 0		;Send Abort if Tx underrun
bop 			EQU 0		;Bit Oriented Protocol (as opposed to Byte Control)
burst2			EQU 1000h	;Cirrus chip dma burst 2 word - R&B
b38lcw 			EQU 14
SG5000			EQU 15		;ProcessorType for SuperGen
PCAT			EQU 16		; ProcessorType for PCAT
bEnableEarChan0 EQU 11h		;ear enabled in word mode - GateArray
cws186 			EQU 7
dcd  			EQU 040H
dcdIntEnb 		EQU 08h
dmarcv			EQU 02h		;flushing FIFO command to Cirrus - R&B
endxmit			EQU 05h		;Stop xmit, keep dmatxe on - R&B
FIFOEmptyBit	EQU 0100h	; R&B
hbe				EQU 1		; R&B
mode3 			EQU 036h
modeCtr2 		EQU 080h
NewGen 			EQU 9
NotrSom			EQU 0FEh
NotREOMorRSOM	EQU 0FCh
rcvEnb  		EQU 022h
rcvEnbDmaDis	EQU 020h
rEom  			EQU 02h
rErr  			EQU 80h
rOvrn  			EQU 08h
rAbort  		EQU 04h
rts  			EQU 080h
RxDAorRxSA		EQU	3
RxSA			EQU	2
sam 			EQU 10h		;secondary address mode
t2 				EQU 5
tErr  			EQU 80h
timer2MaxCntA 	EQU 0ff62h
timerLatchCnt 	EQU 40h
tSom  			EQU 01h
tUndrn  		EQU 010h
txwcen			EQU 0401h	;Cirrus chip transmit word count enable - R&B
xmtDone  		EQU 020h
xmtEnb  		EQU 044h
xmtEnbNoDma 	EQU 040h
xmtEnbRcvEnb	EQU 064h

; Move the below equates into the init code..
latch8254		EQU 00h			; latch counter 0
latch8254Sgen	EQU 80h			; latch Counter 2

;*****************************************************************************
; XBlock Definition: This has been moved to MstrEqu.Idf to be common to
; 					 all modules that use it.
; Xid Definition: ditto.
;*****************************************************************************
%INCLUDE(:f1:MstrEqu.idf)

; Values of xbFlags:

xbIFrame 			EQU 1
xbXBlockToFill 		EQU 2

;*****************************************************************************
; Include common code
;*****************************************************************************
%INCLUDE(WsLph_all.asm)

;*****************************************************************************
;  EXTERNAL DATA
;*****************************************************************************
DGroup GROUP DATA
Data SEGMENT PUBLIC 'DATA'

EXTRN cascadeOCW1_8259:WORD
EXTRN cascadeOCW2_8259:WORD
EXTRN commDmaAddr:WORD
EXTRN commDmaWrdCnt:WORD
EXTRN commDmaEar:WORD
EXTRN commDmaEarHigh:WORD
EXTRN DMABytePtrClr:WORD
EXTRN counter08254NGen:WORD
EXTRN cPort:WORD
EXTRN DmaMask:WORD
EXTRN DmaMode:WORD
EXTRN DmaCommand:WORD
EXTRN DmaEarEnable:WORD
EXTRN ExtCtlReg:WORD
EXTRN modeWord8254NGen:WORD
EXTRN ModeWord8254RTC:WORD
EXTRN RTCCounter8254:WORD
EXTRN OCW1_8259:WORD
EXTRN OCW2_8259:WORD
EXTRN pcsarL:WORD
EXTRN pcsarH:WORD
EXTRN pcrl:WORD
EXTRN pcrH:WORD
EXTRN processorType:BYTE
EXTRN rdsrL:WORD
EXTRN rdsrH:WORD
EXTRN stat:WORD
EXTRN tdsrL:WORD
EXTRN tdsrH:WORD
EXTRN timerCtl:WORD
EXTRN timerComm:WORD
EXTRN shCPUSpeed:BYTE
EXTRN vf:BYTE
$INCLUDE(:f1:vfequ.idf)

;*****************************************************************************
;  GLOBAL DATA
;*****************************************************************************
PUBLIC parameterControl, ackInDmaAddr, ackInDmaEar, ackOutDmaAddr, ackOutDmaEar
PUBLIC BaudRateCtlReg, BaudRateCtlMask, timerCommEOP, commDataReg, pBaudRateCtlWord, BaudRateCtlWord
PUBLIC ccrRcvEnb, ccrXmtEnb, ccrStopDMA						;R&B
PUBLIC commDmaEnable, commDmaDisable, modeDmaFromMemory, modeDmaToMemory
PUBLIC StoreRdsrH
PUBLIC fMaskParallelDMA, parallelDmaMask, parallelDmaWrdCnt	;R&B
PUBLIC parallelMask, parallelUnmask							;R&B

parameterControl DB 0	;contains bop + apa + idle (+ sam if not search id)

   EVEN
PUBLIC SafeOutSbAddress
SafeOutSbAddress	DW 0
ackInDmaAddr 		DW 0	;Set up by init code
ackInDmaEar 		DW 0	;Set up by init code
ackOutDmaAddr 		DW 0	;Set up by init code
ackOutDmaEar 		DW 0	;Set up by init code
BaudRateCtlReg 		DW 0	;Set up by init code
BaudRateCtlMask 	DW 0	;Set up by init code
timerCommEOP 		DW 0	;Set up by init code
commDataReg 		DW 0	;Set up by init code
pBaudRateCtlWord 	DD 0	;Set up by init code
BaudRateCtlWord 	DW 0	;A word used in SetLineSpeed routine
							;   to genericize that routine
ccrRcvEnb			DW 0	;Set up by init code
ccrXmtEnb			DW 0	;Set up by init code
ccrStopDMA			DW 0	;Set up by init code - R&B
commDmaEnable		DB 0	;Set up by init code
commDmaDisable 		DB 0	;Set up by init code
modeDmaFromMemory 	DB 0	;Set up by init code
modeDmaToMemory 	DB 0	;Set up by init code
parallelDmaMask		DW 0	;Set up by init code (initClstr)	R&B ...
parallelDmaWrdCnt	DW 0	;Set up by init code (initClstr)
parallelMask		DB 0	;Set up by init code (initClstr)
parallelUnMask		DB 0	;Set up by init code (initClstr)
fMaskParallelDMA	DB 0	;Set up by init code (initClstr)	... R&B

; Collision detection variables
StoreRdsrH			DB 0
shProcSpeed			DB 0	
xmtFlags			DB 0
xmtData				DB 0	
CollisionErc		DW 0

Data ENDS

WsLph SEGMENT PUBLIC 'CODE'
	ASSUME CS: WsLph, DS: Dgroup
; The following labels are necessary to resolve an external reference from InitWs1:

IsrNull:
IsrRxSpec:
IsrExStat:

BitStartWrite PROC FAR
PUBLIC BitStartWrite
;*****************************************************************************
; We get called by 
;	    the interrupt handler when a read has completed, and we wish to
;	    start the next poll.
; pollState contains our current polling state.
;   pollState = StateIdle means we are not doing anything. Otherwise:
;	pollState = StateWaitDcdDrop means we are waiting for the data
;		    carrier detect to drop so we can start a write.
;	pollState = StateWriting means a write is currently in progress
;	pollState = StateReading means a read is in progress
; N.B. INTERRUPTS MUST BE DISABLED
;*****************************************************************************
	CMP  pollState,OFFSET StateIdle
	JE   PrepareDma
	MOV  AX,ercBusyBitIO
	PUSH AX
	CALL Crash


PrepareDma:
PUBLIC PrepareDma
	INC  pollSequenceNumber

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,logBufferIndex
	ADD  DI,logEntrySize
	CMP  DI,logBufferIndexMax
	JBE  logBufferIndexOk
	MOV  DI,0
logBufferIndexOk:
	MOV  logBufferIndex,DI
	MOV  logIntIdle[DI],0
	MOV  logIntDcd[DI],0
	MOV  logIntRead[DI],0
	MOV  logIntWrite[DI],0
	MOV  logBadCrc[DI],0
	MOV  AX,pollSequenceNumber
	MOV  logFrameNumber[DI],AX
	MOV  logFrameOut[DI],0
	MOV  AX,lineState
	MOV  logLineState[DI],AX
	CMP  AX,lastLineState
	JBE  StateOk1
	MOV  AX,ercInternalConsistencyCheck
	PUSH AX
	CALL Crash
StateOk1:
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; Start the timer. If 2 seconds go by, we did not get a response.
	MOV  WORD PTR trbInterval,trbTimeoutValue
	XOR  AX,AX
	MOV  WORD PTR trbCEvents,AX
	CMP  bitIoErc,AX
	JNE  StartWrite
; Zero the retry count if the last poll did not have an error.
	MOV  retryCount,AX
StartWrite:
; Come here from interrupt routine when Dcd changes.
	MOV  DX,stat
	IN   AL,DX							; Byte I/O
	JMP  $+2
	AND  AL,dcd
	JZ   NoWaitForDcdDrop
; We must wait for an interrupt for data carrier detect to change before we
; can continue. 
	MOV  AX,dcdIntEnb
	MOV  DX,cPort
	OUT  DX,AL							; Byte I/O
	JMP  $+2
	MOV  DX,stat
	IN   AL,DX							; Byte I/O
	JMP  $+2
	AND  AL,dcd
	JZ   NoWaitForDcdDrop
	MOV  pollState,OFFSET StateWaitDcdDrop
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  nWaitDcd
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	RET

NoWaitForDcdDrop:
	PUSHF
	CLI	
	CALL DisableDma

; Check to see if we need an XBlock.
	MOV  SI,lineState
	TEST lstFlags[SI],fReadIntoXBlock
	JZ   CheckWriteFrom

; We need an XBlock.  Check to see if we have an XBlock to read into.
	CMP  saXBlockCurrent,0
	JNE  CheckWriteFrom
;See if an XBlock is available.  If so, unchain it from the free list.
	CALL GetXBlock
	JNZ  CheckWriteFrom				

; We don't have an XBlock.  See if we can change our state to stateWsNotReady.
	TEST lstFlags[SI],fElseWsNotReady
	JZ   NoXBlock
	MOV  SI,WORD PTR lstFlags[SI]
	AND  SI,fElseWsNotReady
	MOV  lineState,SI
	JMP  SHORT CheckWriteFrom

; We don't have an XBlock.  This is a program error!
NoXBlock:
PUBLIC NoXBlock
	MOV  AX,ercInternalConsistencyCheck
	PUSH AX
	CALL Crash

CheckWriteFrom:
; We will write from the ackOutBuf or the XBlock.  The lineStateTable tells
; where we write from.  If lstCbAckOut <> 0, then we write from the ackOutBuf. 
	XOR  CX,CX
	OR   CL,lstCbAckOut[SI]		;CX = count from ackOutBuf
	JZ   WriteFromXBlock		;or zero if out from XBlock
; We are writing from the ackOutBuf. Set up the buffer with the station 
; address and frame type. 

OrSeqBits:
; OR in sequence number bits.  NR will always be zero when frame types do not
; require a sequence number.
	MOV  AL,stationAddress		;AL = our station address
	MOV  AH,lstFrameType[SI]
	OR   AH,NR									
	MOV  ES,saAckOutBuf
	MOV  ES:[0],AX			;Store stationFrame in ackOutBuf

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logFrameOut[DI],AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	MOV  AX,ackOutDmaAddr
	TEST vf_fPCAT,01
	JNZ	 NoDec1				; ClusterCard does not use R&B
	TEST vf_f4MBit, 0FFh	;R&B ... If we're a cirrus, need to dec DMA
	JZ   NoDec1				;  address because the first word put in the FIFO
	DEC  AX					;  is going to get trashed by the 2652.
NoDec1:						;... R&B
	MOV  BX,ackOutDmaEar
	JMP  SHORT SetUpDmaForWrite

WriteFromXBlock:
; We are writing from the XBlock pointed to by saXBlockOut.  The XBlock
; contains the Dma word address and count.  This is an IFrame, so we must OR
; in the sequence numbers. 

	MOV  ES,saXBlockOut			;ES points to XBlock
	MOV  AX,WORD PTR NR			;AL = NR, AH = NS
	SHR  AH,4					;NS to be in bits 3 2 1								
	OR   AH,AL					;NR to be in bits 7 6 5
	OR   AH,lstFrameType[SI]
	MOV  AL,stationAddress		;AL = our station address
	MOV  ES:[xbStation],AX		;store frame type and sequence numbers

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logFrameOut[DI],AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	MOV  AX,ES:[xbDmaAddr]
	TEST vf_fPCAT,01
	JNZ	 NoDec2					;ClusterCard is 4Mbit but does not use R&B.
	TEST vf_f4MBit, 0FFh		;R&B ... If we're a cirrus, need to dec DMA
	JZ   NoDec2					;  address because the first word put in the
	DEC  AX						;  FIFO is going to get trashed by the 2652.
NoDec2:							;... R&B
	MOV  BX,ES:[xbDmaEar]
	MOV  CX,ES:[xbCbData]
	CMP  nXBlocksToSend,0		;Do we have another XBlock to send?
	JE   SetUpDmaForWrite		;No
; If we have another XBlock to send, and if this master can handle the
; protocol, set the high bit of xbCbDataSent to tell the master we want to be
; polled again.
	CMP  masterRevisionLevel,90h
	JBE  SetUpDmaForWrite
	OR   ES:BYTE PTR[xbCbDataSent+1], 80h

SetUpDmaForWrite:
	MOV  DL,modeDmaFromMemory
	SHR  CX,1
	TEST vf_fPCAT,01			;ClusterCard is 4Mbit but is not PCAT.
	JNZ	 NoInc
	TEST vf_f4MBit, 0FFh		;R&B ... If we're a cirrus, need to inc count
	JZ   NoInc					;  of words to adjust for the earlier dec of
	INC  CX						;  the DMA address.
	CALL SetUpCommDma
	DEC  CX						;Put CX back to preserve word count
	JMP  SHORT DmaSetUpToWrite
NoInc:
	CALL SetUpCommDma

DmaSetUpToWrite:				;... R&B
	MOV  DX,rdsrH
	IN   AL,DX    				;Reset Read Data Status Register Byte I/O
	JMP  $+2
	CALL EnableDma				;Start Dma
	POPF

	TEST fMaskParallelDma,0FFh	;R&B ... 
	JZ   TestForCirrus
	MOV  DX,parallelDmaMask		;Mask Parallel DMA
	MOV  AL,parallelMask
	OUT  DX,AL
	JMP  $+2

TestForCirrus:					;... R&B
	TEST vf_f4MBit, 0FFh		;R&B ...
	JNZ  StartWriteCirrus

StartWriteIOGA:
PUBLIC StartWriteIOGA			;... R&B
	MOV  AL,tsom				;send flags
	MOV  DX,tdsrH
	OUT  DX,AX					;Send Start of Message to 2652
	MOV  DX,cPort
	MOV  AX,rts					;rts enable to send flags
	OUT  DX,AX
	PUSH CX						;Save CX 
	MOV  CL,SioClock			;timming loop to insure at least 4 flags
	MOV  CH,0
	LOOP $						;Wait for flags
	Jmp  $+2
	MOV  AX,rts+xmtEnbNoDma		;rts enables transmitter and
	OUT  DX,AX					;xmtEnbNoDma sets TxE in 2652 sends flags
; The purpose of the loop$ is to allow time for the 2652 to send the flag
; characters. If we don't allow time to send flags, the workstation won't
; see the frame. 
	MOV  AL,SioClock
	SHR  AL,1					;timming loop to insure at least 4 flags
	MOV  CL,shCPUSpeed
	MOV  AH,0
	SHL  AX,CL
	MOV  CX,AX
	LOOP $						;Wait for flags
	POP  CX                     ;Restore CX
	MOV  AX,rts+xmtEnb
	MOV  DX,cPort				;xmtEnb enables DMA
	OUT  DX,AX
	JMP  StartWriteSetState

; R&B ... This new code preloads the cirrus FIFO before enabling the transmit ; clock.  This allows us to buffer up short frames and transmit them completely
; out of the FIFO.  This dramatically reduces the number of underruns and 
; timeouts.
StartWriteCirrus:
PUBLIC StartWriteCirrus
	TEST vf_fPCAT,01
	JNZ	 StartWriteClusterCard	; PCAT only.
	MOV  AL,05			; Prepare ones - Get rid of first dummy byte
	MOV  DX,tdsrH
	OUT  DX,AL
	MOV  AX,ccrXmtEnb	; Fill FIFO
	MOV  DX,cPort		; Send start of message 
	OUT  DX,AX			; Start DMA transfers to FIFO
	JMP  .+2			; Synchronize code execution to DMA transfers
	JMP  .+2			; Filling FIFO generates 1 or 2 bogus flags (TXC off)
	JMP  .+2			; Four Jumps due to 4-level FIFO (= 4 XFERS)
	JMP  .+2			; Now, the Cirrus chip has turned off TSOM
	PUSH CX				; Therefore, TXSO will become "1" within one flag time
	PUSH SI				;
	MOV  CH,0			; Filling FIFO generates 2 bogus flags
	MOV  CL,SioClock	;   so wait to discard them
	SHL  CX,1
	MOV  DX,stat
;Set SI so REP OUTSB won't fault - JF - 04/01/93
	MOV  SI,SafeOutSbAddress	
	REP  OUTSB			; This OUTSB is harmless and provides a reliable,
	MOV  DX,cPort		;   machine independent wait function.
	MOV  AX,ccrXmtEnb
	OR   AX,rts			; Send ones
	OUT  DX,AX
	MOV  CH,0			; Timming loop to insure at least 8 ones
	MOV  CL,SioClock
	SHL  CX,1
	MOV  DX,stat
	REP  OUTSB			; Use OUTSB for a delay again
	MOV  DX,tdsrh
	MOV  AL,tsom
	OUT  DX,AL			; Send flags and get rid of second dummy byte
	MOV  CH,0			; Wait for flags
	MOV  CL,SioClock
	SHL  CX,3			; Telecluster fix - was 2
	MOV  DX,stat
	REP  OUTSB			; Use OUTSB for a delay again
	POP  SI				; Restore SI
	POP  CX				; Restore CX
	MOV  AL,0
	MOV  DX,tdsrh
	OUT  DX,AL			; Send data
	JMP  StartWriteSetState
; ... R&B
; New for PCAT -GWH
StartWriteClusterCard:
PUBLIC StartWriteClusterCard
	MOV	 AL,Tsom		; Send preceeding ones for telecluster hub.
	MOV	 DX,TdsrH
	OUT  DX,AL
	JMP  $+2
	MOV  DX,CPort
	MOV  AL,rts
	OUT  DX,AL
	JMP  $+2
; Wait while Sending out preceeding ones for telecluster hub sync.
	MOV  DX,TimerCtl		;ClusterCard 8254 Control Port (offset F)
	MOV  AL,50h				;Program Channel 1 to mode 0
	OUT  AL,DX
	JMP  $+2	
	SUB  DX,2				;ClusterCard 8254 Channel 1 count port (offset D)
	MOV  AL,0A0h			;adusted for linespeed
	OUT  AL,DX
	JMP  $+2
; Now Poll Timer to see when alarm goes off
WaitPreOnes:
	MOV  DX,TimerCtl		;ClusterCard 8254 Control Register (offset F)
	MOV  AL,0E4h		; Channel 1 Read Back Command
	OUT  DX,AL
	JMP  $+2
	JMP  $+2
	SUB  DX,2			; ClusterCard 8254 chnl 1 Count Reg (offset D)
	IN	 AL,DX
	JMP  $+2
	AND  AL,80h
	JZ	 WaitPreOnes

;Now Send TSOM (Front Porch) flags
	MOV  DX,CPort
	MOV  AL,rts+xmtEnbNoDma
	OUT  DX,AL
	JMP  $+2
; Program a Wait while flags go out on the line
	MOV DX,TimerCtl					; ClusterCard 8254 Control Register
	MOV AL,70h					; Select Channel 1 for Mode 0
	OUT DX,AL
	JMP $+2
	SUB DX,2h			; ClusterCard 8254 Channel 1 Count Register (offset D)
	MOV AX,0A0h			; This value adusted for linespeed (WORD)
	OUT AL,DX					; LSB
	JMP $+2
	XCHG AH,AL
	OUT DX,AL					; MSB
	JMP $+2
	JMP $+2

; Poll 8254 for TSOM interval expiration
WaitTSOMFlags:
	MOV DX,TimerCtl			; ClusterCard 8254 Control Register(offset F)
	MOV AL,0E4h				; Read-Back command - Get status channel 1
	OUT AL,DX
	JMP $+2
	SUB DX,2				; ClusterCard 8254 Chnl 1 Count Register(offsetD)
	IN  AL,DX				; Read Status
	JMP $+2
	AND AL,80h				; Check for null count
	JZ	WaitTSOMFlags
; Now Enable DMA out on the line
	MOV DX,cPort
	MOV AL,rts+XmtEnb
	OUT DX,AL
	JMP $+2
	
StartWriteSetState:		; R&B ...
	MOV  pollState,OFFSET StateWriting	;set state to Writing
	CMP  SioClock,4		; Always return when running slower than 1.8M
	JG   WriteNotDone
	CMP  CX,4			; Jump if message size was 4 bytes or less
	JBE  WaitForWrite	; ... R&B

WriteNotDone:
	RET   		          ;Write has not finished.  Wait for next interrupt.

Underrun:
; There isn't much we can do about underrun.  It should not happen if the
; hardware is working properly.  If the workstation doesn't like the frame, it
; will simply time out.  We need to stop the Cirrus chip DMA requests.
	PUSH AX							;R&B ...
	TEST vf_fPCAT,01
	JNZ  UnderrunPCAT
	MOV  AX,ccrStopDMA				;If using a Cirrus chip, this enables the
	OR   AL,rts
	MOV  DX,cport					;  terminal counter, we load the counter
	OUT  DX,AX						;  with 1 to cause DMA to stop early.
	ADD  DX,2						;cPort + 2 is address of terminal count
									;  register on NGen with Cirrus chip
	MOV  AX,1						;Load 1 in counter
	OUT  DX,AX
	MOV  AX,rts OR endxmit OR txwcen;This turns off TXE.  On the IOGA, DMA
									;  will finish after the stacker empties.
	MOV  DX,cport					;Restore Control Register port address
	OUT  DX,AX						;... R&B
	JMP  ExitUnderrun
UnderrunPCAT:
; On PCAT, leave clock going and shut of Tx.  We may want to so something to
; stop additional DMA requests.
	MOV  DX,CPort
	MOV  AL,rts
	OUT  DX,AL
	JMP  $+2

ExitUnderrun:
	POP  AX
	INC  nUnderrun
	JMP  TestXmitDone

WaitForWrite:
; Come here if we have a short packet to send.
	MOV  DX,stat
	MOV  CX,10				; R&B... Set a maximum number of times we'll
StillWaiting:				;   loop before giving up and returning.
	IN   AL,DX				; Byte I/O
	JMP  $+2
	TEST AL,xmtDone
	JNZ  WriteDone
	LOOP StillWaiting
	RET
WriteDone:					; ...R&B
BitStartWrite ENDP
CheckWriteAndStartRead PROC FAR
PUBLIC CheckWriteAndStartRead
;*****************************************************************************
; Enter here from the interrupt routine when a write has completed.
; SI = lineState
;*****************************************************************************
	MOV  DX,stat
	IN   AL,DX
	TEST AL,tUndrn
	JNZ  Underrun
TestXmitDone:					; R&B ... Unmask parallel DMA if we masked
	PUSH AX						;   it above.
	TEST fMaskParallelDma,0FFh
	JZ   ParallelDmaUnmasked
	MOV  DX,parallelDmaWrdCnt	; Is the word count not 0FFFFh?
	PUSHF
	CLI
	IN   AL,DX
	XCHG AL,AH
	JMP  $+2
	IN   AL,DX
	POPF
	CMP  AX,0FFFFh
	JZ   ParallelDmaUnmasked
	MOV  AL,parallelUnmask		; Unmask the channel
	MOV  DX,parallelDmaMask
	OUT  DX,AL
ParallelDmaUnmasked:
	POP  AX

	TEST AL,xmtDone
	JNZ  WaitForCRC
	JMP  WriteNotDone			; ... R&B

WaitForCRC:
; This loop needs to be machine speed independent.
	MOV  CX,10
	MOV  DX,stat				; R&B ... Now wait for CRC & flags to be sent
WaitingForCRC:
	IN   AL,DX					; Byte I/O
	JMP  $+2
	TEST AX,4h
	JNZ  GotCRC
	LOOP WaitingForCRC
GotCRC:
; On ClusterCard use the on board 8254 for TEOM flag timing.
	TEST vf_fPCAT,01
	JZ	 RepTimingTEOM
;ClusterCard Only:
	MOV  DX,TimerCtl	;ClusterCard 8254 Control Register (offset F)
	MOV  AL,70h
	OUT  DX,AL	
	JMP  $+2
	SUB  DX,2			;ClusterCard 8254 Channel 1 Count Reg (offset D)
	MOV  AX,01FFh
	OUT  DX,AL
	JMP  $+2
	XCHG AH,AL
	OUT  DX,AL
	JMP  $+2
	JMP  $+2

Latch_TEOM:
	MOV DX,TimerCtl		;ClusterCard 8254 Control Register (offset F)
	MOV AL,0E4h
	OUT DX,AL
	JMP $+2
	SUB DX,2			;ClusterCard 8254 Chnl 1 Count Reg (offset D)
	IN  AL,DX
	JMP $+2
	AND AL,80h
	JZ	Latch_TEOM
	JMP  TurnOffTx

RepTimingTEOM:
	PUSH SI
;Set SI so REP OUTSB won't fault - JF - 04/01/93
	MOV  SI,SafeOutSbAddress	
	MOV  CH,0
	MOV  CL,SioClock
	SHL  CX,2
	MOV  DX,stat
	REP  OUTSB					; Machine independent wait function.
	POP  SI						;... R&B
; Now that the 2652 has finished, we can turn off the state machine. 
; We must leave the transmitter on so that the line will be busy until we are
; ready to read.  The workstation will not attempt to answer until we turn the
; transmitter off and turn the receiver on.
TurnOffTX:
	MOV  AX,rts
	MOV  DX,cPort
	OUT  DX,AL					; Byte I/O 
	JMP  $+2

BitStartRead:
PUBLIC BitStartRead
;*****************************************************************************
; Entry from PrepareIdSearch and SearchId
; SI = lineState
;*****************************************************************************
; Start the timer.  If 2 seconds go by, we did not get a response.
	MOV  WORD PTR trbInterval,trbTimeoutValue
	MOV  WORD PTR trbCEvents,0

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,logBufferIndex
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	PUSHF
	CLI	
	CALL DisableDma				;disable Dma in case it's enabled
; We will read into the ackInBuf or the XBlock. The state table tells
; us if we are reading into the XBlock.
	TEST BYTE PTR lstFlags[SI],fReadIntoXBlock
	JNZ  ReadToXBlock
	MOV  AX,ackInDmaAddr
	MOV  CX,ackInBufSize		;save max count to calculate count 
	MOV  cbReadMax,CX			;read when the read completes.
	MOV  BX,ackInDmaEar
	JMP  SHORT SetUpDmaForRead

ReadToXBlock:
; We are reading into the XBlock.  saXBlockCurrent points to the XBlock to
; read into.  The XBlock contains the Dma word address and count.
	MOV  ES,saXBlockCurrent		;ES points to XBlock
	MOV  AX,ES:[xbDmaAddr]
	MOV  CX,sXBlk
	ADD  CX,2					;for (station,frame)
	MOV  cbReadMax,CX					
	MOV  BX,ES:[xbDmaEar]
SetUpDmaForRead:
	SHR  CX,1					;convert count to words
	DEC  CX						;the 8237 terminates on 0FFFFH
	MOV  DL,modeDmaToMemory
	CALL SetUpCommDma				
; The Dma has been set up to read.
	MOV	 DX,rdsrH
	IN   AL,DX 				;Reset Read Data Status Register, Byte I/O

; Set up the parameter control register. The 2652 will not work unless we do
; this each time. It will be bop+idle if we are searching for id. Otherwise it
; will be bop+sam+idle (for secondary address mode).
	MOV	 AL,parameterControl
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
; For testing, we can send a bad CRC every n frames.
	CMP  nSendBadCrc,0
	JE   SendGoodCrc
	DEC  nSendBadCrc
	JNZ  SendGoodCrc
	MOV  DX,nSendBadCrcReload
	MOV  nSendBadCrc,DX
	OR   AL,1					;CRC Preset to zero -- normally preset to ones
	MOV  logBadCrc[DI],0BADCh
SendGoodCrc:
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	MOV	 DX,pcsarH
	OUT	 DX,AL
	CALL EnableDma				;Turn on the Dma
	POPF
	MOV  AX,ccrRcvEnb			;rcvEnb enables DMA, sets burst size
	MOV  DX,cPort				;Turn on the receiver
	TEST vf_fPCAT,01
	JNZ	 PCATrcvEnb
	OUT  DX,AX
	JMP  SetPollStateReading
PCATrcvEnb:
	OUT  DX,AL					; Byte I/O
	JMP  $+2
SetPollStateReading:
	MOV  pollState,OFFSET StateReading
	RET
CheckWriteAndStartRead ENDP
SigIsr PROC FAR
;*****************************************************************************
; This is the interrupt service routine for the 2652
; Each poll can have from one to three interrupts:
; (1) We always get an interrupt when the read completes.
; (2) When we are ready to respond to the poll, we may have to wait for 
;     data carrier detect to drop. If so, we wait for the dcd interrupt,
;     then start the write.
; (3) If the write is for more than two byte, we will get an interrupt when
;     the write completes. If the write is only two bytes, we will wait for it
;     to complete, then start the next read.
;*****************************************************************************

	MOV  SI,lineState			;SI points to state table
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,logBufferIndex
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	JMP  pollState
StateWriting:
PUBLIC StateWriting
; Come here if a write just completed. Go start the read.
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  logIntWrite[DI]
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	CALL CheckWriteAndStartRead
	JMP  IsrDone

StateWaitDcdDrop:
PUBLIC StateWaitDcdDrop
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  logIntDcd[DI]
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	CALL FAR PTR StartWrite
	JMP  IsrDone
StateIdle:
PUBLIC StateIdle
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  logIntIdle[DI]
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	JMP  IsrDone

IsrError:
; It's a mystery to me why we get this interrupt. It only seems to occur
; during Id search when we are reading every frame that goes by.
	MOV  CX,ercReadNotComplete
	JMP  HandleReadComplete

AbortReceived:
	MOV  CX,ercLengthError			; ABorts are really Server TX underrun.
	INC	 nAbortError				; Track as abort sent.
	JMP  HandleReadComplete

Overrun:
; If we are in stateSearchId, we are reading every frame that goes by, and
; we can expect to get overruns, so don't count these.
	CMP  lineState,stateSearchId
	JE   SetErcOverrun
	INC  nOverrunError
SetErcOverrun:
	MOV  CX,ercOverrun
	JMP  SHORT HandleReadComplete
CrcError:
	INC  nCrcError
	MOV  CX,ercCrc
	JMP  SHORT HandleReadComplete
StateReading:
PUBLIC StateReading
;*****************************************************************************
; Come here when a read just completed.
; This is the heart of the interrupt routine. Based on the frame received
; and the line state, decide what to do next.
;*****************************************************************************
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  logIntRead[DI]
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	MOV  DX,stat
	IN   AL,DX
	TEST AL,rEom
	JZ   IsrError
	XCHG AH,AL					; Keep a copy of status to check below.

	MOV  DX,rdsrH
	IN   AL,DX					;AL = Receiver Data Status Register High
	TEST AL,rOvrn				;R&B - Test overrun only (not rAbort)
	JNZ  Overrun

	TEST AL,rAbort				; Detect abort condition and report as length
	JZ   NoAbort				; Error.
; There may be an abort condition.  The IOGA/Cirrus spec says that a received
; abort does not set rxda in the Status Register.
	TEST AH,01h					; test rxda bit in status.
	JZ  AbortReceived			; Jump if rxda bit in status is 0.
	
NoAbort:
	TEST AL,rErr
	JNZ  CrcError
	MOV  CX,0					; Assume no error.

HandleReadComplete:					;R&B ...
	TEST vf_fPCAT,01
	JNZ	 DrainFIFOPCAT
	MOV  AX,dmarcv OR burst2 OR hbe	;Disable RXE, but leave RXDMA on
	MOV  DX,cPort					;on the Cirrus CD610 (B39 & B28EV)
	OUT  DX,AX						;Turn off the state machine
; Wait for DMA to complete.  For IOGA we just delay, for Cirrus we 
; wait for the FIFO to indicate empty.
	PUSH CX							;Save Error Code.
	MOV  AH,0
	MOV  AL,SioClock				;Max number of loops is based on SIO clock
	MOV  CL,shCPUSpeed				;   and CPU speed.
	SHL  AX,CL
	MOV  CX,AX
	MOV  DX,stat					;Cirrus status register
	TEST vf_f4Mbit, 0FFh
	JNZ  SHORT CheckFIFOReadError
	LOOP $							;IOGA: idle loop
	JMP  SHORT EndReadError
DrainFIFOPCAT:
; Here we program the onboard 8254 for an interval equal to the worst case
; of transfering 4 words (single mode) from FIFO to memory.  This needs to
; be timed against the Logic Analyzer to be efficient.
	PUSH CX
	MOV  DX,timerCtl
	MOV  AL,70h
	OUT  DX,AL	
	JMP  $+2
	SUB  DX, 2
	MOV  AX,090h
	OUT  DX,AL
	JMP  $+2
	XCHG AH,AL
	OUT  DX,AL
	JMP  $+2
	JMP  $+2

Latch_DrainFIFO:
	MOV DX,timerCtl
	MOV AL,0E4h
	OUT DX,AL
	JMP $+2
	SUB DX, 2
	IN  AL,DX
	JMP $+2
	AND AL,80h
	JZ	Latch_DrainFIFO
	JMP EndReadError

CheckFIFOReadError:					;Cirrus: Loop a maximum number of times 
	IN   AX,DX						;  until the FIFO is empty.
	TEST AX,FIFOEmptyBit			;Check FIFO empty bit
	JNZ  SHORT EndReadError
	LOOP CheckFIFOReadError
EndReadError:						;... R&B
; Calculate the number of bytes that were read and save the count in DI.
	CALL FetchDmaWrdCnt
	POP  CX							; POP error code from Chip status read.


	XOR  AX,AX
	MOV  DX,cPort
	TEST vf_fPCAT,01
	JNZ  ShutDownPCAT
	OUT  DX,AX						;Turn off the state machine
	JMP  DisableRxDMA
ShutDownPCAT:
	OUT  DX,AL						; Byte I/O
	JMP  $+2

DisableRxDMA:
	CALL DisableDma					;Disable DMA after turning off state mach
	MOV  pollState,OFFSET StateIdle
	TEST BYTE PTR lstFlags[SI],fReadIntoXBlock
	JNZ  WsReplyInXBlock
	MOV  ES,saAckInBuf
	MOV  AX,ES:[0]
	JMP  SHORT TestErc
WsReplyInXBlock:
	MOV  ES,saXBlockCurrent
	MOV  ES:[xbCbData],DI
	MOV  AX,ES:[xbStation]
TestErc:
	OR   CX,CX					;Test for error completion
	JNZ  JumpToState			;Don't check station if error already
; The read completed without error. If we have a station address, see if the
; station address is correct.
	CMP  lineState,stateAckSnrmWithXid
	JLE  JumpToState
	CMP  AL,stationAddress
	JNE  StationAddressError
JumpToState:
PUBLIC JumpToState
;*****************************************************************************
; When we jump to handle the read depending on the state:
; AL = station address (already checked)
; AH = frameType
; CX = error code 
; ES points to XBlock if reply read into XBlock, else ES points to ackInBuf
; SI points to lineStateTable entry
; conditionCode is set if there was an error
; DI and xbCbData (if XBlock used) contain number of bytes read 
; pollState = StateIdle
;*****************************************************************************

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	CMP  SI,lastLineState
	JBE  StateOk
	MOV  AX,ercInternalConsistencyCheck
	PUSH AX
	CALL Crash
StateOk:
	PUSH DI
	MOV  DI,logBufferIndex
	MOV  logFrameIn[DI],AX
	MOV  logErc[DI],CX
	POP  DI
	OR   CX,CX					;Set condition code again
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	MOV  stationAddressIn,AL
	MOV  bitIoErc,CX			;Save completion code
	OR   CX,CX					;Set condition code
	JMP  lstState[SI]
StationAddressError:
; If we got an address error, we don't want to try to reply because the frame
; is not addressed to us. Restart the read.
	INC  nAddressError
	OR   CX,ercAddressError
	JMP  SHORT JumpToState
IsrDone:
	RET
SigIsr ENDP
DelayBeforeAckSNRM PROC NEAR
;*****************************************************************************
; Delay for a random amount of time before we try to respond.  Other stations 
; may also be trying to respond.
; On entry, AL = station id.
; On exit, condition code = zero if data carrier detect is off, meaning no
; other station is currently responding.
;*****************************************************************************
	MOV  stationAddress,AL
	MOV  DX,pcSarL
	OUT  AL,DX					;Send station address to the 2652
	MOV  DX,stat

WaitForDcdToDrop:
; Wait for dcd to be zero (this means the line is not busy).
	IN   AX,DX
	TEST AL,dcd
	JNZ  WaitForDcdToDrop
; Generate a random number for a delay count.
	CALL GenerateRandomNumber
	AND  CL,03fh
	MOV  AL,6
	MUL  CL
	MOV  CX,AX					;CX has delay count
	MOV  DX,stat
; Now see if anyone else starts to use the line during this delay count.
DelayLoop:
	IN   AX,DX
	TEST AL,dcd
	LOOPE DelayLoop				;Loop if CX <> 0 and dcd = 0
	RET
DelayBeforeAckSNRM ENDP

GenerateRandomNumber PROC NEAR
;*****************************************************************************
; Use the baud rate generator to generate a random number.
; returns number in CL
;*****************************************************************************

	CMP  processorType, b38Lcw
	JNE  RtcClockNgen

; Use kbdBaudRateGenerator on B38-LCW only.
	MOV  AL,timerLatchCnt
	MOV  DX,timerCtl
	OUT  DX,AL
	JMP  $+2
;	MOV  DX,timerComm
	MOV  DX,140h ;R&B - kbdBaudRateGenerator, need to make a label for this
	IN   AL,DX
	JMP  $+2
	MOV  CL,AL
	IN   AL,DX
	RET

RtcClockNGEN:
	CMP		processortype,PCAT	; On PCAT use a channel on the clustercard.
	JE		LatchCC8254

; On Ngens latch 8254 used for Real Time Clock.
	MOV    	AL,Latch8254   ; Works for all hardware types excpt. b38LCW/Sgen
	CMP		processortype,SG5000
	JNE		RTCTemp
	MOV		AL,Latch8254SGen	; Sgen latch counter 2 (RTC).
RTCTemp:
	MOV		DX,modeWord8254RTC  ;latch real time clock
	OUT		DX,AL
	JMP		$+2
	MOV		DX,RTCCounter8254   ;real time clock
	IN		AL,DX			    ;low order byte
	JMP		$+2
	MOV		CL,AL
	IN		AL,DX			    ;high order byte
	XOR		CL,AL
	RET

LatchCC8254:
;
; On the  B25-PC4 AT clustercard, latch counter 1 for random number.
; This channel is initialized to mode 0 and the Counting Element wraps
; and keeps counting after terminal count.
	MOV	 DX,TimerCtl
	MOV  AL,040h;			  ; Latch counter 1
	OUT  DX,AL
	JMP  $+2
	MOV	 DX,TimerComm		  ; address of counter 0
	INC  DX					  ; address of counter 1
	IN   DX,AL				  ; read lsb
	JMP  $+2
	MOV  CL,AL
	IN   DX,AL				  ; read msb
	JMP  $+2
	XOR  CL,AL				  ; Return random number in CL
	RET
	
GenerateRandomNumber ENDP

StopRead PROC NEAR
;*****************************************************************************
; Called by Timeout
;*****************************************************************************
; R&B ...
	CMP  pollState,OFFSET stateWriting
	JNE  NotWriting
	TEST vf_fPCAT,01
	JNZ  StopWritePCAT
	MOV  AX,ccrStopDMA				;With a Cirrus chip, this enables the
	OR   AL,rts						;  terminal counter, we load the counter
	MOV  DX,cport					;  with 1 to cause DMA to stop early.
	OUT  DX,AX						;cPort + 2 is address of terminal count
	ADD  DX,2						;  register on NGen with Cirrus chip
	MOV  AX,1						;Load 1 in counter
	OUT  DX,AX
	MOV  AX,rts OR endxmit OR txwcen;This turns off TXE. On the IOGA, DMA
	OUT  DX,AX						;  will finish after the stacker empties.
	JMP  Short DoDelay
StopWritePCAT:
	MOV  AL,rts
	MOV  DX,cport
	OUT  DX,AL
	JMP  DoDelayPCAT
NotWriting:
	CMP  pollState,OFFSET stateReading
	JE	 StopARead
	JMP  NotReading
StopARead:
	TEST vf_fPCAT,01
	JNZ  StopReadPCAT
	MOV  AX,dmarcv OR burst2 OR hbe	;Disable RXE, but leave RXDMA/burst on
	MOV  DX,cPort
	OUT  DX,AX
	JMP  Short DoDelay
StopReadPCAT:
; Not much to do here, just wait (below) and then turn off cirrus chip
	MOV  AX,dmarcv
	MOV  DX,cport
	OUT  DX,AL
	JMP  DoDelayPCAT
DoDelay:
; Wait for DMA to complete.  For IOGA we just delay, for Cirrus we 
; wait for the FIFO to indicate empty.
	MOV  AH,0
	MOV  AL,SioClock				; Max number of loops is based on SIO clock
	MOV  CL,shCPUSpeed				;   and CPU speed.
	SHL  AX,CL
	MOV  CX,AX
	MOV  DX,stat					; Cirrus status register
	TEST vf_f4Mbit, 0FFh
	JNZ  SHORT CheckFIFOStopRead
	LOOP $							; IOGA: idle loop
	JMP  SHORT EndStopRead
CheckFIFOStopRead:					; Cirrus: Loop a max number of time until 
	IN   AX,DX						;   the FIFO is empty.
	TEST AX,FIFOEmptyBit			; Check FIFO empty bit
	JNZ  SHORT EndStopRead
	LOOP CheckFIFOStopRead
	JMP  EndStopRead

DoDelayPCAT:
; In order to provide a machine independent delay, program the on board 8254
; to delay while the FIFO empties before when turn off cirrus chip.
	MOV  DX,timerCtl		;ClusterCard 8254 Control Port (offset F)
	MOV  AL,70h
	OUT  DX,AL	
	JMP  $+2
	SUB  DX,2				;ClusterCard 8254 Channel 1 count port (offset D)
	MOV  AX,090h			; Not exact, needs to be calibrated.
	OUT  DX,AL
	JMP  $+2
	XCHG AH,AL
	OUT  DX,AL
	JMP  $+2
	JMP  $+2

Latch_DoDelayPCAT:
	MOV DX,timerCtl
	MOV AL,0E4h
	OUT DX,AL
	JMP $+2
	SUB  DX,2
	IN  AL,DX
	JMP $+2
	AND AL,80h
	JZ	Latch_DoDelayPCAT
	XOR AX,AX
	MOV DX,cport
	OUT DX,AL						;Byte IO
	CALL DisableDma
	JMP NotReading

EndStopRead:
	MOV  DX,cPort
	XOR  AX,AX						; Disable state machine before dma
	OUT  DX,AX						; Turn off the state machine
	CALL DisableDma					; Disable DMA
; ... R&B
NotReading:
	MOV  pollState,OFFSET StateIdle	;We need an XBlock.  Check to see if we 
	CMP  saXBlockCurrent,0			;have an XBlock to read into.
	JNE  StopReadRet
	CALL GetXBlock					;See if an XBlock is available. If so, 
	JNZ  StopReadRet				;unchain it from the free list.
	JMP  NoXBlock
StopReadRet:
	RET
StopRead ENDP

SetLineSpeed PROC NEAR
PUBLIC SetLineSpeed
;*****************************************************************************
; BL contains the value of sioClock: 24 for 307Kbps, 4 1.8M or 2 for 3.68M
;*****************************************************************************
	MOV  DX,timerCtl
	MOV  AL,mode3				; Select counter 0 for SGEN xmt clock.
	CMP  ProcessorType,SG5000
	JE	 ModeSetSGEN
	CMP  ProcessorType,PCAT		; PCAT uses counter 0 for xmt clock
	JE	 ModeSetSGEN
	MOV  AL,mode3+modeCtr2		; All other Hardware select counter 2 for xmt.

ModeSetSGEN:
	OUT  DX,AL
	JMP  $+2
	MOV  DX,timerComm
	MOV  AL,BL					;value of sioClock
	OUT  DX,AL
	JMP  $+2
	MOV  AL,0
	OUT  DX,AL
	RET
SetLineSpeed ENDP

DisableDma PROC NEAR
PUBLIC DisableDma
;*****************************************************************************
; Disables RS422 DMA for either CWS or NGens
;*****************************************************************************
	MOV  DX, dmaMask
	MOV  AL, commDmaDisable
	OUT  DX, AL
	RET
DisableDma ENDP

SetUpCommDma PROC NEAR
PUBLIC SetUpCommDma
;*****************************************************************************
; Initializes 8237 or 186 DMA for either a read or write DMA operation
; The following registers must be set up prior to calling this routine
; CX = count, AX = low order 16 bits of buffer address
; DX = 8237 mode word, BX = high order 4/8/16 bits of buffer address
;*****************************************************************************

	PUSH DX                     ;save mode word
	PUSH BX						;save ear
	MOV  BX, AX					;save dma addr for compare

; Debugging supergen
;	CMP	 processorType,SG5000
;	JNE	 NoClearFF
; 02/17/92 by GWH - Always clear byte flip-flop on all HDW.
	MOV  DX, DMABytePtrClr;
	XOR  AX,AX
	OUT  DX,AL
	JMP  $+2
	MOV  AX,BX
NoClearFF:
; end Debugging Supergen.

	MOV  DX, commDmaAddr
	OUT  DX, AL
	JMP  $+2                    ;flush pipe
	MOV  AL, AH
	OUT  DX, AL
	JMP  $+2                    ;flush pipe

; read back addr, compare
	IN   AL, DX					;low byte
	JMP  $+2                    ;flush pipe
	MOV  AH, AL
	IN   AL, DX					;high byte
	JMP  $+2                    ;flush pipe
	XCHG AL, AH
	CMP  AX, BX
	JE   DmaAddrOk
	PUSH ercDmaAddr
	CALL Crash
DmaAddrOk:
	POP  BX						; POP both Low And High Page Bytes.

	MOV  AX, CX
	MOV  DX,commDmaWrdCnt
	OUT  DX, AL
	JMP  $+2                    ;flush pipe
	MOV  AL, AH
	OUT  DX, AL
	JMP  $+2                    ;flush pipe
	POP  AX                     ;AX = mode word
	MOV  DX, dmaMode
	OUT  DX, AL 
	JMP  $+2                    ;flush pipe
	MOV  AX, BX					; Upper 16 bits of buffer address
	MOV  DX, commDmaEar
	OUT  DX, AL					; Was word I/O, now byte for SGEN
	JMP  $+2                    ;flush pipe
	CMP  processorType, NewGen
	JE   EnableNewGenEar
	TEST vf_fEISABus,1
	JNZ	 EnableSGenEisaEar      ; Enable >16Mb DMA ear on EISA and SGen  FW 
	TEST vf_fSGEN,1
	JZ	 xxx8237Ret
EnableSGenEisaEar:
	MOV	 DX, commDmaEarHigh
	XCHG AL,AH					; Get High order byte of 32 bit address in AL
	OUT  DX,AL
	JMP  xxx8237Ret


EnableNewGenEar:
	MOV  DX, DmaEarEnable
	IN   AL, DX
	JMP  $+2                    ;flush pipe
	OR   AL, bEnableEarChan0    ;
	OUT  AL, DX                 ;enable ear
	TEST vf_f386, 1     	    ;is this a 386i?
	JZ   xxx8237Ret             ;no.
	MOV  AX, BX                 ;yes, get upper >16Mb ear
	SHL  AX, 1
	MOV  AL, AH
	MOV  DX, commDmaEarHigh     ;
	OUT  DX, AL                 ;upper 16Mb bank of 386i ram
xxx8237Ret:
	RET
SetUpCommDma ENDP

EnableDma PROC NEAR
PUBLIC EnableDma
;*****************************************************************************
; Enables RS422 DMA for either CWS or NGens
;*****************************************************************************
	MOV  AL,commDmaEnable
	MOV  DX,dmaMask
	OUT  DX,AL
	RET
EnableDma ENDP

FetchDmaWrdCnt PROC NEAR
PUBLIC FetchDmaWrdCnt
;*****************************************************************************
; Returns word count from DMA read operation for either a CWS or NGens
;*****************************************************************************
	MOV  DI,cbReadMax

;We need to disable all interrupts during this section of DMA programming.
;Else, the DMA first/last Flip Flop may be in an inconsistent state.
	PUSHF
	CLI

	MOV	 DX,DMABytePtrClr
	XOR  AX,AX
	OUT  DX,AL					; Clear byte pointer flip flop.
	JMP  $+2

	MOV  DX,commDmaWrdCnt
	IN   AL,DX
	MOV  CL,AL
	JMP  .+2					;R&B
	IN   AL,DX
	POPF

	MOV  CH,AL
	CMP  CH,0FFH
	JE   ReadMaxNumBytes
	SHL  CX,1
	SUB  DI,CX
	DEC  DI
	DEC  DI
ReadMaxNumBytes:
	RET
FetchDmaWrdCnt ENDP

WsLph ENDS

; LOG
; 3/28/84 by Jim Frandeen: Created file
; 8/21/84 by Mike Ober: Added support for CWS
; 4/2/85 by Jim Frandeen: Add dummy labels IsrNull, IsrRxSpec, IsrExStat
; 4/2/85 by Mike Ober: Added support for GWS
; 2/7/86 by JM: longer wait for flags from 2652 for T2.
; 4/16/86 by FW: Added support for NewGen.
; 4/28/86 by DR: CTOSp
; 9/23/86 by DR: CTOS II 2.0 merge
;  10/15/86 by GVW : Fix for CP003
; 3/12/87 by JA/RLM: read dma addr back, compare.  Trying to catch hdwr bug.
; 02/17/88 by RLM change nop's for flag timming to shcpu loop$
; 03/22/88 by FW: Allow operation in >16Mb on 386i machines.
; 04/06/88 by RLM: set TSOM before TxE on transmitt. caused erc 6 no flags sent
; 08/29/88 by RLM: set up lineSpeedTable for 3.68 linespeed
; 09/26/88 by JA: use vf_f386
; 10/16/89 by GWH: Removed Xblock Def to WsEqu.idf and Realigned for 3.0
; 12/14/89 by GWH: Changed Timing Loops after sioclock - helps 4.0mbps.
; 01/09/90 by AT: Add size of request, not size of header on sXBlk from master.
; 08/15/90 by JC, Added BTOS Collision detection mods, RS232 latency mods
; 08/20/90 by GWH: SetUpCommDma uses commDmaWrdCnt to load DX for count 
;                  register
; 08/23/90 by JC: Deleted collision detection (won't work with new
;                 variable polling rates)
; 08/24/90 by JA: fElseWsNotReady masks new state.
; 09/05/90 by SG: Added b38lcw support 
; 09/06/90 by JA: don't ever call MediateIntHandler.
; 10/17/90 by SG: Add dma mode variables, remove b38lcw runtime checks
; 12/07/90 by JA: add 2 to dma count on read for (station,frame)
; 01/09/91 by AT: Use 2-word burst demand mode DMA on SG5000.
; 02/18/91 by GWH: Rework SetUpCommDma for SG5000. Remove 186 code.
; 03/12/91 by BA: Merged rotate and burst changes from s2.4.3 (see R&B 
;                 comments)
; 04/01/91 by JF: In StopRead, make sure we have an XBlock when we change
;                 pollState to StateIdle.
; 04/01/91 by BA: In the timing loops, replaced CPU timing dependend LOOP $ 
;                 instructions with with timing independent REP OUTSB.
; 04/12/91 by GWH: Fixed GenerateRandomNumber for SG-5000.
; 04/15/91 by BA: Fixed GenerateRandomNumber for SG-2000, since this processor 
;                 doesn't have a real time clock, use the kbd baud rate 
;                 generator which generates a nice random number.
; 05/07/91 by BA: As a consequence of rotating priority, a busy parallel port 
;				  can cause underruns on the cluster.  To get around this, 
;				  we mask the parallel DMA before transmitting and unmask
;				  it after (but only if the fMaskParallelDma flag was set at
;				  init time (in InitClstr)).
; 05/09/91 by BA: Corrected TSOM logic.
; 05/14/91 by BA: Don't count aborts when counting overruns in StateReading.
; 05/16/91 by BA: Reduce TSOM delay (temporary latency fix) and reverse sense.
; 05/20/91 by BA: Forget the above TSOM fixes, this is the real one.
; 05/21/91 by BA: Send more flags for telecluster.
; 10/01/91 by GWH: Port GenerateRandomNumber and SetLineSpeed to PCAT and 
;				   ClusterCard.
; 01/28/92 by JM: merge from 3.3.3 (
; 	12/10/91 by GWH:In StateReading, call FetchDMAWrdCount after waiting for
;				  fifo to empty.
; 	12/11/91 by GWH:Test for TX ABort conditions again.  Count them as length
;				  Errors.
; 	12/14/91 by GWH:Enhance abort checking. Add naborterror,nBadRcbError.
;	)
; 02/17/92 by GWH: StartWriteClusterCard.  Changed Word I/O to Byte I/O where
;				   applicable ( commented with 'byte I/O' where changed ).
;				   Also clear byte pointer flip-flop in setupcommdma for
;				   all hardware.
; 03/19/92 by GWH: Use offsets to calculate the I/O port address for the
;				   8254 count register (on board the clustercard) used in
;				   timing loops.  Clear DMA Byte pointer flip flop for all
;				   hardware in fetchDMAWrdCnt
; 05/15/92 by JA:  More of the same.
; 04/06/93 by JF: Set SI addressable non-zero so OUTSB won't fault when
;			soft bus version uses DS allocation.  
; 05/12/93 by FW   Support for ClientCard in EISA boxes with > 16Mb RAM.
	END
